package csbase.logic.algorithms.parsers;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.URLProtocol;

/**
 * Analisador de parmetros URL de entrada.
 *
 * @author Tecgraf
 */
public final class OutputURLParameterParser extends
  AbstractURLParameterParser<OutputURLParameter> {

  /**
   * <p>
   * O atributo
   * {@value #OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE} do
   * elemento {@link OutputURLParameter}.
   * </p>
   *
   * <p>
   * Indica se o {@link OutputFileParameter arquivo de sada} deve permitir
   * sadas mltiplas,  opcional, o seu valor-padro 
   * {@link #OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE} e 
   * do tipo booleano.
   * </p>
   */
  protected static final String OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE =
    "permitir_bifurcacao";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE} do
   * elemento {@link OutputURLParameter}
   * </p>
   * <p>
   * O seu valor 
   * {@value #OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_DEFAULT_VALUE =
    false;

  /**
   * {@inheritDoc}
   */
  @Override
  protected OutputURLParameter createURLParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, String[] types, String localization,
    FileParameterMode mode, FileURLValue defaultValue,
    FileParameterPipeAcceptance usesPipe, EnumSet<URLProtocol> allowedProtocols)
    throws ParseException {

    if (types != null && types.length > 1) {
      throw new ParseException(
        "URL de sada no admite mltiplos tipos de arquivo");
    }

    boolean allowMultipleOutput = parser.extractAttributeValueAsBoolean(
      OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE,
      OUTPUT_URL_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_DEFAULT_VALUE);
    return new OutputURLParameter(name, label, description, defaultValue,
      isOptional, isVisible, commandLinePattern, localization, types, mode,
      usesPipe, allowedProtocols, allowMultipleOutput);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Class<? extends Parameter>> getParameterTypes() {
    return Collections.singletonList(OutputURLParameter.class);
  }
}
