package csbase.logic.algorithms.validation;

import java.util.ArrayList;
import java.util.List;

/**
 * Interface de um resultado de validao.
 */
public class Validation {
  /** Resultado principal da validao. Se passou ou no */
  private boolean isWellSucceeded = true;

  /**
   * Mensagens do resultado. Se for um agrupamento esta lista ser anulada
   * internamente, e o getMessages buscar no filhos para respornder.
   */
  private List<LocalizedMessage> messages;

  /**
   * Lista de subresultados que compes este resultado. Se no for um grupo esta
   * lista  anulada.
   */
  private List<Validation> children;

  //----------------------

  /**
   * Contrutor padro de resultado que  um grupo. Permite adicionar
   * sub-resultados depois. Se consultar o status antes de adicionar
   * sub-resultados, este ser TRUE;
   */
  public Validation() {
    isWellSucceeded = true;
    messages = null;
    children = new ArrayList<>();
  }

  /**
   * Construtor cpia, incorpora outro resultado. No adiciona como sub
   * resultado e sim COPIA o outro para si. Se o outro tiver filhos, sero
   * copiados tambm;
   * 
   * @param result O resultado que ser copiado.
   */
  public Validation(Validation result) {
    this.isWellSucceeded = result.isWellSucceeded();
    this.children = result.getChildren();
    this.messages = result.getMessage();
  }

  /**
   * Consturtor de resultado final, sem filhos. No permite adicionar
   * subresultados depois.
   * 
   * @param isWellSuceeded Status do resultado
   * @param message Mensagem associada ao resultado (pode ser nula)
   */
  public Validation(boolean isWellSuceeded, LocalizedMessage message) {
    this.isWellSucceeded = isWellSuceeded;
    this.children = null;
    messages = new ArrayList<>();
    if (message != null) {
      this.messages.add(message);
    }
  }

  //--------------------------

  /**
   * Obtm o status do resultado
   * 
   * @return Se passou no teste ou no
   */
  public boolean isWellSucceeded() {
    return isWellSucceeded;
  }

  /**
   * Obtm todas as mensagens associadas ao resultado.  uma lista concatenada
   * das mensagens de todos os subresultados
   * 
   * @return Lista de mensagens
   */
  public List<LocalizedMessage> getMessage() {

    if (children != null) {
      List<LocalizedMessage> returnMessages = new ArrayList<>();
      for (Validation child : children) {
        returnMessages.addAll(child.getMessage());
      }
      return returnMessages;
    }

    return messages;
  }

  /**
   * Obtm a lista de sub-resultados
   * 
   * @return Lista de subresultado, pode ser NULO para indicar que este  um
   *         resultado folha.
   */
  public List<Validation> getChildren() {
    return children;
  }

  /**
   * Adiciona um sub resultado. S pode ser usado se o resultado foi construdo
   * como grupo. Caso contrrio levanta IllegalArgumentException
   * 
   * @param validation Validao a ser adicionada como sub-resultado
   */
  public void addChild(Validation validation) {

    if (children == null) {
      throw new IllegalArgumentException("Not a group result");
    }

    children.add(validation);
    isWellSucceeded &= validation.isWellSucceeded();
  }
}