package csbase.logic;

/**
 * Filtro para os Tipos de Arquivos/Diretrios.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class ProjectFileTypeFilter {
  /**
   * Indica se o tipo  aceito ou no segundo os critrios deste filtro.
   * 
   * @param type O tipo (pode ser null).
   * 
   * @return <code>true</code> se for aceito ou <code>false</code> caso 
   * contrrio.
   */
  public abstract boolean accept(ProjectFileType type);
  
  /**
   * Cria um filtro que seleciona todos os tipos de arquivos de projetos.
   * 
   * @return O filtro.
   */
  public static ProjectFileTypeFilter all() {
     return new ProjectFileTypeFilter() {
      @Override
      public boolean accept(ProjectFileType type) {
        return true;
      }
    };
  }

  /**
   * Cria um filtro que seleciona apenas os tipos aplicveis a diretrios.
   * 
   * @return O filtro.
   */
  public static ProjectFileTypeFilter directories() {
    return directory(true);
  }
  
  /**
   * Cria um filtro que seleciona apenas os tipos aplicveis a arquivos.
   * 
   * @return O filtro.
   */
  public static ProjectFileTypeFilter files() {
    return directory(false);
  }

  /**
   * Cria um filtro que seleciona nenhum tipo de arquivo de projeto.
   * 
   * @return O filtro.
   */
  public static ProjectFileTypeFilter none() {
    return new ProjectFileTypeFilter() {
      
      @Override
      public boolean accept(ProjectFileType type) {
        return false;
      }
    };
  }

  /**
   * Cria um filtro que seleciona apenas os tipos aplicveis a diretrios
   * se {@code directory} for {@code true} ou a arquivo se {@code directory} for 
   * {@code false}.
   * 
   * @param directory {@code true} para diretrios ou {@code false} para
   * arquivos.
   * 
   * @return O filtro.
   */
  public static ProjectFileTypeFilter directory(final boolean directory) {
    return new ProjectFileTypeFilter() {
      @Override
      public boolean accept(ProjectFileType type) {
        if (type == null) {
          return false;
        }
        return type.isDirectory() == directory;
      }
    };
  }

  /**
   * Cria um filtro que seleciona apenas o tipo cujo o cdigo  o cdigo
   * fornecido.
   * 
   * @param typeCode O cdigo (Se for {@code null} ser falso).
   *
   * @return O filtro.
   */
  public static ProjectFileTypeFilter typeCode(final String typeCode) {
    return new ProjectFileTypeFilter() {
      @Override
      public boolean accept(ProjectFileType type) {
        if (type == null) {
          return false;
        }
        return type.getCode().equals(typeCode);
      }
    };
  }

  /**
   * Cria um filtro que faz a operao or de todos os filtros informados.
   * 
   * @param filters Os filtros.
   *
   * @return O filtro.
   */
  public static ProjectFileTypeFilter or(final ProjectFileTypeFilter... filters) {
    if (filters.length == 0) {
      throw new IllegalArgumentException("filters.length == 0");
    }
    return new ProjectFileTypeFilter() {
      @Override
      public boolean accept(ProjectFileType type) {
        for (ProjectFileTypeFilter filter : filters) {
          if (filter.accept(type)) {
            return true;
          }
        }
        return false;
      }
    };
  }
  
  /**
   * Cria um filtro que faz a operao and de todos os filtros informados.
   * 
   * @param filters Os filtros.
   *
   * @return O filtro.
   */
  public static ProjectFileTypeFilter and(final ProjectFileTypeFilter... filters) {
    if (filters.length == 0) {
      throw new IllegalArgumentException("filters.length == 0");
    }
    return new ProjectFileTypeFilter() {
      @Override
      public boolean accept(ProjectFileType type) {
        for (ProjectFileTypeFilter filter : filters) {
          if (!filter.accept(type)) {
            return false;
          }
        }
        return true;
      }
    };
  }

  /**
   * Cria um filtro que faz a operao not no filtro informado.
   * 
   * @param filter O filtro.
   *
   * @return O filtro.
   */
  public static ProjectFileTypeFilter not(final ProjectFileTypeFilter filter) {
    if (filter == null) {
      throw new IllegalArgumentException("filter == null");
    }
    return new ProjectFileTypeFilter() {
      @Override
      public boolean accept(ProjectFileType type) {
        return !filter.accept(type);
      }
    };
  }
}
