package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;
import java.text.MessageFormat;

import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * Validador do {@link csbase.logic.algorithms.parameters.TextParameter
 * Parmetro Tipo Texto}.
 * 
 * @author lmoreira
 */
public final class TextParameterValidator extends
  SimpleParameterValidator<String> {
  /**
   * Quantidade mxima de caracteres para que o texto seja considerado vlido.
   */
  private Integer maxCharacters;

  /**
   * Cria um validador.
   * 
   * @param isOptional Indica se o valor  opcional/obrigatrio.
   * @param maxCharacters Quantidade mxima de caracteres para que o texto seja
   *        considerado vlido (Se no houver limite {@code null} - no pode ser
   *        menor ou igual a 0).
   */
  public TextParameterValidator(boolean isOptional, Integer maxCharacters) {
    super(isOptional);
    setMaxCharacters(maxCharacters);
  }

  /**
   * Obtm a quantidade mxima de caracteres para que o texto seja considerado
   * vlido.
   * 
   * @return Quantidade mxima de caracteres para que o texto seja considerado
   *         vlido ou {@code null} se no houver limite.
   */
  public Integer getMaxCharacters() {
    return this.maxCharacters;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validateValue(SimpleParameter<?> parameter, String value,
    ValidationContext context) throws RemoteException {
    Validation validationResult =
      super.validateValue(parameter, value, context);
    if (!validationResult.isWellSucceeded()) {
      return validationResult;
    }
    if (parameter.isEnabled() && parameter.isVisible()) {
      if (value != null) {
        if (getMaxCharacters() != null
          && value.length() > getMaxCharacters().intValue()) {
          LocalizedMessage message =
            new LocalizedMessage(TextParameterValidator.class,
              "above_max_characteres", new Object[] { parameter.getLabel(),
                  value, value.length(), getMaxCharacters() });
          return new ValidationError(message);
        }
        if (!value.matches("[:/#\\w\\.-]+")) {
          LocalizedMessage message =
            new LocalizedMessage(TextParameterValidator.class,
              "illegal_characteres",
              new Object[] { parameter.getLabel(), value });
          return new ValidationError(message);
        }
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Atribui a quantidade mxima de caracteres para que o texto seja considerado
   * vlido.
   * 
   * @param maxCharacters Quantidade mxima de caracteres para que o texto seja
   *        considerado vlido (Se no houver limite {@code null} - no pode ser
   *        menor ou igual a 0).
   */
  private void setMaxCharacters(Integer maxCharacters) {
    if (maxCharacters != null && maxCharacters.intValue() <= 0) {
      throw new IllegalArgumentException(MessageFormat.format(
        "O parmetro maxCharacters deveria ser positivo.\n"
          + "Valor encontrado: {0}.", new Object[] { maxCharacters }));
    }
    this.maxCharacters = maxCharacters;
  }
}
