package csbase.logic.filters;

import java.util.regex.Pattern;

import csbase.logic.ClientProjectFile;
import csbase.logic.ProjectFileFilter;

/**
 * <p>
 * Define um filtro para arquivos de projeto. Este filtro 
 * <i>Case-Insensitive</i>. Essa classe verifica se o arquivo como argumento no
 * mtodo accept casa com o filtro do construtor. Se o arquivo passado  um
 * diretrio, o mtodo accept verifica apenas se o nome do diretrio casa com o
 * filtro e no verifica se arquivos filhos desse diretrio tambm casam.
 * </p>
 * <p>
 * Os filtros, que so argumentos para o construtor, no so expresses
 * regulares mas uma simplificao convencionada para permitir flexibilidade sem
 * adicionar a complexidade das expresses regulares. Alm do caracteres
 * alfanumricos, os caracteres especiais reconhecidos pelo filtro so:
 * </p>
 * <ol>
 * <li>'<b>?</b>': Aceita um caracter alfanumrico qualquer;</li>
 * <li>'<b>*</b>': Aceita qualquer nmero de caracteres alfanumricos;</li>
 * <li>'<b>$</b>': Marca o fim do arquivo; por <i>default</i>, a classe
 * considera que o filtro entrado  sempre o <u>incio</u> do filtro, isto ,
 * aceita qualquer nmero de caracteres alfanumricos aps aqueles digitados
 * pelo usurio. Se o caracter '$' for entrado, o filtro no aceitar nenhum
 * caracter alm daqueles digitados pelo usurio.</li>
 * </ol>
 * 
 * 
 * @author Tecgraf
 */
public class ProjectFileNameFilter implements ProjectFileFilter {

  /** Expresso regular a ser comparada com o nome do arquivo. */
  private Pattern namePattern;

  /**
   * Cria o filtro.
   * 
   * @param name filtro de nome definido pelo usurio. Nulos no so permitidos.
   *        Se um valor vazio for passado, o filtro permitir quaisquer nomes de
   *        arquivos.
   */
  public ProjectFileNameFilter(String name) {
    if (name == null) {
      throw new IllegalArgumentException("name == null");
    }
    name = name.replaceAll("\\.", "\\\\.");
    name = name.replaceAll("\\?", "\\.");
    name = name.replaceAll("\\*", "\\.\\*");
    if ((name.length() == 0 || name.charAt(name.length() - 1) != '*') && (name
      .indexOf('$') == -1)) {
      name += ".*";
    }
    namePattern = Pattern.compile(name, Pattern.CASE_INSENSITIVE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean accept(final ClientProjectFile file) {
    final boolean result = nameMatches(file);
    return result;
  }

  /**
   * Verifica se o nome do arquivo  aceito pelo filtro.
   * 
   * @param file arquivo sendo verificado.
   * 
   * @return true se o nome do arquivo for aceito pelo filtro.
   */
  private boolean nameMatches(ClientProjectFile file) {
    return namePattern.matcher(file.getName()).matches();
  }

}