/*
 * $Id: UsersNotification.java 116866 2011-04-01 15:13:57Z cassino $
 */
package csbase.logic;

import java.util.Collection;

/**
 * Classe que modela uma notificao de compartilhamento.
 */
public class UsersNotification extends Notification {

  /** Indica que usurios foram includos no compartilhamento. */
  public static final int INSERT = 0;

  /** Indica que usurios foram removidos no compartilhamento. */
  public static final int REMOVE = 1;

  /** Indica que o tipo de compartilhamento foi alterado. */
  public static final int CHANGE = 2;

  /** Indica que o projeto foi removido */
  public static final int REMOVE_PROJECT = 3;

  /** Strings de auditagem da notificao de gerencimento de usurios */
  private final String[] strType = { "INSERT", "REMOVE", "CHANGE",
      "REMOVE_PROJECT" };

  /** Informao sobre o gerenciamento: administrador do projeto */
  private Object ownerID;

  /** Informao sobre o gerenciamento: quem est sendo inserido/removido */
  private Collection<Object> usersToNotify;

  /** * Informao sobre o gerenciamento: - qual projeto */
  private String projectName;

  /** Atributo que indica a ao de publicao ou despublicao. */
  private int type;

  /**
   * Definio da volatilidade da notificao.
   * 
   * @return flag indicativo.
   */
  @Override
  public boolean isVolatile() {
    return false;
  }

  /**
   * Consulta o tipo de evento de gerenciamento de usurios.
   * 
   * @return o cdigo do evento, que pode ser INSERT ou REMOVE.
   */
  public int getType() {
    return type;
  }

  /**
   * Consulta o usurio que inseriu ou removeu o usurio do projeto.
   * 
   * @return o identificador do usurio que originou o evento.
   */
  public Object getUserId() {
    return ownerID;
  }

  /**
   * Consulta o nome do projeto ao qual o usurio foi inserido ou removido.
   * 
   * @return o nome do projeto.
   */
  public String getProjectName() {
    return projectName;
  }

  /**
   * String para auditagem.
   * 
   * @return uma string
   */
  @Override
  public String toString() {
    return String.format("PRJ_PERMS (%s) From: %s Project: %s To: %s",
      strType[type], ownerID, projectName, usersToNotify.toString());
  }

  /**
   * Cria um evento para notificao de compartilhamento de projeto. Notifica um
   * conjunto de usurios avisandoq que estes foram inseridos ou removidos do
   * compartilhamento de um determinado projeto.
   * 
   * @param sender Quem est enviando a notificao (no pode ser {@code null}).
   * @param type - tipo do evento (insero ou remoo)
   * @param usersToNotify - usurios que sero notificados
   * @param ownerID - ID do dono do projeto que deu origem  notificao
   * @param projectName - nome do projeto que deu origem  notificao
   */
  public UsersNotification(String sender, int type,
    Collection<Object> usersToNotify,
    Object ownerID, String projectName) {
    super(sender);
    this.type = type;
    this.usersToNotify = usersToNotify;
    this.ownerID = ownerID;
    this.projectName = projectName;
  }
}
