/*
 * $Id: Category.java 108630 2010-08-03 18:23:44Z pizzol $
 */

package csbase.logic.algorithms;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import csbase.logic.AlgoEvent;
import csbase.logic.IdInterface;

/**
 * Categoria de algoritmo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class Category implements Comparable<Category>, Serializable,
  IdInterface {

  /**
   * Categoria pai, ou null, caso seja de primeiro nvel
   */
  private Category parentCategory;

  /**
   * O identificador da categoria.
   */
  private String id;

  /**
   * O nome da categoria.
   */
  private String name;

  /**
   * Os algoritmos filhos desta categoria.
   */
  private Set<AlgorithmInfo> algorithms;

  /**
   * As categorias que so filhas desta categoria.
   */
  private SortedSet<Category> categories;

  /** Objeto que permite a observao local dessa categoria */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Construtor
   * 
   * @param parentCategory categoria pai. Se for uma categoria raiz, esse
   *        parmetro deve ser null.
   * @param id O identificador (No aceita {@code null}).
   * @param name O nome (No aceita {@code null}).
   */
  public Category(final Category parentCategory, final String id,
    final String name) {
    algorithms = new HashSet<AlgorithmInfo>();
    categories = new TreeSet<Category>();
    setParentCategory(parentCategory);
    setId(id);
    setName(name);
  }

  /**
   * Construtor.
   * 
   * @param id o identificador (No aceita {@code null}).
   * @param name O nome (No aceita {@code null}).
   */
  public Category(final String id, final String name) {
    algorithms = new HashSet<AlgorithmInfo>();
    categories = new TreeSet<Category>();
    setParentCategory(null);
    setId(id);
    setName(name);
  }

  /**
   * Constri uma categoria a partir de uma categoria base.
   * 
   * @param category categoria base
   */
  public Category(Category category) {
    this(category.getParentCategory(), category.getId(), category.getName());
    algorithms = new HashSet<AlgorithmInfo>(category.getAlgorithms());
    categories = new TreeSet<Category>(category.getCategories());
  }

  /**
   * Atualiza uma categoria a partir dos dados da categoria especificada.
   * 
   * @param category categoria cujos dados sero copiados
   */
  public void updateCategory(Category category) {
    this.name = category.getName();
    this.algorithms = category.getAlgorithms();
    this.categories = category.getCategories();
  }

  /**
   * Adiciona um algoritmo a esta categoria.
   * 
   * @param algorithm O de algoritmo (No aceita {@code null}).
   * @return {@code true} em caso de sucesso ou {@code false} em caso de erro.
   */
  public boolean addAlgorithm(final AlgorithmInfo algorithm) {
    if (algorithm == null) {
      throw new IllegalArgumentException("algorithm == null");
    }

    return algorithms.add(algorithm);
  }

  /**
   * Adiciona uma categoria filha.
   * 
   * @param category A categoria filha (No aceita {@code null}).
   * @return {@code true} em caso de sucesso ou {@code false} se a categoria j
   *         est na lista de categorias filhas ou se for ela mesma.
   */
  public boolean addCategory(final Category category) {
    if (category == null) {
      throw new IllegalArgumentException("category == null");
    }
    if (equals(category)) {
      return false;
    }
    return categories.add(category);
  }

  /**
   * {@inheritDoc}
   */
  public int compareTo(final Category other) {
    return getName().compareTo(other.getName());
  }

  /**
   * Verifica se o algoritmo est nesta categoria. No faz a busca em suas
   * sub-categorias.
   * 
   * Veja tambm: @see
   * {@link Category#containsAlgorithmInChildren(AlgorithmInfo)}
   * 
   * @param algorithm O algoritmo (no aceita {@code null}).
   * @return retorna true se a categoria contm o algoritmo, caso contrrio,
   *         retorna false
   */
  public boolean containsAlgorithm(final AlgorithmInfo algorithm) {
    if (algorithm == null) {
      throw new IllegalArgumentException("algorithm == null");
    }
    if (algorithms.contains(algorithm)) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se um algoritmo com o nome especificado pertence a essa categoria.
   * No faz a busca em suas sub-categorias.
   * 
   * @param algoName nome do algoritmo procurado
   * @return retorna true se a categoria contm o algoritmo, caso contrrio,
   *         retorna false
   */
  public boolean containsAlgorithm(String algoName) {
    for (AlgorithmInfo algoInfo : algorithms) {
      if (algoInfo.getName().equals(algoName)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se o algoritmo est nesta categoria ou em outras sub-categorias.
   * 
   * @param algorithm O algoritmo (no aceita {@code null}).
   * @return indicativo
   */
  public boolean containsAlgorithmInChildren(final AlgorithmInfo algorithm) {
    if (algorithm == null) {
      throw new IllegalArgumentException("algorithm == null");
    }
    if (algorithms.contains(algorithm)) {
      return true;
    }
    for (final Category category : categories) {
      if (category.containsAlgorithmInChildren(algorithm)) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object obj) {
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    final Category other = (Category) obj;
    if (!getId().equals(other.getId())) {
      return false;
    }
    return true;
  }

  /**
   * Obtm os algoritmos desta categoria.
   * 
   * @return Os algoritmos desta categoria. O conjunto retornado  imutvel.
   *         Caso no haja algoritmos, retornar um conjunto vazio.
   */
  public Set<AlgorithmInfo> getAlgorithms() {
    return algorithms;
    // return Collections.unmodifiableSet(algorithms);
  }

  /**
   * Obtm as categorias filhas desta categoria.
   * 
   * @return As categorias filhas. O conjunto retornado  imutvel. Caso no
   *         haja categorias filhas, retornar um conjunto vazio.
   */
  public SortedSet<Category> getCategories() {
    return categories;
    // return Collections.unmodifiableSortedSet(categories);
  }

  /**
   * Obtm o identificador.
   * 
   * @return o identificador.
   */
  public String getId() {
    return id;
  }

  /**
   * Consulta o nome
   * 
   * @return o nome
   */
  public String getName() {
    return name;
  }

  /**
   * Consulta o nome completo da categoria, incluindo o nome completo da
   * categoria pai. Por exemplo, "categoriaA:categoriaA1:categoriaA11".
   * 
   * @return o nome completo da categoria
   */
  public String getFullName() {
    String fullName = getName();
    try {
      Category parentCategory = getParentCategory();

      if (parentCategory != null) {
        fullName = getParentCategory().getFullName();
        fullName += ":" + getName();
      }
    }
    catch (Exception e) {
      // fullName retorna o prprio nome da categoria
    }
    return fullName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return id.hashCode();
  }

  /**
   * Remove um algoritmo desta categoria.
   * 
   * @param algorithm O de algoritmo (No aceita {@code null}).
   * @return {@code true} em caso de sucesso ou {@code false} em caso de erro.
   */
  public boolean removeAlgorithm(final AlgorithmInfo algorithm) {
    if (algorithm == null) {
      throw new IllegalArgumentException("algorithm == null");
    }
    return algorithms.remove(algorithm);
  }

  /**
   * Remove todos os algoritmos dessa categoria.
   * 
   */
  public void removeAllAlgorithms() {
    algorithms.clear();
  }

  /**
   * Remove uma sub-categoria dessa categoria.
   * 
   * @param category categoria a ser removida
   * 
   * @return retorna true se a sub-categoria foi removida com sucesso, caso
   *         contrrio, retorna false
   */
  public boolean removeCategory(Category category) {
    if (category == null) {
      throw new IllegalArgumentException("category == null");
    }
    return categories.remove(category);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return name;
  }

  /**
   * Atribui um identificador a esta categoria.
   * 
   * @param id O identificador (No aceita {@code null}).
   */
  protected void setId(final String id) {
    if (id == null) {
      throw new IllegalArgumentException("id == null");
    }
    this.id = id;
  }

  /**
   * Atribui o nome a esta categoria.
   * 
   * @param name O nome (No aceita {@code null}).
   */
  private void setName(final String name) {
    if (name == null) {
      throw new IllegalArgumentException("name == null");
    }
    this.name = name;
  }

  /**
   * Estabelece a categoria pai desta categoria.
   * 
   * @param parentCategory categoria pai
   */
  public void setParentCategory(Category parentCategory) {
    this.parentCategory = parentCategory;
  }

  /**
   * Obtm o identificador da categoria pai desta categoria.
   * 
   * @return o identificador da categoria pai
   */
  public String getParentId() {
    if (parentCategory == null) {
      return null;
    }
    return parentCategory.getId();
  }

  /**
   * Obtm a categoria pai desta categoria.
   * 
   * @return a categoria pai
   */
  public Category getParentCategory() {
    return parentCategory;
  }

  /**
   * Esse mtodo  chamado quando o servio de algoritmos sofre alguma alterao
   * relativa a categoria de algoritmos.
   * 
   * Sempre que a classe  avisada pelo servio de algoritmos de que alguma
   * mudana ocorreu, os observadores locais tambm so notificados.
   * 
   * @param event o evento que ocorreu no servio de algoritmo
   * 
   * @throws Exception Em caso de erro.
   */
  public static void update(AlgoEvent event) throws Exception {
    observable.notifyObservers(event);
  }

  /**
   * Adiciona um observador local da classe.
   * 
   * @param o um observador local
   */
  public static void addObserver(Observer o) {
    observable.addObserver(o);
  }

  /**
   * Remove um observador local da classe.
   * 
   * @param o o observador a ser removido
   */
  public static void deleteObserver(Observer o) {
    observable.deleteObserver(o);
  }

  /**
   * Verifica se a categoria contm uma sub-categoria com um determinado nome.
   * 
   * @param categoryName nome da categoria
   * 
   * @return retorna true se existir uma categoria com o nome especificado, caso
   *         contrrio, retorna false
   */
  public boolean containsCategory(String categoryName) {
    for (final Category category : categories) {
      if (category.getName().trim().equals(categoryName.trim())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm uma determinada categoria dentro do seu conjunto de sub-categorias.
   * 
   * @param id identificador da categoria procurada
   * @return a categoria procurada, caso contrrio, retorna null
   */
  public Category getCategory(String id) {
    if (id == null) {
      return null;
    }
    if (this.id.equals(id)) {
      return this;
    }
    return getCategory(id, categories);
  }

  /**
   * Obtm uma determinada categoria na lista de categorias especificada. Essa
   * categoria pode estar no primeiro nvel ou ser uma sub-categoria.
   * 
   * @param id identificador da categoria procurada
   * @param categories conjunto de categorias onde procurar a categoria
   * @return a categoria procurada, caso contrrio, retorna null
   */
  private Category getCategory(String id, SortedSet<Category> categories) {
    for (Category category : categories) {
      if (category.getId().equals(id)) {
        return category;
      }
      Category subCategory = getCategory(id, category.getCategories());
      if (subCategory != null) {
        return subCategory;
      }
    }
    return null;
  }

  public Category getRootCategory() {
    if (getParentCategory() == null) {
      return this;
    }
    return getRootCategory(getParentCategory());
  }

  private Category getRootCategory(Category rootCategory) {
    Category parentCategory = rootCategory.getParentCategory();
    if (parentCategory == null) {
      return rootCategory;
    }
    return getRootCategory(parentCategory);
  }

  /**
   * Estabelece um novo conjunto de sub-categorias para essa categoria.
   * 
   * @param categories novo conjunto de categorias
   */
  public void setCategories(SortedSet<Category> categories) {
    this.categories = categories;
  }

}
