package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.ClientSGAFileParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parsers.elements.attributes.ArrayAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.BooleanAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.StringAttribute;
import csbase.logic.algorithms.parsers.elements.ParameterStructure;
import csbase.logic.algorithms.parsers.elements.ParsedSimpleParameter;
import csbase.logic.algorithms.parsers.elements.SimpleParameterStructure;

/**
 * Fbrica de parmetros do tipo {@link ClientSGAFileParameter}.
 *
 * @author Tecgraf
 */
public class ClientSGAFileParameterFactory extends
  SimpleParameterParser<ClientSGAFileParameter> {

  /**
   * Atributo que define se o usurio pode escolher mais de um arquivo do SGA.
   * NOTA: este atributo s  usado se o valor do atributo
   * {@link ClientSGAFileParameterFactory#MODE_ATTRIBUTE} for "entrada".
   */
  private static final String MULTIPLE_SELECTION_ATTRIBUTE = "selecao_multipla";

  /**
   * Atributo que define se o usurio escolhe arquivos ou diretrios.
   */
  private static final String CATEGORY_ATTRIBUTE = "categoria";

  /**
   * Define que o usurio pode escolher apenas arquivos.
   */
  private static final String CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE = "arquivo";

  /**
   * Define que o usurio pode escolher apenas diretrios.
   */
  private static final String CATEGORY_ATTRIBUTE_VALUE_DIRECTORY = "diretorio";

  /**
   * Atributo que define se o uso do parmetro {@link ClientSGAFileParameter} 
   * leitura ou escrita.
   */
  private static final String MODE_ATTRIBUTE = "modo";

  /**
   * Define que o usurio vai escolher arquivo(s) como entrada.
   */
  private static final String MODE_ATTRIBUTE_VALUE_INPUT = "entrada";

  /**
   * Define que o usurio vai escolher um arquivo como saida.
   */
  private static final String MODE_ATTRIBUTE_VALUE_OUTPUT = "saida";

  /**
   * Define uma lista de tipos a serem filtrados.
   */
  private static final String TYPE_ATTRIBUTE = "tipo";

  /**
   * Nome do parmetro.
   */
  public static final String CLIENT_SGA_PARAMETER_ELEMENT = "arquivo_sga";

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientSGAFileParameter createSimpleParameter(XmlParser parser,
    ParsedSimpleParameter definition, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {

    ClientSGAFileParameter parameter =
      new ClientSGAFileParameter(definition.getName(), definition.getLabel(),
        definition.getDescription(), null, definition.isOptional(),
        definition.isVisible(), definition.getCommandLinePattern());

    parameter.setMultipleSelection(
      definition.getAttributeValue(MULTIPLE_SELECTION_ATTRIBUTE));

    String category = parser.extractAttributeValue(CATEGORY_ATTRIBUTE,
      CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE);
    if (!category
      .equalsIgnoreCase(CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE) && !category
      .equalsIgnoreCase(CATEGORY_ATTRIBUTE_VALUE_DIRECTORY)) {
      category = CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE;
    }
    parameter.setShowFiles(
      category.equalsIgnoreCase(CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE));

    String mode =
      parser.extractAttributeValue(MODE_ATTRIBUTE, MODE_ATTRIBUTE_VALUE_INPUT);
    if (!mode.equalsIgnoreCase(MODE_ATTRIBUTE_VALUE_INPUT) && !mode
      .equalsIgnoreCase(MODE_ATTRIBUTE_VALUE_OUTPUT)) {
      mode = MODE_ATTRIBUTE_VALUE_INPUT;
    }
    parameter.setInputMode(mode.equalsIgnoreCase(MODE_ATTRIBUTE_VALUE_INPUT));

    parameter.setTypes(definition.getAttributeValue(TYPE_ATTRIBUTE));

    if (parameter.isMultipleSelection() && !parameter.isInputMode()) {
      throw new ParseException("O parmetro " + definition
        .getName() + " no pode ser configurado simultaneamente para seleo " +
        "" + "mltipla e modo save.");
    }

    return parameter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ParameterStructure<ClientSGAFileParameter> getParameterStructure() {
    ParameterStructure<ClientSGAFileParameter> structure =
      new SimpleParameterStructure<>(CLIENT_SGA_PARAMETER_ELEMENT,
        ClientSGAFileParameter.class);
    structure
      .addAttribute(new BooleanAttribute(MULTIPLE_SELECTION_ATTRIBUTE, false));
    structure.addAttribute(new StringAttribute(CATEGORY_ATTRIBUTE,
      CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE));
    structure.addAttribute(
      new StringAttribute(MODE_ATTRIBUTE, MODE_ATTRIBUTE_VALUE_INPUT));
    structure.addAttribute(new ArrayAttribute(TYPE_ATTRIBUTE, null));
    return structure;
  }

}
