/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.Serializable;
import java.util.Set;

import csbase.exception.ParseException;
import csbase.exception.algorithms.FormulaCreationException;
import csbase.exception.algorithms.FormulaEvaluationException;

/**
 * Uma expresso de validao.
 */
public final class ValidationExpression implements Serializable {

  /** A mensagem de erro. */
  private String errorMessage;

  /** A expresso. */
  private Expression expression;

  /** O configurador. */
  private SimpleAlgorithmConfigurator configurator;

  /**
   * Construtor.
   * 
   * @param configurator o configurador.
   * @param expressionText o texto que representa a expresso.
   * @param errorMessage a mensagem de erro.
   * 
   * @throws FormulaCreationException em caso de falha na criao da frmula.
   * @throws ParseException em caso de falha na leitura do texto.
   */
  public ValidationExpression(SimpleAlgorithmConfigurator configurator,
    String expressionText, String errorMessage)
    throws FormulaCreationException, ParseException {
    setConfigurator(configurator);
    setErrorMessage(errorMessage);
    this.expression = new Expression(expressionText);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    ValidationExpression exp = (ValidationExpression) obj;
    if (!getConfigurator().equals(exp.getConfigurator())) {
      return false;
    }
    return toString().equals(obj.toString());
  }

  /**
   * Avalia a expresso.
   * 
   * @return verdadeiro, se bem sucedido ou falso, caso contrrio.
   * @throws FormulaEvaluationException em caso de falha na avaliao.
   */
  public boolean evaluate() throws FormulaEvaluationException {
    return this.expression.evaluate(getConfigurator());
  }

  /**
   * Obtm o configurador.
   * 
   * @return o configurador.
   */
  public SimpleAlgorithmConfigurator getConfigurator() {
    return this.configurator;
  }

  /**
   * Obtm a mensagem de erro.
   * 
   * @return a mensagem de erro.
   */
  public String getErrorMessage() {
    return this.errorMessage;
  }

  /**
   * Obtm o conjunto de parmetros da expresso.
   * 
   * @return os parmetros.
   */
  public Set<SimpleParameter<?>> getParameters() {
    return this.expression.getParameters();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return toString().hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.expression.toString();
  }

  /**
   * Atribui o configurador.
   * 
   * @param config o configurador.
   */
  private void setConfigurator(SimpleAlgorithmConfigurator config) {
    if (config == null) {
      throw new IllegalArgumentException(
        "O parmetro configurator est null.\n");
    }
    this.configurator = config;
  }

  /**
   * Atribui a mensagem de erro.
   * 
   * @param errorMessage a mensagem de erro.
   */
  private void setErrorMessage(String errorMessage) {
    if (errorMessage == null) {
      throw new IllegalArgumentException(
        "O parmetro expressionText est nulo.\n");
    }
    this.errorMessage = errorMessage;
  }
}
