/*
 * CommandAction.java
 * 
 * $Author: cassino $ $Revision: 116055 $ - $Date: 2008-06-06 16:24:00 -0300 (Fri,
 * 06 Jun 2008) $
 */
package csbase.logic;

import csbase.remote.RemoteEvent;

/**
 * Representa eventos que podem ocorrer em um comando. Objetos dessa classe so
 * criados pelo servidor sempre que acontecer alguma modificao em um comando.
 * Os metodos desta classe nao devem lancar excecoes, uma vez que so chamados
 * por metodos que definem a idl sga-manager.
 */
public final class CommandEvent extends RemoteEvent {
  /**
   * @author ururahy
   * 
   */
  public enum EventType {

    /** Tipo de evento que indica que um comando foi perdido. */
    LOST,
    /** Tipo de evento que indica que um comando foi abortado. */
    KILLED,
    /** Tipo de evento que indica que um comando nao foi iniciado. */
    INIT_FAILURE,
    /**
     * Tipo de evento que indica que um comando terminou (mas no se sabe como).
     */
    COMPLETED,
    /** Tipo de evento que indica que um comando terminou com erro. */
    ERROR,
    /** Tipo de evento que indica que um comando terminou com sucesso. */
    SUCCESS,
    /**
     * Tipo de evento que indica que um comando terminou, mas no foi possvel
     * ler o cdigo de retorno, para determinar se foi bem sucedido.
     */
    NO_CODE;
  }

  /**
   * Indica qual foi o evento ocorrido no servidor. Pode ser uma das constantes
   * definidas nessa classe.
   */
  private Object commandId;

  /** Indica o tipo do evento. */
  private EventType type;

  /** Informaes sobre a finalizao do comando. */
  private CommandFinalizationInfo finalizationInfo;

  /**
   * Tempo de parede (em segundos).
   */
  private Integer elapsedTime;
  /**
   * Tempo em modo usurio (em segundos).
   */
  private Integer userTime;
  /**
   * Tempo em modo sistema (em segundos).
   */
  private Integer cpuTime;

  /**
   * Cria um evento de comando.
   * 
   * @param commandId O identificador do comando.
   * @param type O tipo do comando.
   * @param finalizationInfo Informaes sobre a finalizao do comando.
   */
  public CommandEvent(Object commandId, EventType type,
    CommandFinalizationInfo finalizationInfo) {
    this(commandId, type, finalizationInfo, null, null, null);
  }

  /**
   * Cria um evento.
   * 
   * @param commandId Identificador do comando
   * @param type O tipo do evento.
   * @param finalizationInfo Informaes sobre a finalizao do comando.
   * @param elapsedTime Tempo de parede (em segundos).
   * @param userTime Tempo em modo usurio (em segundos).
   * @param cpuTime Tempo em modo sistema (em segundos).
   */
  public CommandEvent(Object commandId, EventType type,
    CommandFinalizationInfo finalizationInfo, Integer elapsedTime,
    Integer userTime, Integer cpuTime) {
    this.commandId = commandId;
    this.type = type;
    this.finalizationInfo = finalizationInfo;
    setElapsedTime(elapsedTime);
    setUserTime(userTime);
    setCpuTime(cpuTime);
  }

  /**
   * Retorna as informaes sobre a finalizao do comando.
   * 
   * @return finalizationInfo as informaes.
   */
  public CommandFinalizationInfo getFinalizationInfo() {
    return this.finalizationInfo;
  }

  /**
   * Obtem o identificador do comando.
   * 
   * @return .
   */
  public Object getCommandId() {
    return this.commandId;
  }

  /**
   * Obtm o tipo do evento.
   * 
   * @return O tipo do evento.
   */
  public EventType getType() {
    return this.type;
  }

  /**
   * Obtm o tempo de parede.
   * 
   * @return O tempo de parede.
   */
  public Integer getElapsedTime() {
    return this.elapsedTime;
  }

  /**
   * Obtm o tempo do usurio.
   * 
   * @return O tempo do usurio.
   */
  public Integer getUserTime() {
    return this.userTime;
  }

  /**
   * Obtm o tempo do sistema.
   * 
   * @return O tempo do sistema.
   */
  public Integer getCpuTime() {
    return this.cpuTime;
  }

  /**
   * Atribui o tempo de parede.
   * 
   * @param elapsedTime O tempo de parede.
   */
  private void setElapsedTime(Integer elapsedTime) {
    if (elapsedTime != null && elapsedTime < 0) {
      throw new IllegalArgumentException("elapsedTime < 0 ");
    }

    this.elapsedTime = elapsedTime;
  }

  /**
   * Atribui o tempo do usurio.
   * 
   * @param userTime O tempo do usurio.
   */
  private void setUserTime(Integer userTime) {
    if (userTime != null && userTime < 0) {
      throw new IllegalArgumentException("userTime < 0 ");
    }

    this.userTime = userTime;
  }

  /**
   * Atribui o tempo do sistema.
   * 
   * @param cpuTime O tempo do sistema.
   */
  private void setCpuTime(Integer cpuTime) {
    if (cpuTime != null && cpuTime < 0) {
      throw new IllegalArgumentException("cpuTime < 0 ");
    }

    this.cpuTime = cpuTime;
  }
}
