package csbase.logic.algorithms;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Contexto para a criao da linha de comando de um fluxo. Contm os contextos
 * para a criao da linha de comando dos seus ns.
 *
 * @author isabella.
 */
public class FlowCommandLineContext extends CommandLineContext {

  /**
   * Contextos de execuo para ns utilizados na execuo de fluxos mapeados
   * pelos identificadores dos ns.
   */
  private Map<Integer, CommandLineContext> flowNodeContexts;

  /**
   * Pipes utilizados na conexo entre ns de fluxo em parmetros de sada,
   * mapeados pelos identificadores dos ns.
   */
  private Map<Integer, Map<String, String>> fromPipes;

  /**
   * Pipes utilizados na conexo entre ns de fluxo em parmetros de entrada,
   * mapeados pelos identificadores dos ns.
   */
  private Map<Integer, Map<String, String>> toPipes;

  /**
   * Diretrios de ligao utilizados na conexo entre ns de fluxo, mapeados
   * pelos identificadores dos ns.
   */
  private Map<Integer, Map<String, String>> linkDirectories;

  /**
   * Construtor.
   *
   * @param context o contexto com informaes da execuo simples.
   * @param fromPipes o mapa de pipes dos parmetros de sada utilizados nas
   *        conexes entre ns, indexados pelo identificador dos ns.
   * @param toPipes o mapa de pipes dos parmetros de entrada utilizados nas
   *        conexes entre ns, indexados pelo identificador dos ns.
   * @param linkDirectories o mapa de diretrios de ligao utilizados nas
   *        conexes entre ns, indexados pelo identificador dos ns.
   */
  protected FlowCommandLineContext(CommandLineContext context,
    Map<Integer, Map<String, String>> fromPipes,
    Map<Integer, Map<String, String>> toPipes,
    Map<Integer, Map<String, String>> linkDirectories) {
    super(context);
    if (fromPipes == null) {
      this.fromPipes = Collections.emptyMap();
    }
    else {
      this.fromPipes = fromPipes;
    }
    if (toPipes == null) {
      this.toPipes = Collections.emptyMap();
    }
    else {
      this.toPipes = toPipes;
    }
    if (linkDirectories == null) {
      this.linkDirectories = Collections.emptyMap();
    }
    else {
      this.linkDirectories = linkDirectories;
    }
    this.flowNodeContexts = new HashMap<Integer, CommandLineContext>();
  }

  /**
   * Adiciona o contexto para a gerao da linha de comando de um n de um
   * fluxo.
   *
   * @param nodeId O identificador do n.
   * @param flowNodeContext O contexto para gerao da linha de comando do n.
   */
  public void addFlowNodeContext(Integer nodeId,
    CommandLineContext flowNodeContext) {
    this.flowNodeContexts.put(nodeId, flowNodeContext);
  }

  /**
   * Obtm o contexto para a gerao da linha de comando de um n de um fluxo.
   *
   * @param nodeId O identificador do n.
   * @return o contexto para a gerao da linha de comando do n.
   */
  public CommandLineContext getFlowNodeContext(Integer nodeId) {
    return this.flowNodeContexts.get(nodeId);
  }

  /**
   * Apaga todos os contextos para a gerao da linha de comando de ns de um
   * fluxo.
   */
  public void clearFlowNodeContexts() {
    this.flowNodeContexts.clear();
  }

  /**
   * Retorna os pipes utilizados na conexo entre ns de fluxo em parmetros de
   * sada, mapeados pelos identificadores dos ns.
   *
   * @return o mapa de pipes.
   */
  public Map<Integer, Map<String, String>> getFromPipes() {
    return fromPipes;
  }

  /**
   * Pipes utilizados na conexo entre ns de fluxo em parmetros de entrada,
   * mapeados pelos identificadores dos ns.
   *
   * @return o mapa de pipes.
   */
  public Map<Integer, Map<String, String>> getToPipes() {
    return toPipes;
  }

  /**
   * Diretrios de ligao utilizados na conexo entre ns de fluxo, mapeados
   * pelos identificadores dos ns.
   *
   * @return o mapa de diretrios.
   */
  public Map<Integer, Map<String, String>> getLinkDirectories() {
    return linkDirectories;
  }

  /**
   * Cria o contexto para a execuo de um fluxo.
   *
   * @param context o contexto com informaes da execuo simples.
   * @param fromPipes o mapa de pipes dos parmetros de sada utilizados nas
   *        conexes entre ns, indexados pelo identificador dos ns.
   * @param toPipes o mapa de pipes dos parmetros de entrada utilizados nas
   *        conexes entre ns, indexados pelo identificador dos ns.
   * @param linkDirectories o mapa de diretrios de ligao utilizados nas
   *        conexes entre ns, indexados pelo identificador dos ns.
   * @return o contexto para execuo de fluxo.
   */
  public static FlowCommandLineContext createFlowContext(
    CommandLineContext context, Map<Integer, Map<String, String>> fromPipes,
    Map<Integer, Map<String, String>> toPipes,
    Map<Integer, Map<String, String>> linkDirectories) {
    return new FlowCommandLineContext(context, fromPipes, toPipes,
      linkDirectories);
  }
}
