/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import csbase.exception.ParseException;
import tecgraf.javautils.core.lng.LNG;
import csbase.logic.algorithms.parameters.validators.DoubleParameterValidator;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;

/**
 * Coluna de reais.
 *
 * @author lmoreira
 */
public final class DoubleColumn extends BasicColumn<Double> {
  /**
   * Indica se o valor mximo deve ser considerado vlido.
   */
  private boolean isMaximumIncluded;

  /**
   * Indica se o valor mnimo deve ser considerado vlido.
   */
  private boolean isMinimumIncluded;

  /**
   * O valor mximo (Pode ser {@code null}).
   */
  private Double maximum;

  /**
   * O valor mnimo (Pode ser {@code null}).
   */
  private Double minimum;

  /**
   * Cria uma coluna.
   *
   * @param id O identificador (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param defaultValue O valor-padro (No aceita {@code null}).
   * @param isOptional Indica se o valor da clula  opcional.
   * @param isEditable Indica se a coluna  editvel.
   * @param maximum O valor mximo (Aceita {@code null}).
   * @param isMaximumIncluded Indica se o valor mximo deve ser considerado
   *        vlido.
   * @param minimum O valor mnimo (Aceita {@code null}).
   * @param isMinimumIncluded Indica se o valor mnimo deve ser considerado
   *        vlido.
   */
  public DoubleColumn(String id, String label, Double defaultValue,
    boolean isOptional, boolean isEditable, Double maximum,
    boolean isMaximumIncluded, Double minimum, boolean isMinimumIncluded) {
    super(id, label, defaultValue, isOptional, isEditable);
    setMaximum(maximum, isMaximumIncluded);
    setMinimum(minimum, isMinimumIncluded);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Double getItemValueFromText(String itemValue) throws ParseException {
    if (itemValue == null) {
      return null;
    }
    try {
      double doubleValue = Double.parseDouble(itemValue);
      return new Double(doubleValue);
    }
    catch (NumberFormatException e) {
      throw new ParseException(LNG.get(
    		  "csbase.logic.algorithms.parameters.InvalidDouble"),
    		  new Object[] { itemValue, this });
    }
  }

  /**
   * Obtm o valor mximo.
   *
   * @return O valor mximo (Pode ser {@code null}).
   */
  public Double getMaximum() {
    return this.maximum;
  }

  /**
   * Obtm o valor mnimo.
   *
   * @return O valor mnimo (Pode ser {@code null}).
   */
  public Double getMinimum() {
    return this.minimum;
  }

  /**
   * Indica se o valor mximo deve ser includo nos valores vlidos.
   *
   * @return .
   */
  public boolean isMaximumIncluded() {
    return this.isMaximumIncluded;
  }

  /**
   * Indica se o valor mnimo deve ser includo nos valores vlidos.
   *
   * @return .
   */
  public boolean isMinimumIncluded() {
    return this.isMinimumIncluded;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SimpleParameterValidator<Double> getValidator() {
    return new DoubleParameterValidator(isOptional(), getMaximum(),
      isMaximumIncluded(), getMinimum(), isMinimumIncluded());
  }

  /**
   * Atribui o valor mximo.
   *
   * @param maximum O valor mximo (Aceita {@code null}).
   * @param isMaximumIncluded Indica se o valor mximo deve ser considerado um
   *        dos valores vlidos.
   */
  private void setMaximum(Double maximum, boolean isMaximumIncluded) {
    if (this.minimum != null) {
      if (maximum != null && maximum.compareTo(this.minimum) < 0) {
        throw new IllegalArgumentException(LNG.get(
        		"csbase.logic.algorithms.parameters.MaxLessThanMin"));
      }
    }
    if (maximum == null) {
      this.isMaximumIncluded = false;
    }
    else {
      this.isMaximumIncluded = isMaximumIncluded;
    }
    this.maximum = maximum;
  }

  /**
   * Atribui o valor mnimo.
   *
   * @param minimum O valor mnimo (Aceita {@code null}).
   * @param isMinimumIncluded Indica se o valor mnimo deve ser considerado um
   *        dos valores vlidos.
   */
  private void setMinimum(Double minimum, boolean isMinimumIncluded) {
    if (this.maximum != null) {
      if (minimum != null && minimum.compareTo(this.maximum) > 0) {
        throw new IllegalArgumentException(LNG.get(
          "csbase.logic.algorithms.parameters.MinGreaterThenMax"));
      }
    }
    if (minimum == null) {
      this.isMinimumIncluded = false;
    }
    else {
      this.isMinimumIncluded = isMinimumIncluded;
    }
    this.minimum = minimum;
  }
}
