/*
 * $Author$ $Date$ $Revision$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import tecgraf.javautils.core.lng.LNG;
import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.EnumerationParameterValidator;

/**
 * <p>
 * Parmetro enumerao
 * </p>
 *
 * <p>
 * O parmetro enumerao  utilizado para parmetros que aceitem um conjunto
 * pequeno e discreto de dados. Exemplo: dias da semana.
 * </p>
 *
 * <p>
 * Um parmetro enumerao possui 1 ou mais tems de enumerao (
 * {@link EnumerationItem}), porm apenas 1 nico item est selecionado por vez.
 * O item selecionado pode ser obtido utilizando o mtodo {@link #getValue()} e
 * pode ser modificado utilizando o mtodo
 * {@link SimpleParameter#setValue(Object) setValue(Object)}.
 * </p>
 *
 * @author lmoreira
 */
public final class EnumerationParameter extends
SimpleParameter<EnumerationItem> {
  /**
   * Um representao textual do tipo deste parmetro.
   */
  public static final String TYPE = "ENUMERATION";

  /**
   * Os itens desta enumerao.
   */
  private List<EnumerationItem> items;

  /**
   * Os observadores deste parmetro.
   */
  private transient List<EnumerationParameterListener> listeners;

  /**
   * Indica se a enumerao deve estar ordenada.
   */
  private boolean mustSortItems;

  /**
   * Cria um parmetro enumerao.
   *
   * @param name O nome do parmetro (No aceita {@code null}).
   * @param label O rtulo do parmetro (No aceita {@code null}).
   * @param description A descrio do parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro do parmetro (No aceita {@code null}).
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param items Os itens da enumerao (No aceita {@code null}), o
   *        valor-padro deve ser um dos itens.
   * @param mustSortItems indica se a enumerao deve estar ordenada.
   */
  public EnumerationParameter(String name, String label, String description,
    EnumerationItem defaultValue, boolean isVisible, String commandLinePattern,
    List<EnumerationItem> items, boolean mustSortItems) {
    super(name, label, description, defaultValue, false, isVisible,
      commandLinePattern);
    listeners = new LinkedList<EnumerationParameterListener>();
    if (items == null) {
      throw new IllegalArgumentException("items == null");
    }
    this.items = new ArrayList<EnumerationItem>(items);
    this.mustSortItems = mustSortItems;
    if (this.mustSortItems) {
      Collections.sort(this.items);
    }
  }

  /**
   * Adiciona um observador a este parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public void addEnumerationParameterListener(
    EnumerationParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("listener == null");
    }
    listeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    Map<String, Object> parameterValues = new HashMap<String, Object>();
    EnumerationItem item = getValue();
    if (item != null) {
      parameterValues.put(getName(), item.getId());
    }
    return parameterValues;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void importValue(Map<String, Object> parameterValues) {
    if (parameterValues == null) {
      throw new IllegalArgumentException("parameterValues == null");
    }
    String id = (String) parameterValues.get(getName());
    EnumerationItem item = getItem(id);
    if (item != null) {
      setValue(item);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationItem getDefaultValue() {
    EnumerationItem item = super.getDefaultValue();
    if (item == null) {
      return null;
    }
    if (item.isVisible()) {
      return item;
    }
    for (EnumerationItem anItem : getItems()) {
      if (anItem.isVisible()) {
        return anItem;
      }
    }
    return null;
  }

  /**
   * Um item de enumerao no pode ser utilizado em expresses.
   *
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return null;
  }

  /**
   * Obtm um item de enumerao dado um identificador.
   *
   * @param id O identificador (No aceita {@code null}).
   *
   * @return O item de enumerao ou {@code null} se no houver um item de
   *         enumerao com o identificador fornecido.
   */
  public EnumerationItem getItem(String id) {
    Iterator<EnumerationItem> itemIterator = this.items.iterator();
    while (itemIterator.hasNext()) {
      EnumerationItem item = itemIterator.next();
      if (item.getId().equals(id)) {
        return item;
      }
    }
    return null;
  }

  /**
   * <p>
   * Obtm uma lista de itens de enumerao.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return A lista de itens.
   */
  public List<EnumerationItem> getItems() {
    return Collections.unmodifiableList(this.items);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * O valor retornado  o identificador do item que est selecionado no
   * momento.
   *
   * {@inheritDoc}
   */
  @Override
  public String getValueAsText() {
    EnumerationItem item = getValue();
    if (item == null) {
      return null;
    }
    return item.getId();
  }

  /**
   * <p>
   * Exibe/Oculta um item de enumerao.
   * </p>
   *
   * <p>
   * Se bem sucedido, gera o evento
   * {@link EnumerationParameterListener#enumerationItemWasSetVisible(EnumerationParameter, EnumerationItem)}
   * .
   * </p>
   *
   * <p>
   * Gera o evento
   * {@link SimpleParameterListener#valueWasChanged(SimpleParameter)} o item de
   * item que est selecionado no momento for o item desejado e est tentando
   * desabilitar o item.
   * </p>
   *
   * @param itemId O identificador do item (No aceita {@code null}).
   * @param isVisible Indica se deve habilitar ({@code true}) ou desabilitar (
   *        {@code null}) o item.
   *
   * @return {@code false} se o item no existir ou se ele no mudar de estado.
   */
  public boolean setVisibleItem(String itemId, boolean isVisible) {
    EnumerationItem theItem = getItem(itemId);
    if (theItem == null) {
      return false;
    }
    if (theItem.isVisible() == isVisible) {
      return false;
    }
    theItem.setVisible(isVisible);
    if (getValue() == null || !getValue().isVisible()) {
      EnumerationItem newValue = null;
      for (EnumerationItem anItem : getItems()) {
        if (anItem.isVisible()) {
          newValue = anItem;
          break;
        }
      }
      setValue(newValue);
    }
    fireEnumerationItemWasSetVisible(theItem);
    return true;
  }

  /**
   * O valor utilizado  o identificador de um item de enumerao.
   *
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    EnumerationItem item;
    if (parameterValue == null) {
      item = getDefaultValue();
    }
    else {
      item = getItem(parameterValue);
      if (item == null) {
        throw new ParseException(LNG.get(
    		"csbase.logic.algorithms.parameters.EnumOutsiderIndex"),
          parameterValue, this, getItems());
      }
    }
    setValue(item);
  }

  /**
   * Indica se a enumerao deve estar ordenada.
   *
   * @return .
   */
  boolean mustSortItems() {
    return mustSortItems;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCommandValue(CommandLineContext context) {
    EnumerationItem item = getValue();
    if (item == null) {
      return null;
    }
    return item.getCommandValue(context);
  }

  /**
   * Dispara o evento
   * {@link EnumerationParameterListener#enumerationItemWasSetVisible(EnumerationParameter, EnumerationItem)}
   *
   * @param item O item modificador (No aceita {@code null}).
   */
  private void fireEnumerationItemWasSetVisible(EnumerationItem item) {
    for (EnumerationParameterListener listener : listeners) {
      listener.enumerationItemWasSetVisible(this, item);
    }
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in Leitor de objetos
   *
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
  ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<EnumerationParameterListener>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationParameterValidator createParameterValidator() {
    return new EnumerationParameterValidator(isOptional());
  }

}
