package csbase.logic.algorithms.parameters.triggers;

import java.text.MessageFormat;

import csbase.logic.algorithms.parameters.DoubleListParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.conditions.Condition;
import tecgraf.javautils.core.lng.LNG;

/**
 * Gatilho Trocar Mximo para Parmetros com Nmeros Reais: Modifica o valor
 * mximo para parmetro do tipo {@link DoubleParameter Real} e
 * {@link DoubleListParameter Lista de Reais} quando a {@link Condition
 * condio} for aceita.
 * 
 * @author lmoreira
 */
public final class ChangeMaximumForDoublesTrigger extends
  Trigger<SimpleParameter<?>> {
  /**
   * Indica se o valor mximo deve ser considerado vlido.
   */
  private boolean isMaximumIncluded;

  /**
   * Mximo ou {@code null} se ele no existir.
   */
  private Double maximum;

  /**
   * Cria o gatilho.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param condition A condio (No aceita {@code null}).
   * @param maximum O mximo (Aceita {@code null}).
   * @param isMaximumIncluded Indica se o valor mximo deve ser considerado
   *        vlido.
   */
  public ChangeMaximumForDoublesTrigger(DoubleParameter parameter,
    Condition condition, Double maximum, boolean isMaximumIncluded) {
    super(parameter, condition);
    this.isMaximumIncluded = isMaximumIncluded;
    this.maximum = maximum;
  }

  /**
   * Cria o gatilho.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param condition A condio (No aceita {@code null}).
   * @param maximum O mximo (Aceita {@code null}).
   * @param isMaximumIncluded Indica se o valor mximo deve ser considerado
   *        vlido.
   */
  public ChangeMaximumForDoublesTrigger(DoubleListParameter parameter,
    Condition condition, Double maximum, boolean isMaximumIncluded) {
    super(parameter, condition);
    this.isMaximumIncluded = isMaximumIncluded;
    this.maximum = maximum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (!super.equals(obj)) {
      return false;
    }
    ChangeMaximumForDoublesTrigger trigger =
      (ChangeMaximumForDoublesTrigger) obj;
    if (this.isMaximumIncluded != trigger.isMaximumIncluded) {
      return false;
    }
    if (this.maximum == null && trigger.maximum == null) {
      return true;
    }
    return (this.maximum != null && this.maximum.equals(trigger.maximum));
  }

  /**
   * Obtm o mximo.
   * 
   * @return O mximo ou {@code null} se ele no existir.
   */
  public Double getMaximum() {
    return this.maximum;
  }

  /**
   * Indica se o valor mximo deve ser considerado um dos valores vlidos.
   * 
   * @return .
   */
  public boolean isMaximumIncluded() {
    return this.isMaximumIncluded;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void doAction() {
    SimpleParameter<?> parameter = getParameter();
    if (parameter instanceof DoubleListParameter) {
      DoubleListParameter doubleListParameter = (DoubleListParameter) parameter;
      doubleListParameter.setMaximum(getMaximum(), isMaximumIncluded());
    }
    else if (parameter instanceof DoubleParameter) {
      DoubleParameter doubleParameter = (DoubleParameter) parameter;
      doubleParameter.setMaximum(getMaximum(), isMaximumIncluded());
    }
    else {
      throw new IllegalStateException(MessageFormat.format(
    	LNG.get("csbase.logic.algorithms.parameters.InvalidType"),
    		new Object[] {parameter.getLabel(), "real", "lista_de_reais" }));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void undoAction() {
    // @todo Auto-generated method stub
  }
}
