/*
 * CommandSubmission.java
 * 
 * $author$ $date$ $revision$
 */
package csbase.logic;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import tecgraf.javautils.core.io.FileUtils;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.serializer.AlgorithmConfigurationSerializerFactory;
import csbase.logic.algorithms.serializer.IAlgorithmConfigurationSerializer;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerException;

/**
 * Representa a submisso do comando.
 *
 * Para evitar que o servidor execute uma linha de um comando que no representa
 * nenhum algoritmo cadastrado na rvore de algoritmos, no enviamos para o
 * servidor o objeto configurador diretamente e sim ele serializado como um
 * array de bytes junto com o nome da classe do serializador que sero
 * utilizados para recriar o configurador atravs do mtodo
 * {@link #createAlgorithmConfigurator()}.
 */
public class CommandSubmission implements Serializable {
  /** String representando o configurador. */
  private String configuratorToString;
  /**
   * Nome da classe do serializador utilizado para transformar o configurador em
   * um array de bytes.
   */
  private String configuratorSerializerClassName;
  /** Array de bytes representando o configurador. */
  private byte[] configuratorData;

  /** Descrio para o comando. */
  private String description;

  /** Nmero de execues que sero disparadas na execuo mltipla. */
  private Integer executionCountForMultipleExecution;

  /**
   * O nmero de execues por sga que ser utilizado na execuo mltipla
   * manual.
   */
  private Integer executionCountPerSGAForMultipleExecution;

  /**
   * Define o tipo de execuo: simples ou mltipla.
   */
  private ExecutionType executionType;

  /** Indica se um e-mail de aviso ser enviado ao trmino do comando. */
  private boolean mailAtEnd;

  /**
   * Array com endereos que devem receber um e-mail de aviso ao trmino do
   * comando.
   */
  private String[] emailList;

  /** Indica qual a plataforma usada na seleo automtica de SGAs. */
  private String platform;

  /** Prioridade do comando. */
  private Priority priority;

  /** Identificador do projeto. */
  private Object projectId;

  /**
   * Servidores escolhidos pelo usurio para execuo caso a execuo seja
   * manual.
   */
  private List<String> sgaNames;

  /** Data e hora da submisso do comando. */
  private Date submittedDate;

  /**
   * Mapa de informaes extras
   */
  private Map<String, String> extraInfoMap;

  /**
   * Constri o um pedido de submisso de comando.
   *
   * @param configurator o configurador do algoritmo.
   * @param projectId o identificador do projeto.
   */
  public CommandSubmission(AlgorithmConfigurator configurator, Object projectId) {
    setAlgorithmConfigurator(configurator);
    this.projectId = projectId;
    this.sgaNames = new LinkedList<String>();
    this.priority = Priority.getDefault();
    this.submittedDate = new Date();
    this.extraInfoMap = new HashMap<String, String>();
    configureSimpleExecution();
  }

  /**
   * Configura os dados da execuo mltipla.
   *
   * @param executionCount Nmero de execues que sero disparadas.
   */
  public void configureMultipleExecution(int executionCount) {
    executionType = ExecutionType.MULTIPLE;
    this.executionCountForMultipleExecution = executionCount;
  }

  /**
   * Configura os dados da execuo mltipla.
   *
   * @param sgaNames Nomes dos sgas selecionados pelo usurio.
   * @param executionCountPerSGA Nmero de execues por servidor.
   */
  public void configureMultipleExecution(List<String> sgaNames,
    int executionCountPerSGA) {
    executionType = ExecutionType.MULTIPLE;
    setSGANames(sgaNames);
    this.executionCountPerSGAForMultipleExecution = executionCountPerSGA;
    this.executionCountForMultipleExecution =
      executionCountPerSGA * this.sgaNames.size();
  }

  /**
   * Configura os dados da execuo simples.
   */
  public void configureSimpleExecution() {
    configureSimpleExecution(null);
  }

  /**
   * Configura os dados da execuo simples.
   *
   * @param sgaName Nome do sgas selecionado pelo usurio.
   */
  public void configureSimpleExecution(String sgaName) {
    executionType = ExecutionType.SIMPLE;
    setSGAName(sgaName);
  }

  /**
   * <p>
   * Cria o configurador do algoritmo.
   * </p>
   * <p>
   * Para evitar que o servidor execute uma linha de um comando que no
   * representa nenhum algoritmo cadastrado na rvore de algoritmos, no
   * enviamos para o servidor o objeto configurador diretamente e sim ele
   * serializado como um array de bytes junto com o nome da classe do
   * serializador. Este mtodo faz uso destes dados e utilizando o serializador
   * ele recria o configurador de algoritmos.
   * </p>
   *
   * @return um configurador do algoritmo.
   *
   * @throws AlgorithmConfigurationSerializerException ocorreram problemas na
   *         desserializao dos dados que representam o configurador.
   * @throws IOException ocorreu um erro ao tentar fechar um stream.
   */
  public AlgorithmConfigurator createAlgorithmConfigurator()
    throws AlgorithmConfigurationSerializerException, IOException {
    ByteArrayInputStream serializerInput = null;
    try {
      IAlgorithmConfigurationSerializer serializer =
        AlgorithmConfigurationSerializerFactory
          .getSerializer(configuratorSerializerClassName);

      serializerInput = new ByteArrayInputStream(configuratorData);
      return serializer.read(serializerInput);
    }
    finally {
      FileUtils.close(serializerInput);
    }
  }

  /**
   * Obtm a descrio do comando.
   *
   * @return .
   */
  public String getDescription() {
    return description;
  }

  /**
   * Obtm o nmero de execues que sero disparadas na execuo mltipla.
   *
   * @return nmero de execues
   */
  public Integer getExecutionCountForMultipleExecution() {
    return executionCountForMultipleExecution;
  }

  /**
   * Obtm o nmero de execues por sga que ser utilizado na execuo mltipla
   * manual.
   *
   * @return .
   */
  public Integer getExecutionCountPerSGAForMultipleExecution() {
    return executionCountPerSGAForMultipleExecution;
  }

  /**
   * Obtm o tipo da execuo do comando {@link ExecutionType}.
   *
   * @return .
   */
  public ExecutionType getExecutionType() {
    return executionType;
  }

  /**
   * Obtm a plataforma na qual o comando dever ser executado.
   *
   * @return plataforma .
   */
  public String getPlatform() {
    return platform;
  }

  /**
   * Obtm a prioridade do comando.
   *
   * @return .
   */
  public Priority getPriority() {
    return priority;
  }

  /**
   * Obtm o identificador do projeto no qual o comando foi submetido.
   *
   * @return .
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Retorna os nomes dos SGAs que foram selecionados pelo usurio.
   *
   * @return sgas Os nomes dos SGAs que foram selecionados pelo usurio.
   */
  public List<String> getSGANames() {
    return sgaNames;
  }

  /**
   * Obtm a data em que o comando foi submetido.
   *
   * @return .
   */
  public Date getSubmittedDate() {
    return submittedDate;
  }

  /**
   * Obtm o mapa de informaes extra
   */
  public Map<String, String> getExtraInfoMap() {
    return this.extraInfoMap;
  }

  /**
   * Indica se a seleo de servidores de algoritmos dever ser feita pelo
   * escalonador.
   *
   * @return true se a seleo dever ser feita pelo escalonador e false caso
   *         contrrio.
   */
  public boolean isAutomatic() {
    return sgaNames.isEmpty();
  }

  /**
   * Indica se o sistema deve enviar um email para o usurio que submeteu o
   * comando quando este terminar.
   *
   * @return true se deve enviar o email e false caso contrrio.
   */
  public boolean isMailAtEnd() {
    return mailAtEnd;
  }

  /**
   * Indica se a seleo de servidores de algoritmos j foi feita pelo usurio.
   *
   * @return true se a seleo j foi feita pelo usurio.
   */
  public boolean isManual() {
    return !isAutomatic();
  }

  /**
   * Define a descrio do comando.
   *
   * @param description a descrio do comando.
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Define se o usurio deve recever um email quando o comando terminar.
   *
   * @param mailAtEnd true se o usurio deve receber o email, false caso
   *        contrrio.
   */
  public void setMailAtEnd(boolean mailAtEnd) {
    this.mailAtEnd = mailAtEnd;
  }

  /**
   * Define a plataforma na qual o comando ser ser executado.
   *
   * @param platform a plataforma na qual o comando ser ser executado.
   */
  public void setPlatform(String platform) {
    this.platform = platform;
  }

  /**
   * Define a prioridade do comando.
   *
   * @param priority a prioridade do comando.
   */
  public void setPriority(Priority priority) {
    this.priority = priority;
  }

  /**
   * Mtodo criado para auxiliar nos testes.
   */
  @Override
  public String toString() {
    StringBuffer buffer = new StringBuffer();
    buffer.append("Submisso:\n");
    buffer.append("configurador: " + configuratorToString + "\n");
    buffer.append("descrio: " + description + "\n");
    buffer.append("prioridade: " + priority + "\n");
    return buffer.toString();
  }

  /**
   * Define o nome do SGA que deve executar o comando.
   *
   * @param sgaName o nome do SGA que deve executar o comando, ou null, caso a
   *        seleo de servidores de algoritmos deva ser feita pelo escalonador
   */
  private void setSGAName(String sgaName) {
    this.sgaNames.clear();
    if (sgaName != null) {
      this.sgaNames.add(sgaName);
    }
  }

  private void setSGANames(List<String> sgaNames) {
    this.sgaNames.clear();
    this.sgaNames.addAll(sgaNames);
  }

  private void setAlgorithmConfigurator(AlgorithmConfigurator configurator) {

    ByteArrayOutputStream serializerOutput = null;
    try {
      IAlgorithmConfigurationSerializer serializer;
      if (configurator.getConfiguratorType() == ConfiguratorType.FLOW
        && configurator.getAlgorithmVersion() == null) {
        serializer =
          AlgorithmConfigurationSerializerFactory
            .getSerializer(FlowAlgorithmConfigurator.class);
      }
      else {
        serializer =
          AlgorithmConfigurationSerializerFactory.getDefaultSerializer();
      }
      configuratorSerializerClassName = serializer.getClass().getName();

      serializerOutput = new ByteArrayOutputStream();
      serializer.write(configurator, serializerOutput);
      configuratorData = serializerOutput.toByteArray();
      configuratorToString = configurator.toString();
    }
    catch (AlgorithmConfigurationSerializerException e) {
      configuratorSerializerClassName = null;
      configuratorData = null;
      throw new IllegalArgumentException(e);
    }
    finally {
      FileUtils.close(serializerOutput);
    }
  }

  /**
   * Define o mapa de informaes extra
   */
  public void addExtraInfo(String key, String value) {
    this.extraInfoMap.put(key, value);
  }

  /**
   * Atribui uma lista de endereos que devem receber email de aviso de trmino
   * de execuo do comando. Se essa lista de emails no estiver configurada,
   * apenas o usurio dono do projeto receber o email. Os emails somente so
   * enviados se estiver configurado para envio de emails.
   *
   * @see #setMailAtEnd(boolean)
   * @param emailList o array com endereos de email
   */
  public void setEmailList(String[] emailList) {
    this.emailList = emailList;
  }

  /**
   * Obtm a lista de endereos que devem receber email de aviso de trmino de
   * execuo do comando. Os emails somente so enviados se estiver configurado
   * para envio de emails.
   *
   * @see #setMailAtEnd(boolean)
   * @return o array com endereos de email
   */
  public String[] getEmailList() {
    return emailList;
  }
}
