/*
 * UserOutline.java
 * 
 * $Id: UserOutline.java 107622 2010-07-06 16:18:07Z pasti $
 */
package csbase.logic;

import java.io.Serializable;
import java.text.Collator;
import java.util.Comparator;

import csbase.util.StringUtils;

/**
 * A classe <code>UserOutline</code> representa informaes resumidas de um
 * usurio: identificao de login e o seu grupo. Guarda tambm a identificao
 * do usurio correspondente.
 * 
 * @author Maria Julia Lima
 * @version $Revision: 107622 $
 */
public class UserOutline implements Serializable {
  /** Identificador do usurio */
  private Object userId;
  /** Login do usurio */
  private String login;
  /** Endereos de e-mail do usurio */
  private String[] emails;
  /** Nome do usurio */
  private String name;
  /** Nome do grupo do usurio */
  private String userGroupName;

  /**
   * Obtm a identificao do usurio.
   * 
   * @return a identificao do usuario
   */
  public Object getId() {
    return userId;
  }

  /**
   * Obtm o login do usurio.
   * 
   * @return o login do usurio
   */
  public String getLogin() {
    return login;
  }

  /**
   * Obtm o nome do usurio.
   * 
   * @return o nome do usurio
   */
  public String getName() {
    return name;
  }

  /**
   * Obtm o email do usurio.
   * 
   * @return o email do usurio
   */
  public String[] getEmails() {
    return emails;
  }

  /**
   * Obtm o nome do UserGroup do usurio.
   * 
   * @return o nome do UserGroup do usuario
   */
  public String getUserGroup() {
    return userGroupName;
  }

  /**
   * Muda o UserGroup do usurio.
   * 
   * @param userGroupName o nome do novo grupo
   */
  public void setUserGroupName(String userGroupName) {
    this.userGroupName = userGroupName;
  }

  /**
   * Obtm um texto representativo desse objeto.
   * 
   * @return a descrio desse objeto
   */
  @Override
  public String toString() {
    String userGroupStr =
      userGroupName.trim().equals("") ? "" : (" - " + userGroupName);
    String naStr = getName().trim().equals("") ? "?" : getName();
    return naStr + " (" + this.login + ")" + userGroupStr;
  }

  /**
   * Obtm um comparator de <code>UserOutline</code> pelo critrio de ordem
   * alfabtica do nome do usurio.
   * 
   * @return um comparador pelo nome dos usurios
   */
  public static Comparator<UserOutline> getNameComparator() {
    /*
     * usamos Collator com strength = TERTIARY para garantir a correta ordenao
     * de acentos, minsculas e maisculas.  a mesma configurao usada nas
     * tabelas ordenveis do JavaUtils
     */
    Comparator<UserOutline> nameComparator = new Comparator<UserOutline>() {
      public int compare(UserOutline r1, UserOutline r2) {
        String n1 = r1.getName().trim().equals("") ? "???" : r1.getName();
        String n2 = r2.getName().trim().equals("") ? "???" : r2.getName();
        return StringUtils.compare(n1, n2);
      }
    };
    return nameComparator;
  }

  /**
   * Obtm um comparator de <code>UserOutline</code> pelo critrio de ordem
   * alfabtica do login do usurio.
   * 
   * @return um comparador por login
   */
  public static Comparator<UserOutline> getLoginComparator() {
    /*
     * usamos Collator com strength = TERTIARY para garantir a correta ordenao
     * de acentos, minsculas e maisculas.  a mesma configurao usada nas
     * tabelas ordenveis do JavaUtils
     */
    final Collator collator = Collator.getInstance();
    collator.setStrength(Collator.TERTIARY);
    Comparator<UserOutline> loginComparator = new Comparator<UserOutline>() {
      public int compare(UserOutline o1, UserOutline o2) {
        return collator.compare(o1.getLogin(), o2.getLogin());
      }
    };
    return loginComparator;
  }

  /**
   * Obtm um comparator de <code>UserOutline</code> pelo critrio de ordem
   * alfabtica do nome do grupo do usurio.
   * 
   * @return um comparador por nome do UserGroup
   */
  public static Comparator<UserOutline> getUserGroupComparator() {
    /*
     * usamos Collator com strength = TERTIARY para garantir a correta ordenao
     * de acentos, minsculas e maisculas.  a mesma configurao usada nas
     * tabelas ordenveis do JavaUtils
     */
    final Collator collator = Collator.getInstance();
    collator.setStrength(Collator.TERTIARY);
    Comparator<UserOutline> userGroupComparator =
      new Comparator<UserOutline>() {
        public int compare(UserOutline o1, UserOutline o2) {
          return collator.compare(o1.getUserGroup(), o2.getUserGroup());
        }
      };
    return userGroupComparator;
  }

  /**
   * Obtm um getter para o login de <code>UserOutline</code>.
   * 
   * @return um getter para o login
   */
  public static Getter getLoginGetter() {
    Getter loginGetter = new Getter() {
      public Object get(Object o) {
        UserOutline userOutline = (UserOutline) o;
        return userOutline.getLogin();
      }
    };
    return loginGetter;
  }

  /**
   * Obtm um getter para o nome do UserGroup de <code>UserOutline</code>.
   * 
   * @return um getter para o nome do UserGroup
   */
  public static Getter getUserGroupGetter() {
    Getter userGroupGetter = new Getter() {
      public Object get(Object o) {
        UserOutline userOutline = (UserOutline) o;
        return userOutline.getUserGroup();
      }
    };
    return userGroupGetter;
  }

  /**
   * Verifica se um resumo de usurio  igual a um outro. Dois resumos de
   * usurios so considerados o mesmo se eles possuirem o mesmo identificador.
   * 
   * @param obj o objeto de comparao.
   * 
   * @return true ou false, se o resumo for ou no igual a um outro.
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof UserOutline)) {
      return false;
    }
    UserOutline outline = (UserOutline) obj;
    return outline.getId().equals(userId);
  }

  /**
   * Calcula o cdigo hash do objeto.
   * 
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    return userId.hashCode();
  }

  /**
   * Cria um resumo de um usurio a partir de um objeto User.
   * 
   * @param user o usurio
   * 
   * @throws Exception se houver algum problema na recuperao do grupo.
   */
  public UserOutline(User user) throws Exception {
    userId = user.getId();
    login = user.getLogin();
    UserGroup userGroup = UserGroup.getUserGroupFromUser(user);
    emails = user.getEmails();
    name = user.getName();
    userGroupName = (userGroup == null) ? "" : userGroup.getName();
  }

  /**
   * Cria um resumo de um usurio.
   * 
   * @param userId identificador do usurio.
   * @param login login do usurio.
   * @param name nome completo do usurio.
   * @param userGroupName nome do grupo do usurio.
   * @param emails endereos de e-mail do usurio.
   */
  public UserOutline(Object userId, String login, String name,
    String userGroupName, String[] emails) {
    this.userId = userId;
    this.login = login;
    this.name = name;
    this.userGroupName = userGroupName;
    this.emails = emails;
  }
}
