/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.validators.DoubleParameterValidator;
import csbase.logic.algorithms.parameters.validators.ListParameterValidator;

/**
 * Parmetro Lista de reais.
 *
 * @author lmoreira
 */
public final class DoubleListParameter extends ListParameter<Double> {

  /**
   * A representao textual do tipo do parmetro.
   */
  public static final String TYPE = "LIST_OF_DOUBLES";

  /**
   * Indica se o valor mximo est entre os valores vlidos.
   */
  private boolean isMaximumIncluded;
  /**
   * Indica se o valor mnimo est entre os valores vlidos.
   */
  private boolean isMinimumIncluded;
  /**
   * O valor mximo.
   */
  private Double maximum;
  /**
   * O valor mnimo.
   */
  private Double minimum;

  /**
   * Os observadores.
   */
  private transient List<DoubleListParameterListener> listeners;

  /**
   * Cria um Validador.
   *
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor  opcional/obrigatrio.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param maximum O mximo (Aceita {@code null}).
   * @param includeMaximum Indica se o valor mximo est entre os valores
   *        vlidos.
   * @param minimum O mnimo (Aceita {@code null}).
   * @param includeMinimum Indica se o valor mnimo est entre os valores
   *        vlidos.
   */
  public DoubleListParameter(String name, String label, String description,
    List<Double> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, Double maximum, boolean includeMaximum,
    Double minimum, boolean includeMinimum) {
    super(name, label, description, defaultValue, isOptional, isVisible, false,
      true, commandLinePattern);
    this.listeners = new LinkedList<DoubleListParameterListener>();
    setMaximum(maximum, includeMaximum);
    setMinimum(minimum, includeMinimum);
  }

  /**
   * Adiciona um observador a este parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public void addDoubleListParameterListener(
    DoubleListParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * Obtm o mximo.
   *
   * @return O mximo ou {@code null} se ele no existir.
   */
  public Double getMaximum() {
    return this.maximum;
  }

  /**
   * Obtm o mnimo.
   *
   * @return O mnimo ou {@code null} se ele no existir.
   */
  public Double getMinimum() {
    return this.minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * Indica se o valor mximo est entre os valores vlidos.
   *
   * @return .
   */
  public boolean isMaximumIncluded() {
    return this.isMaximumIncluded;
  }

  /**
   * Indica se o valor mnimo est entre os valores vlidos.
   *
   * @return .
   */
  public boolean isMinimumIncluded() {
    return this.isMinimumIncluded;
  }

  /**
   * <p>
   * Atribui o mximo e se o mximo  includo.
   * </p>
   *
   * <p>
   * Gera o evento
   * {@link DoubleListParameterListener#minimumWasChanged(DoubleListParameter)}.
   * </p>
   *
   * @param maximum O mximo (Aceita {@code null}).
   * @param maximumIncluded Indica se o mximo deve ou no ser considerado um
   *        valor vlido.
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o mximo
   *         informado  igual ao mximo anterior.
   */
  public boolean setMaximum(Double maximum, boolean maximumIncluded) {
    if (minimum != null) {
      if ((maximum != null) && (maximum.compareTo(minimum) < 0)) {
        throw new IllegalArgumentException(
          "O valor mximo no pode ser menor do que o valor mnimo.");
      }
    }
    boolean maxIncluded = maximumIncluded;
    if (maximum == null) {
      if (this.maximum == null) {
        return false;
      }
      maxIncluded = false;
    }
    else if (maximum.equals(this.maximum)) {
      return false;
    }

    this.isMaximumIncluded = maxIncluded;
    this.maximum = maximum;
    updateValidator();
    for (DoubleListParameterListener listener : this.listeners) {
      listener.maximumWasChanged(this);
    }
    return true;
  }

  /**
   * <p>
   * Atribui o mnimo e se o minimo  includo.
   * </p>
   *
   * <p>
   * Gera o evento
   * {@link DoubleListParameterListener#minimumWasChanged(DoubleListParameter)}.
   * </p>
   *
   * @param minimum O mnimo (Aceita {@code null}).
   * @param minimumIncluded Indica se o mnimo deve ou no ser considerado um
   *        valor vlido.
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o mnimo
   *         informado  igual ao mpinimo anterior.
   */
  public boolean setMinimum(Double minimum, boolean minimumIncluded) {
    if (maximum != null) {
      if ((minimum != null) && (minimum.compareTo(maximum) > 0)) {
        throw new IllegalArgumentException(
          "O valor mnimo no pode ser maior do que o valor mximo.");
      }
    }
    boolean minIncluded = minimumIncluded;
    if (minimum == null) {
      if (this.minimum == null) {
        return false;
      }
      minIncluded = false;
    }
    else if (minimum.equals(this.minimum)) {
      return false;
    }
    this.isMinimumIncluded = minIncluded;
    this.minimum = minimum;
    updateValidator();
    for (DoubleListParameterListener listener : this.listeners) {
      listener.minimumWasChanged(this);
    }
    return true;
  }

  /**
   * Atualiza os valores utilizados para validar o parmetro.
   */
  private void updateValidator() {
    ListParameterValidator<Double> listParameterValidator =
      (ListParameterValidator<Double>) getParameterValidator();
    DoubleParameterValidator validator =
      (DoubleParameterValidator) listParameterValidator.getItemValidator();
    validator.setMaximum(this.maximum, this.isMaximumIncluded);
    validator.setMinimum(this.minimum, this.isMinimumIncluded);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Double getItemValueFromText(String itemValue) throws ParseException {
    if (itemValue == null) {
      return null;
    }
    try {
      double doubleValue = Double.parseDouble(itemValue);
      return new Double(doubleValue);
    }
    catch (NumberFormatException e) {
      throw new ParseException("O valor {0} foi atribuido ao parmetro {1}, "
        + "porm ele no  um valor vlido para um nmero real.", itemValue,
        this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected DoubleParameterValidator createItemValidator() {
    return new DoubleParameterValidator(isOptional(), maximum,
      isMaximumIncluded, minimum, isMinimumIncluded);
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in Leitor de objetos
   *
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<DoubleListParameterListener>();
  }
}
