package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import csbase.logic.algorithms.parameters.BooleanColumn;
import csbase.logic.algorithms.parameters.Column;
import csbase.logic.algorithms.parameters.DoubleColumn;
import csbase.logic.algorithms.parameters.EnumerationColumn;
import csbase.logic.algorithms.parameters.IntegerColumn;
import csbase.logic.algorithms.parameters.RowValue;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parameters.TextColumn;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * Validador do {@link csbase.logic.algorithms.parameters.TableParameter
 * Parmetro do Tipo Tabela}.
 */
public final class TableParameterValidator extends
  SimpleParameterValidator<List<RowValue>> {
  /**
   * Os validadores das colunas.
   */
  private List<SimpleParameterValidator<?>> columnValidators;

  /**
   * ndice da coluna na tabela.
   */
  private int columnIndex;
  /**
   * ndice da linha na tabela.
   */
  private int rowIndex;

  /**
   * Cria um validador de tabela.
   * 
   * @param optional Indica se o valor  opcional/obrigatrio.
   * @param columns Lista de colunas da tabela
   */
  public TableParameterValidator(boolean optional, List<Column<?>> columns) {
    super(optional);
    this.columnValidators =
      new ArrayList<SimpleParameterValidator<?>>(columns.size());
    for (Column<?> column : columns) {
      if (column instanceof BooleanColumn) {
        this.columnValidators.add(new BooleanParameterValidator());
      }
      else if (column instanceof DoubleColumn) {
        DoubleColumn doubleColumn = (DoubleColumn) column;
        this.columnValidators.add(new DoubleParameterValidator(doubleColumn
          .isOptional(), doubleColumn.getMaximum(), doubleColumn
          .isMaximumIncluded(), doubleColumn.getMinimum(), doubleColumn
          .isMinimumIncluded()));
      }
      else if (column instanceof IntegerColumn) {
        IntegerColumn integerColumn = (IntegerColumn) column;
        this.columnValidators
          .add(new IntegerParameterValidator(integerColumn.isOptional(),
            integerColumn.getMaximum(), integerColumn.getMinimum()));
      }
      else if (column instanceof TextColumn) {
        TextColumn textColumn = (TextColumn) column;
        this.columnValidators.add(new TextParameterValidator(textColumn
          .isOptional(), textColumn.getMaxCharacters()));
      }
      else if (column instanceof EnumerationColumn) {
        this.columnValidators.add(new EnumerationParameterValidator(column
          .isOptional()));
      }
      else {
        throw new IllegalStateException(String.format(
          "Tipo inesperado para a coluna %s.\nTipo encontrado: %s.",
          column.getId(), column.getClass()));
      }
    }
  }

  /**
   * @see csbase.logic.algorithms.parameters.validators.SimpleParameterValidator#setMode(csbase.logic.algorithms.validation.ValidationMode)
   */
  @Override
  public void setMode(ValidationMode mode) {
    for (SimpleParameterValidator<?> validator : this.columnValidators) {
      validator.setMode(mode);
    }
    super.setMode(mode);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public Validation validateValue(SimpleParameter<?> parameter,
    List<RowValue> value, ValidationContext context) throws RemoteException {
    if (!parameter.isEnabled() || !parameter.isVisible()) {
      return new ValidationSuccess();
    }
    TableParameter tableParameter = (TableParameter) parameter;
    if (tableParameter.getRowCount() == 0) {
      if (isOptional() || getMode().equals(ValidationMode.ALLOW_EMPY_VALUES)) {
        return new ValidationSuccess();
      }
      LocalizedMessage message =
        new LocalizedMessage(TableParameterValidator.class, "empty_value",
          new Object[] { parameter.getLabel() });
      return new ValidationError(message);
    }
    for (int i = 0; i < tableParameter.getRowCount(); i++) {
      this.rowIndex = i;
      List<Column<?>> columns = tableParameter.getColumns();
      for (int j = 0; j < columns.size(); j++) {
        this.columnIndex = j;
        Object itemValue = tableParameter.getItemValue(i, j);
        @SuppressWarnings("rawtypes")
        SimpleParameterValidator validator = this.columnValidators.get(j);
        Validation result =
          validator.validateValue(tableParameter, itemValue, context);
        if (!result.isWellSucceded()) {
          return result;
        }
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Retorna o ndice da linha na tabela.
   * 
   * @return O ndice da linha.
   */
  public int getRowIndex() {
    return rowIndex;
  }

  /**
   * Retorna o ndice da coluna na tabela.
   * 
   * @return O ndice da coluna.
   */
  public int getColumnIndex() {
    return columnIndex;
  }
}