package csbase.logic.algorithms.serializer;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import csbase.exception.ConfigurationException;
import csbase.exception.ParseException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowAlgorithmParser;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerException;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerIOException;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerParseException;

/**
 * Objeto utilizado para (des)serializar as configuraces de um configurador de
 * fluxo de algoritmos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public final class FlowAlgorithmConfigurationSerializer implements
  IAlgorithmConfigurationSerializer {

  /**
   * Desserializa o fluxo.
   * 
   * @param input fluxo de dados com a configuraco de um
   *        FlowAlgorithmConfigurator serializada (No aceita {@code null}).
   * 
   * @return Um configurador de fluxo de algortmos com seus parmetros
   *         preenchidos.
   * 
   * @throws AlgorithmConfigurationSerializerParseException Quando os dados do
   *         fluxo no estiverem formatados corretamente.
   * @throws AlgorithmConfigurationSerializerIOException Quando ocorrer um erro
   *         de IO no fluxo.
   * @throws AlgorithmNotFoundException Quando a verso do algoritmo utilizado
   *         para criar este configurador no existe mais.
   * @throws AlgorithmConfigurationSerializerException Demais erros que possam
   *         ocorrer na implementaco deste mtodo.
   */
  @Override
  public AlgorithmConfigurator read(InputStream input)
    throws AlgorithmConfigurationSerializerParseException,
    AlgorithmConfigurationSerializerIOException,
    AlgorithmConfigurationSerializerException, AlgorithmNotFoundException {

    try {
      Flow flow = new FlowAlgorithmParser().read(input);
      return new FlowAlgorithmConfigurator(flow);
    }
    catch (ConfigurationException e) {
      throw new AlgorithmConfigurationSerializerException(e);
    }
    catch (ParseException e) {
      throw new AlgorithmConfigurationSerializerParseException(e);
    }
    catch (IOException e) {
      throw new AlgorithmConfigurationSerializerIOException(e);
    }
  }

  /**
   * Serializa o fluxo.
   * 
   * @param configurator Um configurador de um fluxo de algoritmos (No aceita
   *        {@code null}).
   * @param output Fluxo de dados para o qual ser serializada a configuraco do
   *        configurador de algortmos (No aceita {@code null}).
   * 
   * @throws AlgorithmConfigurationSerializerException Quando ocorrer um erro ao
   *         serializar o fluxo.
   */
  @Override
  public void write(AlgorithmConfigurator configurator, OutputStream output)
    throws AlgorithmConfigurationSerializerException {

    if (configurator.getConfiguratorType() != ConfiguratorType.FLOW) {
      throw new IllegalArgumentException(
        "O configurador deve ser do tipo FlowAlgorithmConfigurator");
    }

    FlowAlgorithmConfigurator flowConfigurator =
      (FlowAlgorithmConfigurator) configurator;
    try {
      new FlowAlgorithmParser().write(output, flowConfigurator.getFlow());
    }
    catch (Exception e) {
      throw new AlgorithmConfigurationSerializerException(e);
    }
  }
}
