package csbase.console;

import java.io.File;
import java.io.InputStream;
import java.rmi.RemoteException;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;

import csbase.client.externalresources.StandaloneLocalFile;
import csbase.exception.ServiceFailureException;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.AlgorithmsPack;
import csbase.logic.algorithms.ImportAlgorithmsPackTransferInfo;
import csbase.logic.algorithms.PAImportOperation;
import csbase.remote.ClientRemoteLocator;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

/**
 * Cliente sem interface grfica, para realizar a importao de um Pacote de
 * Algoritmos (PA). Esse cliente deve ser executado diretamente da linha de
 * comando.
 *
 */
class ImportAlgoPack extends AbstractConsoleApp {
  /**
   * Lista de algoritmos retornados do PA lido no servidor.
   */
  private List<AlgorithmInfo> algorithmsList;

  /**
   * Construtor do cliente para importar um PA. Obtm a senha interativamente.
   *
   * @param args parmetros fornecidos na linha de comando
   */
  ImportAlgoPack(String[] args) {
    super(args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected BasicParams createParams() {
    return new AlgorithmsPackParams();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void postLoginInit() {
  }

  /**
   * Efetua a transferncia do arquivo local atravs do canal remoto.
   *
   * @param file referncia para o arquivo local (aplicao standalone)
   * @param info informaes do canal remoto de transferncia (FTC)
   * @return true se a transferncia ocorreu com sucesso, caso contrrio,
   *         retorna false
   *
   * @throws Exception Em caso de falha na operao.
   */
  private static boolean transfer(StandaloneLocalFile file,
    RemoteFileChannelInfo info) throws Exception {
    SyncRemoteFileChannel channel =
      new SyncRemoteFileChannel(info.getIdentifier(), info.isWritable(),
        info.getHost(), info.getPort(), info.getKey());
    channel.open(!info.isWritable());
    InputStream inputStream = file.getInputStream();
    long totalBytes =
      channel.syncTransferFrom(inputStream, 0, file.getLength());
    inputStream.close();
    channel.close();
    return (totalBytes == file.getLength());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean preLogout() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getLogin() {
    AlgorithmsPackParams params = (AlgorithmsPackParams) getParams();
    String login = (String) (params.userLogin == null ? User.getAdminId()
      : params.userLogin);
    return login;
  }

  /**
   * Obtm o arquivo local que ser copiado para o diretrio remoto.
   *
   * @param localSrcPath path para o arquivo local
   * @return o arquivo local, ou <code>null</code> caso este no exista ou seja
   *         um diretrio
   */
  private File getLocalSrcFile(String localSrcPath) {
    File localFile;
    if (localSrcPath.charAt(0) != '/') {
      localFile = new File(getCurrentDir(), localSrcPath);
    }
    else {
      localFile = new File(localSrcPath);
    }
    if (!localFile.exists()) {
      printError(localSrcPath + " no existe");
      return null;
    }
    if (localFile.isDirectory()) {
      printError(localSrcPath + "  um diretrio");
      return null;
    }
    return localFile;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExitCode execute() throws Exception {
    AlgorithmsPackParams params = (AlgorithmsPackParams) getParams();

    if (params.startPA != null) {
      String localPAPath = params.startPA;
      return startPAImport(localPAPath);
    }

    if (params.finishPA) {
      if (params.token == null) {
        printError(
          "token do dado do Pacote de Algoritmos no foi especificado");
        return ExitCode.FAILURE;
      }
      return finishPAImport(params.token);
    }

    if (params.validatePA) {
      if (params.token == null) {
        printError(
          "token do dado do Pacote de Algoritmos no foi especificado");
        return ExitCode.FAILURE;
      }
      return validatePA(params.token);
    }

    if (params.listAlgorithms) {
      if (params.token == null) {
        printError(
          "token do dado do Pacote de Algoritmos no foi especificado");
        return ExitCode.FAILURE;
      }
      listAlgorithms(params.token);
      return ExitCode.SUCCESS;
    }

    if (params.algoToList != null) {
      if (params.token == null) {
        printError(
          "token do dado do Pacote de Algoritmos no foi especificado");
        return ExitCode.FAILURE;
      }
      return listAlgoInfo(params.token, params.algoToList);
    }

    if (!validateAlgorithmsOperations(params)) {
      return ExitCode.FAILURE;
    }

    if (params.replaceAlgorithms) {
      if (params.token == null) {
        printError(
          "token do dado do Pacote de Algoritmos no foi especificado");
        return ExitCode.FAILURE;
      }
      return replaceAlgorithms(params.token);
    }

    if (params.keepAlgorithms) {
      if (params.token == null) {
        printError(
          "token do dado do Pacote de Algoritmos no foi especificado");
        return ExitCode.FAILURE;
      }
      return keepAlgorithms(params.token);
    }

    if (params.mergeAlgorithms) {
      if (params.token == null) {
        printError(
          "token do dado do Pacote de Algoritmos no foi especificado");
        return ExitCode.FAILURE;
      }

      return mergeAlgorithms(params);
    }
    // Se nenhuma operao foi especificada nem realizada, retorna sucesso.
    return ExitCode.SUCCESS;
  }

  /**
   * Valida as combinaes possveis de operaes sobre algoritmos, categorias e
   * verses.
   *
   * @param params parmetros do programa, passados por linha de comando
   *
   * @return retorna true, se as operaes passadas so vlidas, caso contrrio,
   *         retorna false
   */
  private boolean validateAlgorithmsOperations(AlgorithmsPackParams params) {
    boolean validate = true;

    //Valida se h uma e somente uma operao sobre os algoritmos, passada como parmetro
    int algorithmsOpsNumber = 0;
    algorithmsOpsNumber += params.replaceAlgorithms ? 1 : 0;
    algorithmsOpsNumber += params.mergeAlgorithms ? 1 : 0;
    algorithmsOpsNumber += params.keepAlgorithms ? 1 : 0;
    if (algorithmsOpsNumber == 0) {
      validate = false;
      printError(
        "a importao de PA requer a escolha de pelo menos uma operao a ser realizada sobre os algoritmos do PA");
    }
    else if (algorithmsOpsNumber > 1) {
      validate = false;
      printError(
        "escolha somente uma operao para ser realizada sobre os algoritmos do PA");
    }
    else if (params.replaceAlgorithms || params.keepAlgorithms) {
      return true;
    }

    //Valida se h uma e somente uma operao sobre categorias, passada como parmetro
    int categoriesOpsNumber = 0;
    categoriesOpsNumber += params.replaceCategories ? 1 : 0;
    categoriesOpsNumber += params.mergeCategories ? 1 : 0;
    categoriesOpsNumber += params.keepCategories ? 1 : 0;
    if (categoriesOpsNumber == 0) {
      validate = false;
      printError(
        "o merge de algoritmos requer a escolha de uma operao sobre as categorias dos algoritmos do Pacote de Algoritmos");
    }
    else if (categoriesOpsNumber > 1) {
      validate = false;
      printError(
        "escolha somente uma operao para ser realizada sobre as categorias dos algoritmos do PA");
    }

    //Valida se h uma e somente uma operao sobre verses, passada como parmetro
    int versionOpsNumber = 0;
    versionOpsNumber += params.replaceVersions ? 1 : 0;
    versionOpsNumber += params.keepVersions ? 1 : 0;
    if (versionOpsNumber == 0) {
      validate = false;
      printError(
        "o merge de algoritmos requer a escolha de uma operao sobre as verses dos algoritmos do Pacote de Algoritmos");
    }
    else if (versionOpsNumber > 1) {
      validate = false;
      printError(
        "escolha somente uma operao para ser realizada sobre as verses dos algoritmos do PA");
    }
    boolean hasMergeOperations =
      categoriesOpsNumber > 0 || versionOpsNumber > 0;
    if (validate && !params.mergeAlgorithms && hasMergeOperations) {
      validate = false;
      printError(
        "somente a operao de merge de algoritmos requer a escolha de uma operao sobre verses e categorias dos algoritmos do PA");
    }

    return validate;
  }

  /**
   * Valida um pacote de algoritmos, sem realizar a importao.
   *
   * @param token identificador nico do dado do PA
   * @return Cdigo de sada.
   *
   * @throws Exception
   */
  private static ExitCode validatePA(String token) throws Exception {
    String result = "";
    try {
      result =
        ClientRemoteLocator.algorithmService.validateAlgorithmsPack(token);
    }
    catch (ServiceFailureException e) {
      System.err.println(e.getMessage());
      return ExitCode.FAILURE;
    }
    System.out.println(
      "\nResultado da validao do PA com token <" + token + ">:\n" + result);
    return ExitCode.SUCCESS;
  }

  /**
   * finalizar o processo de manipulao do PA para importao, removendo o
   * diretrio temporrio criado para o token especificado.
   *
   * @param token identificador nico do dado do PA
   * @return Cdigo de sada.
   * @throws Exception
   */
  private static ExitCode finishPAImport(String token) throws Exception {
    try {
      ClientRemoteLocator.algorithmService.finishImportAlgorithmsPack(token);
    }
    catch (ServiceFailureException e) {
      System.err.println(e.getMessage());
      return ExitCode.FAILURE;
    }
    System.out.println("Processo de importao do PA com token <" + token
      + "> est finalizado.");
    return ExitCode.SUCCESS;
  }

  /**
   * Inicia o processo de manipulao do PA para importao. Esse comando
   * realiza a transferncia do PA para o servidor e retorna um token para o
   * dado do PA que requisitou essa iniciao. Esse token dever ser passado
   * para todas as demais operaes sobre esse PA.
   *
   * @param localPAPath caminho do arquivo local do PA
   * @return Cdigo de sada.
   * @throws Exception
   */
  private ExitCode startPAImport(String localPAPath) throws Exception {
    ImportAlgorithmsPackTransferInfo importAlgoPackInfo =
      ClientRemoteLocator.algorithmService.prepareImportAlgorithmsPack();
    RemoteFileChannelInfo requestInfo = importAlgoPackInfo.getChannel();

    File localPAFile = getLocalSrcFile(localPAPath);
    if (localPAFile == null) {
      System.err.println("Path inexistente para Pacote de algoritmo.");
      return ExitCode.FAILURE;
    }
    StandaloneLocalFile standAloneFile = new StandaloneLocalFile(localPAFile);

    if (!transfer(standAloneFile, requestInfo)) {
      return ExitCode.FAILURE;
    }
    String token = importAlgoPackInfo.getImportDataToken();
    System.out.println("Token correspondente ao PA: ");
    System.out.println(token);
    return ExitCode.SUCCESS;
  }

  /**
   * Obtm as informaes de um algoritmo a partir do nome especificado.
   *
   * @param token identificador nico do dado do PA
   * @param algoName nome do algoritmo
   * @return as informaes do algoritmo
   */
  private AlgorithmInfo getAlgoInfo(String token, String algoName) {
    List<AlgorithmInfo> algorithms = getAlgorithmsFromPack(token);
    if (algorithms != null) {
      for (AlgorithmInfo algorithmInfo : algorithms) {
        if (algorithmInfo.getName().equals(algoName)) {
          return algorithmInfo;
        }
      }
    }
    return null;
  }

  /**
   * Lista os algoritmos de um pacote de algoritmos especificado pelo token.
   *
   * @param token identificador nico do dado do PA
   */
  private void listAlgorithms(String token) {
    List<AlgorithmInfo> algorithms = getAlgorithmsFromPack(token);
    if (algorithms != null) {
      for (AlgorithmInfo algorithmInfo : algorithms) {
        printAlgoData(algorithmInfo);
      }
    }
  }

  /**
   * Lista as informaes detalhadas de um algoritmo, a partir do seu nome.
   *
   * @param token identificador nico do dado do PA
   * @param algoName nome do algoritmo procurado
   * @return Cdigo de sada.
   */
  private ExitCode listAlgoInfo(String token, String algoName) {
    AlgorithmInfo algoInfo = getAlgoInfo(token, algoName);
    if (algoInfo != null) {
      printAlgoInfo(algoInfo);
      return ExitCode.SUCCESS;
    }
    System.err.println("Algoritmo \"" + algoName + "\" no encontrado.");
    return ExitCode.FAILURE;
  }

  /**
   * Imprime os dados principais do algoritmo lido do metadados do Pacote de
   * Algoritmos.
   *
   * @param algo algoritmo
   */
  private static void printAlgoData(AlgorithmInfo algo) {
    System.out.println("\nDados do algoritmo criado a partir do xml: ");
    System.out.println("Id: " + algo.getId());
    System.out.println("Nome: " + algo.getName());
    System.out.println("Descrio: " + algo.getDescription());
  }

  /**
   * Imprime as informaes de um algoritmo criado a partir da leitura do xml
   * com os metadados do Pacote de Algoritmos.
   *
   * @param algo algoritmo
   */
  private static void printAlgoInfo(AlgorithmInfo algo) {
    printAlgoData(algo);

    //Propriedades do algoritmo
    Hashtable<String, String> propertyValues = algo.getPropertyValues();
    Set<Entry<String, String>> entrySet = propertyValues.entrySet();
    System.out.println("\n>> Propriedades do algoritmo: " + entrySet.size());
    for (Entry<String, String> entry : entrySet) {
      System.out.println(entry.getKey() + " = " + entry.getValue());
    }

    //Verses do algoritmo
    Vector<AlgorithmVersionInfo> versions = algo.getVersions();
    System.out.println("\n>> Verses do algoritmo: " + versions.size());
    for (AlgorithmVersionInfo algoVersionInfo : versions) {
      System.out.println("\n-> Verso: " + algoVersionInfo);
      System.out.println("descrio: " + algoVersionInfo.getDescription());
      Map<String, String> properties = algoVersionInfo.getPropertyValues();
      entrySet.clear();
      entrySet = properties.entrySet();
      System.out
        .println("\nPropriedades da verso do algoritmo: " + entrySet.size());
      for (Entry<String, String> entry : entrySet) {
        System.out.println(entry.getKey() + " = " + entry.getValue());
      }
    }

    //Categorias do algoritmo
    List<String> categoryFullNames = algo.getAlgoPackCategoryFullNames();
    System.out
      .println("\n>> Categorias do algoritmo: " + categoryFullNames.size());
    for (String categoryFullName : categoryFullNames) {
      System.out.println("\n-> Nome completo: " + categoryFullName);
    }
  }

  /**
   * Obtm a lista de algoritmos que fazem parte do pacote de algoritmos lido
   * pelo servidor.
   *
   * @param token
   *
   * @return a lista de algoritmos presentes no PA
   */
  private List<AlgorithmInfo> getAlgorithmsFromPack(String token) {
    AlgorithmsPack algorithmPack = null;
    if (algorithmsList == null) {
      try {
        algorithmPack =
          ClientRemoteLocator.algorithmService.getAlgorithmsPackInfo(token);
      }
      catch (ServiceFailureException e) {
        System.err.println(e.getMessage());
        return null;
      }
      catch (RemoteException e) {
        System.err.println(e.getMessage());
        return null;
      }
      algorithmsList = algorithmPack.getAlgorithms();
    }
    return algorithmsList;
  }

  /**
   * Substitui todo o contudo de um algoritmo lido do PA no repositrio. Ou
   * seja, se j existir o algoritmo com aquele identificador no repositrio, o
   * mesmo  removido e  substitudo pelo novo algoritmo lido.
   *
   * @param token identificador nico do dado do PA
   * @return Cdigo de sada.
   * @throws Exception
   */
  private static ExitCode replaceAlgorithms(String token) throws Exception {
    try {
      PAImportOperation[] operations = new PAImportOperation[1];
      operations[0] = PAImportOperation.REPLACE_ALGORITHMS;

      if (!ClientRemoteLocator.algorithmService.importAlgorithmsPack(token,
        operations)) {
        System.err.println(String.format(
          "Ocorreu uma falha na importao dos algoritmos do PA com token %s.\nExecute o comando para validar o Pacote de Algoritmos.",
          token));
        return ExitCode.FAILURE;
      }
      System.out.println(
        "\nA importao de algoritmos ocorreu com sucesso.\nTodos os algoritmos com mesmo identificador no PA, foram substitudos no repositrio de algoritmos.");
      return ExitCode.SUCCESS;
    }
    catch (ServiceFailureException e) {
      System.err.println(e.getMessage());
      return ExitCode.FAILURE;
    }
  }

  /**
   * Mantm todo o contudo de um algoritmo lido do PA no repositrio. Ou seja,
   * se j existir o algoritmo com aquele identificador no repositrio, o mesmo
   *  mantido e o do PA no  criado no repositrio.
   *
   * @param token identificador nico do dado do PA
   * @return Cdigo de sada.
   * @throws Exception
   */
  private static ExitCode keepAlgorithms(String token) throws Exception {
    try {
      PAImportOperation[] operations = new PAImportOperation[1];
      operations[0] = PAImportOperation.KEEP_ALGORITHMS;

      if (!ClientRemoteLocator.algorithmService.importAlgorithmsPack(token,
        operations)) {
        System.err.println(String.format(
          "Ocorreu uma falha na importao dos algoritmos do PA com token %s.\nExecute o comando para validar o Pacote de Algoritmos.",
          token));
        return ExitCode.FAILURE;
      }
      System.out.println(
        "\nA importao de algoritmos ocorreu com sucesso.\nTodos os algoritmos com mesmo identificador no PA, foram mantidos no repositrio de algoritmos.");
      return ExitCode.SUCCESS;
    }
    catch (ServiceFailureException e) {
      System.err.println(e.getMessage());
      return ExitCode.FAILURE;
    }
  }

  /**
   * @param params
   * @return Cdigo de sada.
   * @throws RemoteException
   */
  private static ExitCode mergeAlgorithms(AlgorithmsPackParams params)
    throws RemoteException {
    String token = params.token;

    try {
      PAImportOperation[] operations = new PAImportOperation[3];
      operations[0] = PAImportOperation.MERGE_ALGORITHMS;
      operations[1] = getVersionsOperation(params);
      operations[2] = getCategoriesOperation(params);

      if (!ClientRemoteLocator.algorithmService.importAlgorithmsPack(token,
        operations)) {
        System.err.println(String.format(
          "Ocorreu uma falha na importao dos algoritmos do PA com token %s.\nExecute o comando para validar o Pacote de Algoritmos.",
          token));
        return ExitCode.FAILURE;
      }
      System.out.println(
        "\nA importao de algoritmos ocorreu com sucesso.\nTodos os algoritmos com mesmo identificador no PA, foram combinados no repositrio de algoritmos, conforme operaes definidas no comando.");
      return ExitCode.SUCCESS;
    }
    catch (ServiceFailureException e) {
      System.err.println(e.getMessage());
      return ExitCode.FAILURE;
    }
  }

  /**
   * Obtm a operao a ser realizada sobre as categorias do algoritmo, quando a
   * categoria do algoritmo do PA j existir no repositrio.
   *
   * @param params parmetros passados na linha de comando
   *
   * @return retorna o operao de importao correspondente ao parmetro
   *         passado para operao de categorias
   */
  private static PAImportOperation getCategoriesOperation(
    AlgorithmsPackParams params) {
    if (params.replaceCategories) {
      return PAImportOperation.REPLACE_CATEGORIES;
    }
    else if (params.mergeCategories) {
      return PAImportOperation.MERGE_CATEGORIES;
    }
    return PAImportOperation.KEEP_CATEGORIES;
  }

  /**
   * Obtm a operao a ser realizada sobre as verses do algoritmo, quando a
   * verso do algoritmo do PA j existir no repositrio.
   *
   * @param params parmetros passados na linha de comando
   *
   * @return retorna o operao de importao correspondente ao parmetro
   *         passado para operao de verses
   */
  private static PAImportOperation getVersionsOperation(
    AlgorithmsPackParams params) {
    if (params.replaceVersions) {
      return PAImportOperation.REPLACE_VERSIONS;
    }
    return PAImportOperation.KEEP_VERSIONS;
  }

}