package csbase.login;

import java.io.FileInputStream;
import java.io.UnsupportedEncodingException;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import tecgraf.javautils.core.properties.PropertiesUtils;
import tecgraf.javautils.core.properties.PropertyException;

import com.novell.ldap.LDAPConnection;
import com.novell.ldap.LDAPConstraints;
import com.novell.ldap.LDAPException;
import com.novell.ldap.LDAPJSSESecureSocketFactory;

import csbase.logic.User;
import csbase.server.plugin.service.loginservice.ILoginService;

/**
 * Protocolo LDAP para login do CSBase
 *
 * @author Tecgraf
 */
public class LDAPLoginProtocol implements ILoginService {
  Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

  /** Porta padro para acesso a servidores LDAP. */
  public static final int LDAP_DEFAULT_PORT = 389;
  /** Marca a ser substituda no padro LDAP pelo usurio. */
  public static final String LDAP_PATTERN_USER = "[%U]";

  /** Lista dos servidores LDAP disponveis para autenticao. */
  private List<String> LDAPServers;

  /**
   * Porta do servio LDAP em cada servidor disponvel para autenticao.
   */
  private Vector<Integer> LDAPPorts;

  /** Lista de padres disponveis para autenticao dos usurios. */
  private List<String> LDAPPatternList;

  /** CharSet utilizado na converso da senha em um byte[]. */
  private String LDAPCharSet;

  /** Define se o LDAP com SSL estar habilitado */
  private Boolean sslEnabled;
  /** Caminho do keyStore do LDAP */
  private String keyStorePath;
  /** Senha do keyStore (se houver) */
  private String keyStorePassword;
  /** Timeout para a conexo com o servidor. */  
  private int LDAPTimeout;

  /**
   * {@inheritDoc}
   */
  @Override
  public LoginProtocol getProtocolType() { 
    return LoginProtocol.PROTOCOL_LDAP;
  }

  @Override
  public void setProperties(Properties properties) {
    try {
      sslEnabled = PropertiesUtils.getBoolenValue(properties, "LDAPSSLEnabled");

      if (sslEnabled) {
        keyStorePath = PropertiesUtils.getValue(properties, "LDAPKeyStorePath");
        keyStorePassword = PropertiesUtils.getValue(properties, "LDAPKeyStorePassword");
      }

      LDAPServers = PropertiesUtils.getValuesList(properties, "LDAPServer");
      int size = LDAPServers.size();
      if (size == 0) {
              throw new PropertyException(
                  "Nenhum servidor LDAP definido para autenticao!");
      }
      LDAPPorts = new Vector<Integer>();
      for (int i = 1; i <= LDAPServers.size(); i++) {
        LDAPPorts.add(PropertiesUtils.getIntValue(properties, "LDAPPort." + i));
      }

      this.LDAPPatternList = PropertiesUtils.getValuesList(properties, "LDAPPattern");
      LDAPCharSet = PropertiesUtils.getValue(properties, "LDAPCharSet");
      LDAPTimeout = PropertiesUtils.getIntValue(properties, "LDAPConnectionTimeout");
    } catch (PropertyException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  @Override
  public User authenticate(User user, String password) {
    byte[] passwd = null;
    try {
      passwd = password.getBytes(LDAPCharSet);
    }
    catch (UnsupportedEncodingException e) {
      logger.log(Level.SEVERE, String.format("LDAPCharSet no suportado: %s.",
        LDAPCharSet), e);
      return null;
    }
    List<String> distinguishedNameList =
      new ArrayList<String>(LDAPPatternList.size());
    for (String pattern : LDAPPatternList) {
      pattern = pattern.replace(LDAP_PATTERN_USER, user.getLogin());
      distinguishedNameList.add(pattern);
    }
    for (int i = 0; i < LDAPServers.size(); i++) {
      String server = LDAPServers.get(i);
      int port = (LDAPPorts.get(i)).intValue();
      LDAPConnection conn = null;
      if (sslEnabled) {
        LDAPJSSESecureSocketFactory ssf = null;
        try {
          SSLContext context = SSLContext.getInstance("TLS", "SunJSSE");
          FileInputStream keyStoreIStream = null;
          char[] keyStorePass = null;
          if (keyStorePassword != null && !keyStorePassword.equals("")) {
            keyStorePass = keyStorePassword.toCharArray();
          }
          //Abre um stream para ler o certificado
          keyStoreIStream = new FileInputStream(keyStorePath);

          KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
          //Inicializa o keystore com os dados lidos
          //se o stream for nulo o caminho o keystore nao existe no caminho
          keyStore.load(keyStoreIStream, keyStorePass);

          TrustManager[] tms = { new LDAPTrustManager(keyStore) };
          context.init(null, tms, null);
          ssf = new LDAPJSSESecureSocketFactory(context.getSocketFactory());
        }
        catch (Exception e) {
          logger.info(String.format(
            "Certificado LDAP no encontrado no keystore padro %s:%d [%s].",
            server, port, e.getMessage()));
          continue;
        }
        conn = new LDAPConnection(ssf);
      }
      else {
        // multiplicamos o timeout por 1000 pois este  definido em segundos
        conn = new LDAPConnection(LDAPTimeout * 1000);
      }
      /*
       * definimos o timeLimit nas restries para o mesmo timeout. O valor
       * default neste caso  0, que significa "ilimitado"
       */
      final LDAPConstraints constraints = conn.getConstraints();
      constraints.setTimeLimit(LDAPTimeout * 1000);
      conn.setConstraints(constraints);
      try {
        conn.connect(server, port);
      }
      catch (LDAPException e) {
        logger.info(String.format(
          "Conexo rejeitada pelo servidor %s:%d [%s].", server, port, e
            .resultCodeToString()));
        continue;
      }
      try {
        for (String distinguishedName : distinguishedNameList) {
          try {
            conn.bind(LDAPConnection.LDAP_V3, distinguishedName, passwd);
            if (conn.getAuthenticationDN() != null) {
              return user;
            }
          }
          catch (LDAPException e) {
            logger.log(Level.SEVERE, String.format(
              "%s rejeitado pelo servidor %s [%s].", distinguishedName, server,
              e.resultCodeToString()), e);
          }
        }
      }
      finally {
        try {
          conn.disconnect();
        }
        catch (LDAPException e) {
          logger.info(String.format(
            "Erro ao desconectar do servidor %s [%s]", server, e
              .resultCodeToString()));
        }
      }
    }
    return null;
  }
  
  /**
   * Classe interna responsvel por fazer a validao do LDAP com SSL
   *
   * @author Tecgraf/PUC-Rio
   */
  private class LDAPTrustManager implements X509TrustManager {

    /**
     * A keystore com a chave para validao SSL.
     */
    private KeyStore keyStore;

    /**
     * Construtor.
     *
     * @param keyStore A keystore com a chave para validao SSL
     */
    public LDAPTrustManager(KeyStore keyStore) {
      this.keyStore = keyStore;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkClientTrusted(X509Certificate[] chain, String arg1)
      throws CertificateException {
      boolean trusted = isChainTrusted(chain);
      if (!trusted) {
        throw new CertificateException("Cliente no  confivel");
      }

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkServerTrusted(X509Certificate[] chain, String arg1)
      throws CertificateException {
      boolean trusted = isChainTrusted(chain);
      if (!trusted) {
        throw new CertificateException("Servidor no  confivel");
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public X509Certificate[] getAcceptedIssuers() {
      X509Certificate[] X509Certs = null;
      try {
        // See how many certificates are in the keystore.
        int numberOfEntry = keyStore.size();
        // If there are any certificates in the keystore.
        if (numberOfEntry > 0) {
          // Create an array of X509Certificates
          X509Certs = new X509Certificate[numberOfEntry];
          // Get all of the certificate alias out of the keystore.
          Enumeration<String> aliases = keyStore.aliases();
          // Retrieve all of the certificates out of the keystore
          // via the alias name.
          int i = 0;
          while (aliases.hasMoreElements()) {
            X509Certs[i] =
              (X509Certificate) keyStore.getCertificate(aliases.nextElement());
            i++;
          }
        }
      }
      catch (Exception e) {
        logger.info(String.format(
          "Erro ao listar certificados do keystore [%s].", e.getMessage()));
        X509Certs = null;
      }
      return X509Certs;
    }

    private boolean isChainTrusted(X509Certificate[] chain) {
      boolean trusted = false;
      try {
        // Start with the root and see if it is in the Keystore.
        // The root is at the end of the chain.
        for (int i = chain.length - 1; i >= 0; i--) {
          if (keyStore.getCertificateAlias(chain[i]) != null) {
            trusted = true;
            break;
          }
        }
      }
      catch (Exception e) {
        logger.info(String.format("Keystore invlida [%s].", e
          .getMessage()));
        trusted = false;
      }
      return trusted;
    }
  }
  
}

