package csbase.rest.adapter.job.v1;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public class LocalCache <K, V> extends ConcurrentHashMap<K, V> {

  /**
   * Mapa que guarda o tempo desde que foi adicionado
   */
  private Map<K, Long> timeMap = new ConcurrentHashMap<K, Long>();

  /**
   * Tempo de expirao em milisegundos
   */
  private long expiryInMillis;

  /**
   * @param expiryInMillis Tempo de expirao em milisegundos.
   */
  public LocalCache(long expiryInMillis) {
    this.expiryInMillis = expiryInMillis;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public V put(K key, V value) {
    timeMap.put(key, System.currentTimeMillis());
    V returnVal = super.put(key, value);
    cleanMap();
    return returnVal;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void putAll(Map<? extends K, ? extends V> m) {
    for (K key : m.keySet()) {
      put(key, m.get(key));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public V putIfAbsent(K key, V value) {
    if (!containsKey(key)) {
      return put(key, value);
    }
    else {
      return get(key);
    }
  }

  /**
   * Retorna todos os valores em cache depois de determinada data
   *
   * @param after a data para filtrar os valores.
   * @return valores cacheados depois de determinadad data
   */
  public Collection<V> valuesAfter(long after) {

    Collection<V> values = new ArrayList<V>();

    for (K key : timeMap.keySet()) {
      if (after <= (timeMap.get(key))) {
        values.add(get(key));
      }
    }

    return values;
  }

  /**
   * Limpa o mapa.
   */
  private void cleanMap() {
    long currentTime = System.currentTimeMillis();
    for (K key : timeMap.keySet()) {
      if (currentTime > (timeMap.get(key) + expiryInMillis)) {
        remove(key);
        timeMap.remove(key);
      }
    }
  }

}
