package csbase.rest.adapter.job.v1;

import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.ProgressData;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.CommandPersistenceServiceInterface;
import csbase.server.plugin.service.commandpersistenceservice.ICommandInfo;
import csbase.server.plugin.service.commandpersistenceservice.ICommandStatusListener;
import ibase.common.ServiceUtil;
import ibase.rest.api.job.v1.adapter.JobDAO;
import ibase.rest.api.job.v1.adapter.JobMonitorListener;
import ibase.rest.model.job.v1.Job;
import ibase.rest.model.job.v1.MonitoredFile;
import ibase.rest.model.job.v1.StatusChangeHistory;
import ibase.rest.model.job.v1.StatusType;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

/**
 * Created by mjulia on 24/10/16.
 */
public class JobStateListener implements ICommandStatusListener {
  private ExecutorService executor;
  private JobDAO jobDAO;
  private Collection<JobMonitorListener> jobsListener =
    Collections.synchronizedList(new ArrayList<JobMonitorListener>());

  public JobStateListener(JobDAO jobDAO) {
    this.executor = Executors.newSingleThreadExecutor();
    this.jobDAO = jobDAO;
  }

  /**
   * Implementa uma ao para tratar a mudana de estado de um comando.`
   *
   * @param commandInfo informaes sobre o comando que mudou de estado
   */
  @Override
  public void statusChanged(final ICommandInfo commandInfo) {
    executor.submit(() -> {
      Job job = null;
      String commandId = commandInfo.getCommandId();
      if (jobDAO.containsJob(commandId)) {
        job = jobDAO.findJobById(commandId);
        jobDAO.updateJob(updateJob(job, commandInfo));
      } else {
        job = createJob(commandInfo);
        jobDAO.insertJob(job);
      }
      //notifica os listeners internos
      for (JobMonitorListener listener : jobsListener) {
        listener.statusChanged(job);
      }
    });
  }

  private Job createJob(ICommandInfo commandInfo) {
    ICommandInfo.CommandStatus status = commandInfo.getStatus();
    CommandPersistenceServiceInterface commandService = ClientRemoteLocator.commandPersistenceService;
    try {
      final CommandInfo cmdInfo = commandService.getCommandInfo(commandInfo.getProjectId(), commandInfo.getCommandId());
      Job job = new Job();
      job.setJobId(cmdInfo.getId());
      String projectId = ServiceUtil.encodeToBase64((String) cmdInfo.getProjectId());
      job.setAlgorithmId(cmdInfo.getConfigurator().getAlgorithmName());
      if (cmdInfo.getConfigurator().getAlgorithmVersionId() != null) {
        job.setAlgorithmVersion(cmdInfo.getConfigurator().getAlgorithmVersionId().toString());
      }
      job.setAutomaticallyMachineSelection(cmdInfo.isAutomatic());
      job.setCpuTime(cmdInfo.getCpuTimeSec());
      job.setDescription(cmdInfo.getDescription());
      if (status==ICommandInfo.CommandStatus.FINISHED) {
        job.setEndTime(LocalDateTime.now().toString());
      }
      job.setExecutionMachine(status==ICommandInfo.CommandStatus.SCHEDULED && cmdInfo.getSGAName() != null ? cmdInfo.getSGAName() : "");
      if (status==ICommandInfo.CommandStatus.FINISHED && cmdInfo.getFinalizationInfo()!=null &&
        cmdInfo.getFinalizationInfo().getFinalizationType()!= CommandFinalizationType.NO_EXIT_CODE) {
        job.setExitCode(cmdInfo.getFinalizationInfo().getExitCode());
      }
      if (status==ICommandInfo.CommandStatus.FINISHED && cmdInfo.getFinalizationInfo()!=null) {
        job.setExitStatus(getExitStatus(cmdInfo.getFinalizationInfo().getFinalizationType()));
      }
      job.setJobOwner((String)cmdInfo.getUserId());
      job.setNumberOfAttempts(cmdInfo.getSubmissionAttempts());
      job.setPriority(cmdInfo.getPriority().ordinal());
      job.setProjectId(projectId);
      job.setRamMemory(cmdInfo.getRAMMemoryMB());
      job.setSessionId((String)cmdInfo.getProjectId());
      ProgressData progressData = cmdInfo.getProgressData();
      if (progressData!=null) {
        job.setProgressInfo(progressData.getDescription());
      }
      StatusType currentStatus = getCurrentStatus(status);
      job.setState(currentStatus);
      List<StatusChangeHistory> statusHistory = new ArrayList<>();
      StatusChangeHistory statusChangeHistory = new StatusChangeHistory();
      statusChangeHistory.setStatus(currentStatus);
      statusChangeHistory.setTimestamp(LocalDateTime.now().toString());
      statusHistory.add(statusChangeHistory);
      job.setStatusHistory(statusHistory);
      job.setSubmissionMachine(cmdInfo.getSGAName() != null ? cmdInfo.getSGAName() : "");
      job.setSubmissionTime(LocalDateTime.ofInstant(Instant.ofEpochMilli(cmdInfo.getSubmittedDate().getTime()),
        ZoneId.systemDefault()).toString());
      job.setWallclockTime(cmdInfo.getWallTimeSec());
      job.setLastModifiedTime(LocalDateTime.now().toString());
      job.setMonitoredFiles(
          cmdInfo.getMonitoredFiles().stream().
              map(monitoredFile -> {
                MonitoredFile file = new MonitoredFile();
                file.setName(monitoredFile.getFileName());
                file.setId(ServiceUtil.encodeToBase64(monitoredFile.getFileName()));
                file.setFormats(Arrays.asList(monitoredFile.getFormats()));
                return file;
              }).
              collect(Collectors.toList())
      );
      return job;
    }
    catch (Throwable e) {
      e.printStackTrace();
      return null;
    }
  }

  private Job updateJob(Job job, ICommandInfo commandInfo){
    ICommandInfo.CommandStatus status = commandInfo.getStatus();
    CommandPersistenceServiceInterface commandService = ClientRemoteLocator.commandPersistenceService;
    try {
      final CommandInfo cmdInfo = commandService.getCommandInfo(commandInfo.getProjectId(), commandInfo.getCommandId());
      job.setCpuTime(cmdInfo.getCpuTimeSec());
      if (status==ICommandInfo.CommandStatus.FINISHED) {
        job.setEndTime(LocalDateTime.now().toString());
      }
      job.setExecutionMachine(cmdInfo.getStatus()!= CommandStatus.SCHEDULED && cmdInfo.getSGAName() != null ? cmdInfo.getSGAName() : "");
      if (status==ICommandInfo.CommandStatus.FINISHED && cmdInfo.getFinalizationInfo()!=null &&
        cmdInfo.getFinalizationInfo().getFinalizationType()!= CommandFinalizationType.NO_EXIT_CODE) {
        job.setExitCode(cmdInfo.getFinalizationInfo().getExitCode());
      }
      if (status==ICommandInfo.CommandStatus.FINISHED && cmdInfo.getFinalizationInfo()!=null) {
        job.setExitStatus(getExitStatus(cmdInfo.getFinalizationInfo().getFinalizationType()));
      }
      job.setNumberOfAttempts(cmdInfo.getSubmissionAttempts());
      job.setPriority(cmdInfo.getPriority().ordinal());
      job.setRamMemory(cmdInfo.getRAMMemoryMB());
      StatusType currentStatus = getCurrentStatus(status);
      job.setState(currentStatus);
      List<StatusChangeHistory> statusHistory = job.getStatusHistory();
      StatusChangeHistory statusChangeHistory = new StatusChangeHistory();
      statusChangeHistory.setStatus(currentStatus);
      statusChangeHistory.setTimestamp(LocalDateTime.now().toString());
      statusHistory.add(statusChangeHistory);
      ProgressData progressData = cmdInfo.getProgressData();
      if (progressData!=null) {
        job.setProgressInfo(progressData.getDescription());
      }
      job.setWallclockTime(cmdInfo.getWallTimeSec());
      job.setMonitoredFiles(
          cmdInfo.getMonitoredFiles().stream().
              map(monitoredFile -> {
                MonitoredFile file = new MonitoredFile();
                file.setName(monitoredFile.getFileName());
                file.setId(ServiceUtil.encodeToBase64(monitoredFile.getFileName()));
                file.setFormats(Arrays.asList(monitoredFile.getFormats()));
                return file;
              }).
              collect(Collectors.toList())
      );
      job.setLastModifiedTime(LocalDateTime.now().toString());
      return job;
    }
    catch (Throwable e) {
      e.printStackTrace();
      return null;
    }
  }

  private Job.ExitStatusEnum getExitStatus (CommandFinalizationType finalizationType) {
    switch (finalizationType) {
      case END:
        return Job.ExitStatusEnum.UNDEFINED;
      case KILLED:
        return Job.ExitStatusEnum.KILLED;
      case SUCCESS:
        return Job.ExitStatusEnum.SUCCESS;
      case LOST:
        return Job.ExitStatusEnum.LOST;
      default:
        return Job.ExitStatusEnum.UNKNOWN;
    }
  }

  private StatusType getCurrentStatus (ICommandInfo.CommandStatus status) {
    switch (status) {
      case SCHEDULED:
        return StatusType.SCHEDULED;
      case INIT:
        return StatusType.INIT;
      case UPLOADING:
        return StatusType.UPLOADING;
      case EXECUTING:
        return StatusType.EXECUTING;
      case DOWNLOADING:
        return StatusType.DOWNLOADING;
      case FINISHED:
        return StatusType.FINISHED;
      default:
        return StatusType.UNKNOWN;
    }
  }

  public void removeJobMonitorLister(JobMonitorListener listener) {
    this.jobsListener.remove(listener);
  }

  public void addJobMonitorLister(JobMonitorListener listener) {
    this.jobsListener.add(listener);

  }

}

