package Demo;

import Requests.Algorithms.Algorithm;
import Requests.Algorithms.AlgorithmExplorer;
import Requests.Authentication.Authentication;
import Requests.Authentication.InvalidLoginOrPasswordException;
import Requests.Authentication.LoginOrPasswordNotProvidedException;
import Requests.Authentication.Token;
import Requests.Jobs.JobInfo;
import Requests.Jobs.JobServices;
import Requests.Projects.FileExplorer;
import Requests.Projects.PermissionException;
import Requests.Projects.Project;
import Requests.Projects.ProjectExplorer;
import Requests.Projects.ProjectOrFileNotFoundException;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Exemplo de submissão programática da execução de um algoritmo em um servidor
 * CSBase.
 *
 * Compatível com a versão 1.4.6 do iBase.
 *
 * @author Tecgraf/PUC-Rio
 */
public class CommandSubmissionClient {

  private static final String ALGORITHM = "zip";
  private static final String REMOTE_INPUT_FILE_1 = "input1.txt";
  private static final String REMOTE_INPUT_FILE_2 = "input2.txt";
  private static final String REMOTE_OUTPUT_FILE = "output.zip";

	/**
	 * Nome do arquivo com as propriedades para a demo
	 */
	private static String configFileName = "config.properties";

	/*
	 * Assertivas:
	 * - usuário já cadastrado
	 * - projeto previamente criado
	 * - usuário é o dono do projeto
	 * - algoritmo zip cadastrado
	 * - arquivos de entrada disponiveis localmente
	 * - caminho válido para arquivo de saída a ser criado localmente
	 * - alguma maquina remota disponivel para executar
	 */
	public static void main(String[] args) {
		System.out.println("-- Inicio da demo");

		Properties props = new Properties();
		if (args.length>0) {
			configFileName = args[0];
			try {
				InputStream input = new FileInputStream(configFileName);
				props.load(input);
				System.out.println("Propriedades carregadas de " + configFileName);

			} catch (Exception e) {
				System.out.println ("Erro na leitura do arquivo " + configFileName);
				System.exit(1);
			}
		} else {
			InputStream input = CommandSubmissionClient.class.getClassLoader().getResourceAsStream(configFileName);
			try {
				props.load(input);
				System.out.println("Propriedades default carregadas");
			} catch (Exception e) {
				System.out.println ("Erro na leitura do arquivo default " + configFileName);
				System.exit(1);
			}
		}
		String host = props.getProperty("host");
		if (host==null || host.isEmpty()) {
			System.out.println ("A propriedade host e' obrigatoria");
			System.exit(1);
		}
		String username = props.getProperty("username");
    if (username == null || username.isEmpty()) {
			System.out.println ("A propriedade username e' obrigatoria");
			System.exit(1);
		}
		String password = props.getProperty("password");
    if (password == null || password.isEmpty()) {
			System.out.println ("A propriedade password e' obrigatoria");
			System.exit(1);
		}
		String projectName = username + "/" + props.getProperty("project");
    if (projectName.equals(username + "/" + null) || projectName.equals(username + "/")) {
			System.out.println ("A propriedade project e' obrigatoria");
			System.exit(1);
		}
		String projectFolderPath = props.getProperty("projectFolderPath");
    if (projectFolderPath == null || projectFolderPath.isEmpty()) {
			System.out.println ("A propriedade projectFolderPath e' obrigatoria");
			System.exit(1);
		}
		String toUploadFilePath1 = props.getProperty("input_file_1");
    if (toUploadFilePath1 == null || toUploadFilePath1.isEmpty()) {
			System.out.println ("A propriedade input_file_1 e' obrigatoria");
			System.exit(1);
		}
		String toUploadFilePath2 = props.getProperty("input_file_2");
    if (toUploadFilePath2 == null || toUploadFilePath2.isEmpty()) {
			System.out.println ("A propriedade input_file_2 e' obrigatoria");
			System.exit(1);
		}
		String toSaveFilePath = props.getProperty("output_file");
    if (toSaveFilePath == null || toSaveFilePath.isEmpty()) {
			System.out.println ("A propriedade output_file e' obrigatoria");
			System.exit(1);
		}

		try {
			/*
			 * Autentica o usuario com login e senha configurados na demo
			 */
			Token token = Authentication.authenticate(host, username, password);

			/*
			 * Procura o projeto com o nome configurado na demo
			 */
			Project project = ProjectExplorer.findProjectByName(host, token, projectName);
			if (project==null) {
				System.out.println("Projeto " + projectName + " nao encontrado");
				System.exit(1);
			}
			System.out.println("Projeto " + projectName + " encontrado");

			/*
			 * Procura o algoritmo com o nome configurado na demo
			 */
			Algorithm algorithm = AlgorithmExplorer.findAlgorithmByName(host, token, ALGORITHM);
			if (algorithm==null) {
				System.out.println("Algoritmo " + ALGORITHM + " nao encontrado");
				System.exit(1);
			}
			System.out.println("Algoritmo " + ALGORITHM + " encontrado");

			/*
			 * Faz o upload dos dados de entrada na raiz do projeto
			 *
			 * Os arquivos que serão necessários para o algoritmo precisam ser enviados para o projeto, pois é de
			 * lá onde o algoritmo consegue ler os arquivos.
			 */
			FileExplorer.uploadFile(host, token, project, projectFolderPath, toUploadFilePath1, REMOTE_INPUT_FILE_1);
			FileExplorer.uploadFile(host, token, project, projectFolderPath, toUploadFilePath2, REMOTE_INPUT_FILE_2);

			/*
			 * Cria a lista que será utilizada como parâmetro do algoritmo.
			 *
			 * O algoritmo Zip recebe como parâmetro de entrada uma lista de arquivos (que
			 * pode conter somente um arquivo). No exemplo, utilizamos dois arquivos.
			 */
      List<String> inputFiles = new ArrayList<>();
      inputFiles.add(REMOTE_INPUT_FILE_1);
      inputFiles.add(REMOTE_INPUT_FILE_2);

      /*
       * O nome dos parâmetros (aqui no exemplo "ENTRADA" e "SAIDA") são definidos pelo próprio algoritmo, dentro de seu arquivo config.xml
       * Acesse os links abaixo para mais informações sobre o config.xml
       * https://jira.tecgraf.puc-rio.br/confluence/pages/viewpage.action?pageId=36504138
       * http://webserver2.tecgraf.puc-rio.br/ftp_pub/csbase/1.5.5/manualConfiguradorXML.pdf
       */
      Map<String, Object> commandArgs = new HashMap<>();
      commandArgs.put("ENTRADA", inputFiles);
      commandArgs.put("SAIDA", REMOTE_OUTPUT_FILE);

      /*
       * Executa o algoritmo Zip (obtido previamente)
       */
			String jobId = JobServices.submitJob(host, token, project, algorithm, algorithm.getVersions().get(0),
					"demo rest java", 0, false, new String[0], commandArgs);
			if (jobId==null) {
				System.out.println("Falha na submissao do algoritmo " + ALGORITHM);
				System.exit(1);
			}
			System.out.println("Algoritmo " + ALGORITHM + " submetido com sucesso");

			/*
			 * Obtém as informações do job criado para a execução do algoritmo.
			 */
			JobInfo jinfo = JobServices.getJobInfo(host, token, jobId);
			if (jinfo==null) {
				System.out.println("Job "+ jobId + " nao foi encontrado");
				System.exit(1);
			}
			System.out.println("Job " + jobId + " encontrado");

      /*
       * Aguarda o job terminar sua execução
       */
			JobServices.awaitJobEnd(host, token, jinfo);

      /*
       * Faz download do arquivo (quando o arquivo é pequeno e somente
       * texto)
       *
       * obs: como o arquivo de saída é um binário (zip), esse exemplo está
       * sendo feito com um dos arquivos de entrada que foi enviado para o
       * servidor, mas basta alterar o nome do arquivo na chamada para obter o
       * arquivo pretendido.
       */
      String textResult = FileExplorer.downloadTextFile(host, token, project, projectFolderPath, REMOTE_INPUT_FILE_1);

      System.out.println("\nResultado em " + REMOTE_INPUT_FILE_1 + ":\n" + textResult);

      /*
       * Faz download de arquivo. (quando o arquivo é pequeno e qualquer formato)
       *
       * obs: neste caso fazemos download do arquivo de saída.
       */
      InputStream fileStream = FileExplorer.downloadBinaryFile(host, token, project, projectFolderPath, REMOTE_OUTPUT_FILE);

      long fileSize = Files.copy(fileStream, new File(toSaveFilePath).toPath(), StandardCopyOption.REPLACE_EXISTING);
      fileStream.close();

      System.out.println("\nResultado em " + REMOTE_OUTPUT_FILE + ": " + fileSize + " bytes ");

			/*
			 * Faz download do arquivo de saída (quando o arquivo é grande)
			 *
			 * obs: colocamos um prefixo no nome do arquivo a ser salvo para diferenciar do arquivo obtido no exemplo acima.
			 */
      FileExplorer.downloadLargeFile(host, token, project, projectFolderPath, REMOTE_OUTPUT_FILE, "OUTRO" + toSaveFilePath);

      System.out.println("\nResultado em " + "OUTRO" + toSaveFilePath + ": " + new File("OUTRO" + toSaveFilePath).length() + " bytes ");

		} catch (LoginOrPasswordNotProvidedException e) {
			e.printStackTrace();
		} catch (InvalidLoginOrPasswordException e) {
			e.printStackTrace();
		} catch (ProjectOrFileNotFoundException e) {
			e.printStackTrace();
		} catch (PermissionException e) {
			e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    }
	}
}
