/*
 * ServerProjectFileTypes.java
 * 
 * $Author: cassino $ $Revision: 116325 $ - $Date: 2006-12-07 15:53:32 -0200
 * (Thu, 07 Dec 2006) $
 */
package csbase.server.services.projectservice;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import csbase.exception.ServiceFailureException;
import csbase.logic.ProjectFileType;
import csbase.logic.ProjectFileTypeInfo;
import csbase.server.Server;

/**
 * Classe responsavel por obter as informacoes relativas aos tipos de arquivos.
 * As propriedades dos tipos de arquivos ficam no arquivo FileTypes.properties
 * cujo o path e' especificado nas propriedades do servico de projeto.
 */
class ServerProjectFileType {
  /** Intancia unica da classe */
  private static ServerProjectFileType instance;
  /** Path do arquivo de propriedades */
  private String resourceName;
  /** Hash do tipos dos arquivos */
  private Hashtable<String, ProjectFileTypeInfo> fileTypes;
  /** Locale para a descricao do tipos */
  private Locale locale;

  public static ServerProjectFileType getInstance() {
    if (instance == null) {
      instance = new ServerProjectFileType();
    }
    return instance;
  }

  /**
   * Carrega as propriedades de todos os arquivo.
   * 
   * @return .
   */
  private Properties loadProperties() {
    try {
      Properties properties = new Properties();
      InputStream in = getClass().getResourceAsStream(resourceName);
      if (in == null) {
        throw new ServiceFailureException(resourceName + " no encontrado.");
      }
      properties.load(in);
      in.close();
      return properties;
    }
    catch (IOException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * Obtem todas as propriedades dos arquivos.
   * 
   * @param loc .
   * 
   * @return .
   */
  public Hashtable<String, ProjectFileTypeInfo> getAllTypes(Locale loc) {
    loadFileTypes(loc);
    return fileTypes;
  }

  /**
   * Obtem todas as propriedades de um tipo de arquivo.
   * 
   * @param typeCode o tipo do arquivo desejado
   * 
   * @return informaes de um tipo de arquivo
   */
  public ProjectFileTypeInfo getFileType(String typeCode) {
    if (fileTypes == null) {
      Locale loc = ProjectService.getInstance().getDefaultLocale();
      loadFileTypes(loc);
    }
    ProjectFileTypeInfo info = fileTypes.get(typeCode);
    if (info == null) {
      info = fileTypes.get(ProjectFileType.UNKNOWN);
    }
    return info;
  }

  private void loadFileTypes(Locale loc) {
    if ((fileTypes != null) && ((locale != null) && (locale.equals(loc)))) {
      return;
    }
    Properties properties = loadProperties();
    if (fileTypes == null) {
      fileTypes = new Hashtable<String, ProjectFileTypeInfo>();
      for (int i = 0; true; i++) {
        ProjectFileTypeInfo fileType = makeFileType(properties, i, loc);
        if (fileType == null) {
          break;
        }
        fileTypes.put(fileType.typeCode, fileType);
      }
    }
    else {
      /* Se entrou aqui e' porque o locale e' diferente. */
      /* Entao, precisa atualizar a description com o novo locale */
      for (Enumeration e = fileTypes.keys(); e.hasMoreElements();) {
        String typeCode = (String) e.nextElement();
        String description = getDescription(typeCode, properties, loc);
        (fileTypes.get(typeCode)).description = description;
      }
    }
    this.locale = loc;
  }

  /**
   * Obtem o mimeType a partir do tipo do arquivo
   * 
   * @param type .
   * 
   * @return .
   */
  public String getMimeType(String type) {
    if (fileTypes == null) {
      loadFileTypes(null);
    }
    String mimeType = (fileTypes.get(type)).mimeType;
    if (mimeType == null) {
      mimeType = "application/octet-stream";
    }
    return mimeType;
  }

  /**
   * Constri um tipo de arquivo a partir das propriedades especificadas.
   * 
   * @param properties .
   * @param id .
   * @param loc .
   * 
   * @return .
   */
  private ProjectFileTypeInfo makeFileType(Properties properties, int id,
    Locale loc) {
    final String typeCode = getTypeCode(id, properties);
    // Se o type code no for encontrado no existe mais tipos
    if (typeCode == null) {
      return null;
    }
    try {
      String description = getDescription(typeCode, properties, loc);
      String mimeType = getMimeType(typeCode, properties);
      byte[] baseIcon = getImage(typeCode, properties, "baseIcon");
      byte[] ucIcon = getImage(typeCode, properties, "ucIcon");
      byte[] cutIcon = getImage(typeCode, properties, "cutIcon");
      List<String> extensions = getExtensions(typeCode, properties);
      ProjectFileTypeInfo fileType =
        new ProjectFileTypeInfo(typeCode, description, mimeType, baseIcon,
          ucIcon, cutIcon, extensions);
      return fileType;
    }
    catch (Exception e) {
      Server.logSevereMessage(e.getMessage(), e);
      return null;
    }
  }

  /**
   * Obtm o valor da propriedade que possui o identificador de um tipo de
   * arquivo.
   * 
   * @param i .
   * @param properties .
   * 
   * @return .
   */
  private String getTypeCode(int i, Properties properties) {
    String propName = "filetype." + i + ".typeCode";
    final String prop = properties.getProperty(propName);
    if (prop == null) {
      return null;
    }
    final String code = prop.trim();
    return code;
  }

  /**
   * Obtm o valor da propriedade que possui a descricao de um tipo de arquivo.
   * 
   * @param typeCode .
   * @param properties .
   * @param loc .
   * 
   * @return .
   */
  private String getDescription(String typeCode, Properties properties,
    Locale loc) {
    if (loc == null) {
      return null;
    }
    String propName = "filetype." + typeCode + ".description_" + loc;
    String description = properties.getProperty(propName);
    return description;
  }

  /**
   * Obtm o valor da propriedade que possui o mime-type de um tipo de arquivo
   * 
   * @param typeCode .
   * @param properties .
   * 
   * @return .
   */
  private String getMimeType(String typeCode, Properties properties) {
    String propName = "filetype." + typeCode + ".mimeType";
    String mimeType = properties.getProperty(propName);
    return mimeType;
  }

  /**
   * Obtm o cone criado a partir de um nome de arquivo definido em uma
   * propriedade para o cone de arquivo regular.
   * 
   * @param typeCode .
   * @param properties .
   * @param imageName .
   * 
   * @return .
   */
  private byte[] getImage(String typeCode, Properties properties,
    String imageName) {
    byte[] image = null;
    try {
      String propName = "filetype." + typeCode + "." + imageName;
      String fileName = properties.getProperty(propName);
      InputStream fin = getClass().getResourceAsStream(fileName);
      InputStream in = new BufferedInputStream(fin);
      int size = in.available();
      image = new byte[size];
      in.read(image);
      fin.close();
    }
    catch (Exception e) {
      /* caso nao encontre a imagem, printa uma mensagem e */
      /* coloca a imagem default (arquivo desconhecido) */
      String errMsg =
        "A imagem (" + imageName + ") do tipo de arquivo " + typeCode
          + " nao foi encontrada.";
      Server.logWarningMessage(errMsg);
      image = (fileTypes.get("UNKNOWN")).baseIcon;
    }
    return image;
  }

  /**
   * Obtm o valor da propriedade que possui as extensoes de um tipo de arquivo.
   * 
   * @param typeCode .
   * @param properties .
   * 
   * @return .
   */
  private List<String> getExtensions(String typeCode, Properties properties) {
    String propName = "filetype." + typeCode + ".extensions";
    String extensions = properties.getProperty(propName);
    if (extensions == null) {
      return null;
    }
    String[] extS = extensions.split(",");
    List<String> extVector = new ArrayList<String>(extS.length);
    for (int i = 0; i < extS.length; i++) {
      extVector.add(extS[i].trim().toLowerCase());
    }
    return extVector;
  }

  /**
   * Obtem o tipo a partir da extensao do nome do arquivo.
   * 
   * @param name Nome do arquivo
   * 
   * @return Tipo do arquivo que possui a extensao ou "UNKNOWN" caso nenhum
   *         arquivo tenha este tipo.
   */
  public String getTypeFromExtension(String name) {
    if (fileTypes == null) {
      loadFileTypes(null);
    }
    /* Obtem a extensao do arquivo */
    int index = name.lastIndexOf(".");
    if (index == -1) {
      return ProjectFileType.UNKNOWN;
    }
    String extension = name.substring(index + 1);
    for (Enumeration e = fileTypes.keys(); e.hasMoreElements();) {
      String code = (String) e.nextElement();
      ProjectFileTypeInfo typeInfo = fileTypes.get(code);
      if ((typeInfo.extensions != null)
        && (typeInfo.extensions.contains(extension.toLowerCase()))) {
        return code;
      }
    }
    return ProjectFileType.UNKNOWN;
  }

  private ServerProjectFileType() {
    ProjectService srv = ProjectService.getInstance();
    resourceName = srv.getFileTypesProperty();
  }
}
