/**
 * $Id$
 */
package csbase.server.services.projectservice.v1_01;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import tecgraf.openbus.data_service.core.v1_01.DataDescription;
import tecgraf.openbus.data_service.core.v1_01.DataKeyWrapper;
import tecgraf.openbus.data_service.core.v1_01.DataView;
import tecgraf.openbus.data_service.core.v1_01.DefaultView;
import tecgraf.openbus.data_service.core.v1_01.InvalidDataKey;
import tecgraf.openbus.data_service.core.v1_01.Metadata;
import tecgraf.openbus.data_service.core.v1_01.ServiceFailure;
import tecgraf.openbus.data_service.core.v1_01.UnstructuredDataView;
import tecgraf.openbus.data_service.core.v1_01.UnstructuredDataViewHelper;
import tecgraf.openbus.data_service.core.v1_01.UnstructuredDataViewImpl;
import tecgraf.openbus.data_service.core.v1_01.UnsupportedView;
import tecgraf.openbus.data_service.hierarchical.v1_01.HierarchicalNodeDataViewHelper;
import tecgraf.openbus.data_service.project.v1_01.ProjectDataView;
import tecgraf.openbus.data_service.project.v1_01.ProjectDataViewHelper;
import tecgraf.openbus.data_service.project.v1_01.ProjectDataViewImpl;
import tecgraf.openbus.data_service.project.v1_01.ProjectItemDataView;
import tecgraf.openbus.data_service.project.v1_01.ProjectItemDataViewHelper;
import tecgraf.openbus.data_service.project.v1_01.ProjectItemDataViewImpl;
import csbase.server.Server;

/**
 * Classe com mtodos estticos para fazer a converso de estruturas de dados e
 * excees entreas as verses 1.1 e 1.2 do DataService.
 * 
 * @author Tecgraf
 */
public class DataBridge {

  /**
   * Converte uma estrutura DataDescription da verso 1.1 para a verso 1.2.
   * 
   * @param dataDescriptionV1_01 DataDescription na verso 1.1 para ser
   *        convertida.
   * 
   * @return DataDescription na verso 1.2.
   * 
   * @throws InvalidDataKey caso a chave da DataDescription na verso 1.1 seja
   *         invlida
   * @throws ServiceFailure caso a chave convertida seja invlida
   * @throws UnsupportedView caso alguma viso da verso 1.1 no seja suportada
   *         pela verso 1.2.
   */
  static tecgraf.openbus.data_service.core.v1_02.DataDescription convertDataDescriptionToV1_02(
    DataDescription dataDescriptionV1_01) throws InvalidDataKey,
    ServiceFailure, UnsupportedView {
    tecgraf.openbus.data_service.core.v1_02.DataDescription dataDescriptionV1_02 =
      new tecgraf.openbus.data_service.core.v1_02.DataDescription();

    // DataKey
    try {
      dataDescriptionV1_02.fKey =
        convertDataKeyToV1_02(dataDescriptionV1_01.fKey);
    }
    catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e) {
      String msg = "Erro ao converter DataDescription para a verso 1.2.";
      Server.logSevereMessage(msg);
      throw new ServiceFailure(msg);
    }

    // Name
    dataDescriptionV1_02.fName = dataDescriptionV1_01.fName;

    // DefaultView
    tecgraf.openbus.data_service.core.v1_02.DefaultView defaultViewV1_02 =
      new tecgraf.openbus.data_service.core.v1_02.DefaultView(
        convertInterfaceNameToV1_02(dataDescriptionV1_01.fDefaultView.fInterfaceName),
        convertDataViewToV1_02(dataDescriptionV1_01.fDefaultView.fValue));
    dataDescriptionV1_02.fDefaultView = defaultViewV1_02;

    // OthersViews
    dataDescriptionV1_02.fOthersViews =
      convertInterfaceNameSeqToV1_02(dataDescriptionV1_01.fOthersViews);

    // Metadata
    tecgraf.openbus.data_service.core.v1_02.Metadata[] metadataV1_02 =
      new tecgraf.openbus.data_service.core.v1_02.Metadata[dataDescriptionV1_01.fMetadata.length];
    for (int i = 0; i < dataDescriptionV1_01.fMetadata.length; i++) {
      metadataV1_02[i] =
        new tecgraf.openbus.data_service.core.v1_02.Metadata(
          dataDescriptionV1_01.fMetadata[i].fName,
          dataDescriptionV1_01.fMetadata[i].fValue);
    }
    dataDescriptionV1_02.fMetadata = metadataV1_02;

    return dataDescriptionV1_02;
  }

  /**
   * Converte uma chave de dado da verso 1.1 para a verso 1.2.
   * 
   * @param keyV1_01 chave de dado na verso 1.1
   * 
   * @return chave de dado na verso 1.2
   * 
   * @throws InvalidDataKey caso a chave de dado na verso 1.1 seja invlida
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         convertida para verso 1.2 seja invlida
   */
  static byte[] convertDataKeyToV1_02(byte[] keyV1_01) throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {
    if (keyV1_01 == null) {
      return null;
    }
    if (keyV1_01.length == 0) {
      return new byte[0];
    }
    DataKeyWrapper dataKeyWrapperV1_01 = new DataKeyWrapper(keyV1_01);
    tecgraf.openbus.data_service.core.v1_02.DataKeyWrapper dataKeyWrapperV1_02 =
      new tecgraf.openbus.data_service.core.v1_02.DataKeyWrapper(
        dataKeyWrapperV1_01.getSystemDeploymentId(), dataKeyWrapperV1_01
          .getDataSourceId(), dataKeyWrapperV1_01.getDataId());
    return dataKeyWrapperV1_02.getKey();
  }

  /**
   * Converte uma lista de chaves de dados da verso 1.1 para a verso 1.2.
   * 
   * @param keySeqV1_01 lista de chaves de dados na verso 1.1
   * 
   * @return lista de chaves de dados na verso 1.2
   * 
   * @throws InvalidDataKey caso alguma chave de dado na verso 1.1 seja
   *         invlida
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso alguma
   *         chave convertida para verso 1.2 seja invlida
   */
  static byte[][] convertDataKeySeqToV1_02(byte[][] keySeqV1_01)
    throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {
    byte[][] keySeqV1_02 = new byte[keySeqV1_01.length][];

    List<byte[]> invalidKeyList = new LinkedList<byte[]>();

    for (int i = 0; i < keySeqV1_01.length; i++) {
      try {
        keySeqV1_02[i] = convertDataKeyToV1_02(keySeqV1_01[i]);
      }
      catch (InvalidDataKey e) {
        for (byte[] k : e.fKeys) {
          invalidKeyList.add(k);
        }
      }
    }

    if (invalidKeyList.size() > 0) {
      throw new InvalidDataKey(invalidKeyList.toArray(new byte[0][]));
    }

    return keySeqV1_02;
  }

  /**
   * Converte uma estrutura DataView da verso 1.1 para a verso 1.2.
   * 
   * @param dataViewV1_01 DataView na verso 1.1 para ser convertida.
   * 
   * @return DataView na verso 1.2.
   * 
   * @throws InvalidDataKey caso a chave da DataView na verso 1.1 seja invlida
   * @throws ServiceFailure caso a chave convertida seja invlida
   */
  static tecgraf.openbus.data_service.core.v1_02.DataView convertDataViewToV1_02(
    DataView dataViewV1_01) throws InvalidDataKey, ServiceFailure {
    if (dataViewV1_01 == null) {
      return null;
    }

    try {
      if (dataViewV1_01.getInterfaceName().equals(
        UnstructuredDataViewHelper.id())) {
        return convertUnstructuredDataViewToV1_02(UnstructuredDataView.class
          .cast(dataViewV1_01));
      }
      else if (dataViewV1_01.getInterfaceName().equals(
        ProjectDataViewHelper.id())) {
        return convertProjectDataViewToV1_02(ProjectDataView.class
          .cast(dataViewV1_01));
      }
      else if (dataViewV1_01.getInterfaceName().equals(
        ProjectItemDataViewHelper.id())) {
        return convertProjectItemDataViewToV1_02(ProjectItemDataView.class
          .cast(dataViewV1_01));
      }
    }
    catch (InvalidDataKey e) {
      throw e;
    }
    catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e) {
      String msg = "Erro ao converter DataView para a verso 1.2.";
      Server.logSevereMessage(msg);
      throw new ServiceFailure(msg);
    }

    return null;
  }

  /**
   * Converte uma UnstructuredDataView da verso 1.1 para a verso 1.2.
   * 
   * @param udvV1_01 a UnstructuredDataView na verso 1.1
   * 
   * @return a UnstructuredDataView na verso 1.2
   * 
   * @throws InvalidDataKey caso a chave da DataView na verso 1.1 seja invlida
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         convertida para verso 1.2 seja invlida
   */
  private static tecgraf.openbus.data_service.core.v1_02.UnstructuredDataView convertUnstructuredDataViewToV1_02(
    UnstructuredDataView udvV1_01) throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {
    return new tecgraf.openbus.data_service.core.v1_02.UnstructuredDataViewImpl(
      convertDataKeyToV1_02(udvV1_01.fKey), udvV1_01.fHost, udvV1_01.fPort,
      udvV1_01.fAccessKey, true);
  }

  /**
   * Converte uma ProjectDataView da verso 1.1 para a verso 1.2.
   * 
   * @param pdvV1_01 a ProjectDataView na verso 1.1
   * 
   * @return a ProjectDataView na verso 1.2
   * 
   * @throws InvalidDataKey caso a chave da DataView na verso 1.1 seja invlida
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         convertida para verso 1.2 seja invlida
   */
  private static tecgraf.openbus.data_service.project.v1_02.ProjectDataView convertProjectDataViewToV1_02(
    ProjectDataView pdvV1_01) throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {

    Set<String> readingUsers =
      new HashSet<String>(pdvV1_01.fReadingUsers.length);
    for (Object user : pdvV1_01.fReadingUsers) {
      readingUsers.add(String.class.cast(user));
    }

    Set<String> writingUsers =
      new HashSet<String>(pdvV1_01.fWritingUsers.length);
    for (Object user : pdvV1_01.fWritingUsers) {
      writingUsers.add(String.class.cast(user));
    }

    return new tecgraf.openbus.data_service.project.v1_02.ProjectDataViewImpl(
      convertDataKeyToV1_02(pdvV1_01.fKey), //key
      pdvV1_01.fOwner, //owner
      pdvV1_01.fDescription, //description
      pdvV1_01.fPath, //path
      pdvV1_01.fSoftware, //software
      pdvV1_01.fSize, //size
      readingUsers, //readingUsers 
      writingUsers, //writingUsers
      pdvV1_01.fCreationDate, //creationDate 
      pdvV1_01.fModificationDate //modificationDate
    );
  }

  /**
   * Converte uma ProjectItemDataView da verso 1.1 para a verso 1.2.
   * 
   * @param pidvV1_01 a ProjectItemDataView na verso 1.1
   * 
   * @return a ProjectItemDataView na verso 1.2
   * 
   * @throws InvalidDataKey caso a chave da DataView na verso 1.1 seja invlida
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         convertida para verso 1.2 seja invlida
   */
  private static tecgraf.openbus.data_service.project.v1_02.ProjectItemDataView convertProjectItemDataViewToV1_02(
    ProjectItemDataView pidvV1_01) throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {
    return new tecgraf.openbus.data_service.project.v1_02.ProjectItemDataViewImpl(
      convertDataKeyToV1_02(pidvV1_01.fKey), //key 
      pidvV1_01.fOwner, //owner
      pidvV1_01.fDescription, //description
      pidvV1_01.fPath, //path
      pidvV1_01.fType, //type
      pidvV1_01.fSize, //size
      pidvV1_01.fIsContainer, //isContainer
      pidvV1_01.fCanRead, //canRead
      pidvV1_01.fCanWrite, //canWrite
      pidvV1_01.fCreationDate, //creationDate 
      pidvV1_01.fModificationDate//modificationDate
    );
  }

  /**
   * Converte o nome de uma viso de dados na verso 1.1 para verso 1.2.
   * 
   * @param interfaceNameV1_01 o nome da viso de dados na verso 1.1
   * 
   * @return o nome da viso de dados na verso 1.2
   * 
   * @throws UnsupportedView caso a viso no seja suportada
   */
  static String convertInterfaceNameToV1_02(String interfaceNameV1_01)
    throws UnsupportedView {
    if (interfaceNameV1_01 == null) {
      return "";
    }

    if (interfaceNameV1_01.length() == 0) {
      return "";
    }

    if (interfaceNameV1_01.equals(UnstructuredDataViewHelper.id())) {
      return tecgraf.openbus.data_service.core.v1_02.UnstructuredDataViewHelper
        .id();
    }
    else if (interfaceNameV1_01.equals(HierarchicalNodeDataViewHelper.id())) {
      throw new UnsupportedView();
    }
    else if (interfaceNameV1_01.equals(ProjectDataViewHelper.id())) {
      return tecgraf.openbus.data_service.project.v1_02.ProjectDataViewHelper
        .id();
    }
    else if (interfaceNameV1_01.equals(ProjectItemDataViewHelper.id())) {
      return tecgraf.openbus.data_service.project.v1_02.ProjectItemDataViewHelper
        .id();
    }

    throw new UnsupportedView();
  }

  /**
   * Converte o nome de uma lista de vises de dados na verso 1.1 para verso
   * 1.2.
   * 
   * @param interfaceNameSeqV1_01 a lista de nomes de vises de dados na verso
   *        1.1
   * 
   * @return a lista de nomes de vises de dados na verso 1.2
   * 
   * @throws UnsupportedView caso a viso no seja suportada
   */
  static String[] convertInterfaceNameSeqToV1_02(String[] interfaceNameSeqV1_01)
    throws UnsupportedView {
    if (interfaceNameSeqV1_01 == null) {
      return new String[0];
    }
    List<String> interfaceNameSeqV1_02 = new LinkedList<String>();
    for (int i = 0; i < interfaceNameSeqV1_01.length; i++) {
      String interfaceNameV1_02 =
        convertInterfaceNameToV1_01(interfaceNameSeqV1_01[i]);
      if ((interfaceNameV1_02 != null) && (interfaceNameV1_02.length() > 0)) {
        interfaceNameSeqV1_02.add(interfaceNameV1_02);
      }
    }
    return interfaceNameSeqV1_02.toArray(new String[0]);
  }

  //Verso 1.2 -> 1.1

  /**
   * Converte uma estrutura DataDescription da verso 1.2 para a verso 1.1.
   * 
   * @param dataDescriptionV1_02 DataDescription na verso 1.2 para ser
   *        convertida.
   * 
   * @return DataDescription na verso 1.1.
   * 
   * @throws InvalidDataKey caso a chave da DataDescription na verso 1.1 seja
   *         invlida
   * @throws ServiceFailure caso a chave convertida seja invlida
   */
  static DataDescription convertDataDescriptionToV1_01(
    tecgraf.openbus.data_service.core.v1_02.DataDescription dataDescriptionV1_02)
    throws InvalidDataKey, ServiceFailure {
    DataDescription dataDescriptionV1_01 = new DataDescription();

    // DataKey
    try {
      dataDescriptionV1_01.fKey =
        convertDataKeyToV1_01(dataDescriptionV1_02.fKey);
    }
    catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e) {
      String msg = "Erro ao converter DataDescription para a verso 1.1.";
      Server.logSevereMessage(msg);
      throw new ServiceFailure(msg);
    }

    // Name
    dataDescriptionV1_01.fName = dataDescriptionV1_02.fName;

    // DefaultView
    DefaultView defaultViewV1_01 =
      new DefaultView(
        convertInterfaceNameToV1_01(dataDescriptionV1_02.fDefaultView.fInterfaceName),
        convertDataViewToV1_01(dataDescriptionV1_02.fDefaultView.fValue));
    dataDescriptionV1_01.fDefaultView = defaultViewV1_01;

    // OthersViews
    dataDescriptionV1_01.fOthersViews =
      convertInterfaceNameSeqToV1_01(dataDescriptionV1_02.fOthersViews);

    // Metadata
    Metadata[] metadataV1_01 =
      new Metadata[dataDescriptionV1_02.fMetadata.length];
    for (int i = 0; i < dataDescriptionV1_02.fMetadata.length; i++) {
      metadataV1_01[i] =
        new Metadata(dataDescriptionV1_02.fMetadata[i].fName,
          dataDescriptionV1_02.fMetadata[i].fValue);
    }
    dataDescriptionV1_01.fMetadata = metadataV1_01;

    return dataDescriptionV1_01;
  }

  /**
   * Converte uma lista de estruturas DataDescription da verso 1.2 para a
   * verso 1.1.
   * 
   * @param dataDescriptionSeqV1_02 a lista de DataDescription na verso 1.2
   *        para ser convertida.
   * 
   * @return a lista de DataDescription na verso 1.1.
   * 
   * @throws InvalidDataKey caso a chave de alguma DataDescription na verso 1.1
   *         seja invlida
   * @throws ServiceFailure caso a chave convertida seja invlida
   */
  static DataDescription[] convertDataDescriptionSeqToV1_01(
    tecgraf.openbus.data_service.core.v1_02.DataDescription[] dataDescriptionSeqV1_02)
    throws InvalidDataKey, ServiceFailure {
    DataDescription[] ddSeqV1_01 =
      new DataDescription[dataDescriptionSeqV1_02.length];

    for (int i = 0; i < dataDescriptionSeqV1_02.length; i++) {
      ddSeqV1_01[i] = convertDataDescriptionToV1_01(dataDescriptionSeqV1_02[i]);
    }

    return ddSeqV1_01;
  }

  /**
   * Converte uma chave de dado da verso 1.2 para a verso 1.1.
   * 
   * @param keyV1_02 chave de dado na verso 1.2
   * 
   * @return chave de dado na verso 1.1
   * 
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         de dado na verso 1.2 seja invlida
   * @throws InvalidDataKey caso a chave convertida para verso 1.1 seja
   *         invlida
   */
  static byte[] convertDataKeyToV1_01(byte[] keyV1_02)
    throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey,
    InvalidDataKey {
    if (keyV1_02.length == 0) {
      return new byte[0];
    }
    tecgraf.openbus.data_service.core.v1_02.DataKeyWrapper dataKeyWrapperV1_02 =
      new tecgraf.openbus.data_service.core.v1_02.DataKeyWrapper(keyV1_02);
    DataKeyWrapper dataKeyWrapperV1_01 =
      new DataKeyWrapper(dataKeyWrapperV1_02.getSystemDeploymentId(),
        dataKeyWrapperV1_02.getDataSourceId(), dataKeyWrapperV1_02.getDataId());
    return dataKeyWrapperV1_01.getKey();
  }

  /**
   * Converte uma lista de chaves de dados da verso 1.2 para a verso 1.1.
   * 
   * @param keySeqV1_02 a lista de chaves de dados na verso 1.2
   * 
   * @return a lista de chaves de dados na verso 1.1
   * 
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         de dado na verso 1.2 seja invlida
   * @throws InvalidDataKey caso a chave convertida para verso 1.1 seja
   *         invlida
   */
  static byte[][] convertDataKeySeqToV1_01(byte[][] keySeqV1_02)
    throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {
    byte[][] keySeqV1_01 = new byte[keySeqV1_02.length][];

    List<byte[]> invalidKeyList = new LinkedList<byte[]>();

    for (int i = 0; i < keySeqV1_02.length; i++) {
      try {
        keySeqV1_01[i] = convertDataKeyToV1_01(keySeqV1_02[i]);
      }
      catch (InvalidDataKey e) {
        for (byte[] k : e.fKeys) {
          invalidKeyList.add(k);
        }
      }
    }

    if (invalidKeyList.size() > 0) {
      throw new InvalidDataKey(invalidKeyList.toArray(new byte[0][]));
    }

    return keySeqV1_01;
  }

  /**
   * Converte uma estrutura DataView da verso 1.2 para a verso 1.1.
   * 
   * @param dataViewV1_02 DataView na verso 1.2 para ser convertida.
   * 
   * @return DataView na verso 1.1.
   * 
   * @throws InvalidDataKey caso a chave da DataView na verso 1.1 seja invlida
   * @throws ServiceFailure caso a chave convertida seja invlida
   */
  static DataView convertDataViewToV1_01(
    tecgraf.openbus.data_service.core.v1_02.DataView dataViewV1_02)
    throws InvalidDataKey, ServiceFailure {
    if (dataViewV1_02 == null) {
      return null;
    }

    try {
      if (dataViewV1_02.getInterfaceName()
        .equals(
          tecgraf.openbus.data_service.core.v1_02.UnstructuredDataViewHelper
            .id())) {
        return convertUnstructuredDataViewToV1_01(tecgraf.openbus.data_service.core.v1_02.UnstructuredDataView.class
          .cast(dataViewV1_02));
      }
      else if (dataViewV1_02.getInterfaceName().equals(
        HierarchicalNodeDataViewHelper.id())) {
        return null;
      }
      else if (dataViewV1_02.getInterfaceName().equals(
        tecgraf.openbus.data_service.project.v1_02.ProjectDataViewHelper.id())) {
        return convertProjectDataViewToV1_01(tecgraf.openbus.data_service.project.v1_02.ProjectDataView.class
          .cast(dataViewV1_02));
      }
      else if (dataViewV1_02.getInterfaceName().equals(
        tecgraf.openbus.data_service.project.v1_02.ProjectItemDataViewHelper
          .id())) {
        return convertProjectItemDataViewToV1_01(tecgraf.openbus.data_service.project.v1_02.ProjectItemDataView.class
          .cast(dataViewV1_02));
      }
    }
    catch (InvalidDataKey e) {
      throw e;
    }
    catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e) {
      String msg = "Erro ao converter DataView para a verso 1.1.";
      Server.logSevereMessage(msg);
      throw new ServiceFailure(msg);
    }

    return null;
  }

  /**
   * Converte uma lista de DataView da verso 1.2 para a verso 1.1.
   * 
   * @param dataViewSeqV1_02 lista de DataView na verso 1.2 para ser
   *        convertida.
   * 
   * @return lista de DataView na verso 1.1.
   * 
   * @throws InvalidDataKey caso alguma chave da DataView na verso 1.1 seja
   *         invlida
   * @throws ServiceFailure caso alguma chave convertida seja invlida
   */
  static DataView[] convertDataViewSeqToV1_01(
    tecgraf.openbus.data_service.core.v1_02.DataView[] dataViewSeqV1_02)
    throws InvalidDataKey, ServiceFailure {
    DataView[] DataViewSeqV1_01 = new DataView[dataViewSeqV1_02.length];

    for (int i = 0; i < dataViewSeqV1_02.length; i++) {
      DataViewSeqV1_01[i] = convertDataViewToV1_01(dataViewSeqV1_02[i]);
    }

    return DataViewSeqV1_01;
  }

  /**
   * Converte uma UnstructuredDataView da verso 1.2 para a verso 1.1.
   * 
   * @param udvV1_02 a UnstructuredDataView na verso 1.2
   * 
   * @return a UnstructuredDataView na verso 1.1
   * 
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         da DataView na verso 1.2 seja invlida
   * @throws InvalidDataKey caso a chave convertida para verso 1.1 seja
   *         invlida
   */
  private static UnstructuredDataView convertUnstructuredDataViewToV1_01(
    tecgraf.openbus.data_service.core.v1_02.UnstructuredDataView udvV1_02)
    throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {
    return new UnstructuredDataViewImpl(convertDataKeyToV1_01(udvV1_02.fKey),
      udvV1_02.fHost, udvV1_02.fPort, udvV1_02.fAccessKey, true);
  }

  /**
   * Converte uma ProjectDataView da verso 1.2 para a verso 1.1.
   * 
   * @param pdvV1_02 a ProjectDataView na verso 1.2
   * 
   * @return a ProjectDataView na verso 1.1
   * 
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         da DataView na verso 1.2 seja invlida
   * @throws InvalidDataKey caso a chave convertida para verso 1.1 seja
   *         invlida
   */
  private static ProjectDataView convertProjectDataViewToV1_01(
    tecgraf.openbus.data_service.project.v1_02.ProjectDataView pdvV1_02)
    throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {

    Set<String> readingUsers =
      new HashSet<String>(pdvV1_02.fReadingUsers.length);
    for (Object user : pdvV1_02.fReadingUsers) {
      readingUsers.add(String.class.cast(user));
    }

    Set<String> writingUsers =
      new HashSet<String>(pdvV1_02.fWritingUsers.length);
    for (Object user : pdvV1_02.fWritingUsers) {
      writingUsers.add(String.class.cast(user));
    }

    return new ProjectDataViewImpl(convertDataKeyToV1_01(pdvV1_02.fKey), //key
      pdvV1_02.fOwner, //owner
      pdvV1_02.fDescription, //description
      pdvV1_02.fPath, //path
      pdvV1_02.fSoftware, //software
      pdvV1_02.fSize, //size
      readingUsers, //readingUsers 
      writingUsers, //writingUsers
      pdvV1_02.fCreationDate, //creationDate 
      pdvV1_02.fModificationDate //modificationDate
    );
  }

  /**
   * Converte uma lista de ProjectDataView da verso 1.2 para a verso 1.1.
   * 
   * @param pdvSeqV1_02 a lista de ProjectDataView na verso 1.2
   * 
   * @return a lista de ProjectDataView na verso 1.1
   * 
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         da DataView na verso 1.2 seja invlida
   * @throws InvalidDataKey caso a chave convertida para verso 1.1 seja
   *         invlida
   */
  static ProjectDataView[] convertProjectDataViewSeqToV1_01(
    tecgraf.openbus.data_service.project.v1_02.ProjectDataView[] pdvSeqV1_02)
    throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {

    ProjectDataView[] pdvSeqV1_01 = new ProjectDataView[pdvSeqV1_02.length];

    for (int i = 0; i < pdvSeqV1_02.length; i++) {
      pdvSeqV1_01[i] = convertProjectDataViewToV1_01(pdvSeqV1_02[i]);
    }

    return pdvSeqV1_01;
  }

  /**
   * Converte uma ProjectItemDataView da verso 1.2 para a verso 1.1.
   * 
   * @param pidvV1_02 a ProjectItemDataView na verso 1.2
   * 
   * @return a ProjectItemDataView na verso 1.1
   * 
   * @throws tecgraf.openbus.data_service.core.v1_02.InvalidDataKey caso a chave
   *         da DataView na verso 1.2 seja invlida
   * @throws InvalidDataKey caso a chave convertida para verso 1.1 seja
   *         invlida
   */
  private static ProjectItemDataView convertProjectItemDataViewToV1_01(
    tecgraf.openbus.data_service.project.v1_02.ProjectItemDataView pidvV1_02)
    throws InvalidDataKey,
    tecgraf.openbus.data_service.core.v1_02.InvalidDataKey {
    return new ProjectItemDataViewImpl(convertDataKeyToV1_01(pidvV1_02.fKey), //key 
      pidvV1_02.fOwner, //owner
      pidvV1_02.fDescription, //description
      pidvV1_02.fPath, //path
      pidvV1_02.fType, //type
      pidvV1_02.fSize, //size
      pidvV1_02.fIsContainer, //isContainer
      pidvV1_02.fCanRead, //canRead
      pidvV1_02.fCanWrite, //canWrite
      pidvV1_02.fCreationDate, //creationDate 
      pidvV1_02.fModificationDate//modificationDate
    );
  }

  /**
   * Converte o nome de uma viso de dados na verso 1.2 para verso 1.1.
   * 
   * @param interfaceNameV1_02 o nome da viso de dados na verso 1.2
   * 
   * @return o nome da viso de dados na verso 1.1
   */
  static String convertInterfaceNameToV1_01(String interfaceNameV1_02) {
    if (interfaceNameV1_02 == null) {
      return "";
    }

    if (interfaceNameV1_02.length() == 0) {
      return "";
    }

    if (interfaceNameV1_02
      .equals(tecgraf.openbus.data_service.core.v1_02.UnstructuredDataViewHelper
        .id())) {
      return UnstructuredDataViewHelper.id();
    }
    else if (interfaceNameV1_02
      .equals(tecgraf.openbus.data_service.hierarchical.v1_02.HierarchicalNodeDataViewHelper
        .id())) {
      return null;
    }
    else if (interfaceNameV1_02
      .equals(tecgraf.openbus.data_service.project.v1_02.ProjectDataViewHelper
        .id())) {
      return ProjectDataViewHelper.id();
    }
    else if (interfaceNameV1_02
      .equals(tecgraf.openbus.data_service.project.v1_02.ProjectItemDataViewHelper
        .id())) {
      return ProjectItemDataViewHelper.id();
    }

    return null;
  }

  /**
   * Converte o nome de uma lista de vises de dados na verso 1.2 para verso
   * 1.1.
   * 
   * @param interfaceNameSeqV1_02 a lista de nomes de vises de dados na verso
   *        1.2
   * 
   * @return a lista de nomes de vises de dados na verso 1.1
   */
  static String[] convertInterfaceNameSeqToV1_01(String[] interfaceNameSeqV1_02) {
    List<String> insV1_01 = new LinkedList<String>();

    for (int i = 0; i < interfaceNameSeqV1_02.length; i++) {
      String interfaceNameV1_01 =
        convertInterfaceNameToV1_01(interfaceNameSeqV1_02[i]);
      if ((interfaceNameV1_01 != null) && (interfaceNameV1_01.length() > 0)) {
        insV1_01.add(interfaceNameV1_01);
      }
    }
    return insV1_01.toArray(new String[0]);
  }
}
