package csbase.server.services.schedulerservice;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import csbase.logic.CommandInfo;
import csbase.logic.SGASet;
import csbase.server.services.sgaservice.SGAService;

/**
 * Poltica de escalonamento que privilegia os SGAs com maior capacidade atual 
 * de processamento, medida obtida pela porcentagem de CPU disponvel juntamente
 * com a capacidade de processamento total da mquina quando no h carga de 
 * trabalho.
 * 
 * @author valeria
 *
 */
public class CPUCapacityPolicy implements SchedulerPolicyInterface {
  public String chooseServer(CommandInfo command, 
                             List<String> serversNames) {
    List<String> sgasNames;
    
    sgasNames = getSortedSGAByCPUCapacity(serversNames);
    
    if(sgasNames.size() != 0)
      return sgasNames.get(0);
    return null;
  }
  
  public List<String> chooseServers(CommandInfo command, 
                                    List<String> serversNames, 
                                    int numServers) {
    List<String> sgasNames;
    
    sgasNames = getSortedSGAByCPUCapacity(serversNames);
    
    if(sgasNames.size() >= numServers)
      return sgasNames.subList(0, numServers);
    return null;
  }
  
  /**
   * Ordena os SGAs de acordo com a capacidade de processamento atual dos SGAs, 
   * onde se considera o valor retornado do benchmark de CPU e a taxa de 
   * utilizao dessa CPU. 
   * 
   * @param serversNames Os servidores disponveis para execuo
   * 
   * @return Lista de servidores ordenados de forma decrescente por capacidade 
   *         atual de processamento
   */
  private List<String> getSortedSGAByCPUCapacity(List<String> serversNames) {
    List<SGASet> sgasSets = new ArrayList<SGASet>();
    List<String> sgasNames = new ArrayList<String>();
    SGAService sgaService = SGAService.getInstance();
    
    for(int i = 0; i < serversNames.size(); i++) {
      SGASet sgaSet = sgaService.getSGASet(serversNames.get(i));
      if(sgaSet != null)
        sgasSets.add(sgaSet);
    }

    Collections.sort(sgasSets, new SGASetCPUCapacityComparator());
    for (SGASet sga : sgasSets){
      sgasNames.add(sga.getName());
    }
    return sgasNames;
  }
}
