/*
 * PlatformIO.java
 * 
 * $Id: PlatformIO.java 115229 2011-02-03 19:57:31Z cassino $
 */
package csbase.server.services.administrationservice;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.Vector;

import csbase.logic.Platform;
import csbase.logic.PlatformInfo;
import csbase.server.Server;

/**
 * A classe <code>PlatformIO</code>  responsvel pela persistncia das
 * Plataformas de Execuo cadastradas no sistema. Nesta verso, os objetos que
 * representam as plataformas so serializados em um arquivo sequencial que 
 * regerado sempre que alguma plataforma  criada, modificada ou removida. Esta
 * implementao no utiliza a cache da classe <code>Platform</code>, mantendo,
 * ao invs disso, um espelho local do arquivo em disco. Note que essa opo
 * levou em considerao o fato de que o nmero de plataformas de execuo 
 * razoavelmente pequeno.
 */
public class PlatformIO {
  /** Propriedade do sistema que configura o nome do arquivo de plataformas */
  private static String FILENAME_PROPERTY = "PlatformIO.filename";
  /** Nome default do arquivo de plataformas */
  private static String FILENAME = "platform.dat";
  /** Nome do arquivo de UNs no existir. */
  private static String fileName = FILENAME;
  /** Arquivo para persistncia das plataformas */
  private File dataFile;

  /**
   * Persiste no arquivo em disco as plataformas armazenadas no espelho local.
   * 
   * @param platforms lista de plataformas armazenadas no espelho local
   * 
   * @throws IOException erro na gravao do arquivo
   */
  private synchronized void writeAll(Vector<Platform> platforms)
    throws IOException {
    PrintWriter out = new PrintWriter(new FileWriter(fileName));
    try {
      for (Platform platform : platforms) {
        out.println(platform.getId() + "\t" + platform.getName() + "\t"
          + platform.getDescription());
      }
    }
    finally {
      out.flush();
      out.close();
    }
  }

  /**
   * Obtm todas as plataformas cadastradas
   * 
   * @return vetor contendo as plataformas cadastradas
   * 
   * @throws IOException erro na leitura do arquivo
   */
  public synchronized List<Platform> readAll() throws IOException {
    List<Platform> allPlatforms = new Vector<Platform>();

    /*
     * Caso o arquivo de dados ainda no tenha sido criado, returnar uma lista
     * vazia de plataformas.
     */
    if (!dataFile.exists()) {
      return allPlatforms;
    }

    BufferedReader in = new BufferedReader(new FileReader(dataFile));
    String id;
    String name;
    String desc;
    String line;
    StringTokenizer st;
    try {
      while ((line = in.readLine()) != null) {
        st = new StringTokenizer(line, "\t");
        try {
          id = st.nextToken();
          name = st.nextToken();
          /* Descrio  opcional */
          desc = st.hasMoreTokens() ? st.nextToken() : "";

          allPlatforms.add(new Platform(new String(id), new PlatformInfo(name,
            desc)));
        }
        catch (NoSuchElementException nse) {
          throw new IOException(
            "Erro ao carregar a lista de plataformas: formato do arquivo invlido");
        }
      }
    }
    finally {
      in.close();
    }

    return allPlatforms;
  }

  /**
   * Obtm a plataforma que possui o identificador especificado
   * 
   * @param id identificador da plataforma procurada
   * 
   * @return plataforma procurada ou null caso essa plataforma no exista
   * 
   * @throws IOException erro na leitura do arquivo
   */
  public synchronized Platform read(Object id) throws IOException {
    try {
      for (Platform platform : Platform.getAllPlatforms()) {
        if (platform.getId().equals(id)) {
          return platform;
        }
      }
      return null;
    }
    catch (Exception e) {
      throw new IOException("Arquivo de plataformas corrompido");
    }
  }

  /**
   * Cria uma nova plataforma e grava no arquivo. Um identificador nico 
   * atribudo  nova plataforma.<br>
   * Na atual implementao, o prprio nome da plataforma  utilizado como
   * identificador. Note que a operao <code>createPlatform</code> da classe
   * <code>Platform</code> garante que no  possvel criar duas plataformas com
   * o mesmo nome.
   * 
   * @param info as informaes da nova plataforma
   * 
   * @return plataforma criada
   * 
   * @throws IOException
   */
  public synchronized Platform writeNew(PlatformInfo info) throws IOException {
    try {
      Vector<Platform> platforms =
        new Vector<Platform>(Platform.getAllPlatforms());
      /* Constri a nova plataforma, usando seu nome como identificador */
      Object id = info.name;
      Platform platform = new Platform(id, info);
      /* Insere a plataforma no espelho local e refaz o arquivo em disco */
      platforms.add(platform);
      writeAll(platforms);
      return platform;
    }
    catch (Exception e) {
      throw new IOException("Arquivo de plataformas corrompido");
    }
  }

  /**
   * Modifica uma plataforma e grava no arquivo. Note que a operao
   * <code>modifyPlatform</code> da classe <code>Platform</code> garante que no
   *  possvel alterar o nome da plataforma, o que  necessrio para podermos
   * usar este nome como identificador!
   * 
   * @param id o identificador da plataforma a ser modificada
   * @param info as novas informaes da plataforma
   * 
   * @return plataforma modificada
   * 
   * @throws IOException se a plataforma no est no arquivo ou se ocorrer algum
   *         erro de gravao
   */
  public synchronized Platform write(Object id, PlatformInfo info)
    throws IOException {
    try {
      Vector<Platform> platforms =
        new Vector<Platform>(Platform.getAllPlatforms());
      Platform platform = Platform.getPlatform(id);
      if (!platforms.remove(platform)) {
        throw new IOException("A plataforma no existe");
      }
      /* Constri a nova plataforma e retorna a mesma */
      platform = new Platform(id, info);
      platforms.add(platform);
      writeAll(platforms);

      return platform;
    }
    catch (Exception e) {
      throw new IOException(e.getMessage());
    }
  }

  /**
   * Remove uma plataforma do arquivo.
   * 
   * @param id o identificador da plataforma a ser removida
   * 
   * @throws IOException se a plataforma no est no arquivo ou se ocorrer algum
   *         erro de gravao
   */
  public synchronized void delete(Object id) throws IOException {
    try {
      Platform platform = Platform.getPlatform(id);
      Vector<Platform> platforms =
        new Vector<Platform>(Platform.getAllPlatforms());

      if (!platforms.remove(platform)) {
        throw new IOException("A plataforma no existe");
      }
      writeAll(platforms);
    }
    catch (Exception e) {
      throw new IOException(e.getMessage());
    }
  }

  /**
   * Mostra na tela o contedo do arquivo de dados.
   * 
   * @throws IOException
   */
  public static void printDataFile() throws IOException {
    try {
      PlatformIO pio = new PlatformIO();
      List<Platform> allPlatforms = pio.readAll();
      if (allPlatforms.size() == 0) {
        System.out.println("Arquivo de plataformas vazio");
      }
      else {
        for (Platform platform : allPlatforms) {
          System.out.println(platform.getId() + ": <" + platform.getName()
            + " , " + platform.getDescription() + ">");
        }
      }
    }
    catch (Exception e) {
      System.out.println(e);
    }
  }

  /**
   * Mtodo <code>main</code> para testes.
   * 
   * @param args
   * @throws IOException
   */
  public static void main(String[] args) throws IOException {
    printDataFile();
  }

  /**
   * Constri um objeto responsvel pela persistncia de Platformas.
   */
  public PlatformIO() {
    final Server server = Server.getInstance();
    AdministrationService srv = AdministrationService.getInstance();
    /* Obtm o nome do arquivo de plataformas */
    fileName = server.getPersistencyRootDirectoryName() + File.separator
        + srv.getStringProperty(FILENAME_PROPERTY);
    dataFile = new File(fileName);
    Server.logInfoMessage("Arquivo de plataformas: " + fileName);
  }
}
