package csbase.server.services.opendreamsservice.opendreams.v1_9;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.omg.CORBA.UserException;
import org.omg.PortableServer.POA;

import csbase.exception.PermissionException;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;
import tecgraf.openbus.DRMAA.v1_9.AuthorizationException;
import tecgraf.openbus.DRMAA.v1_9.InternalException;
import tecgraf.openbus.DRMAA.v1_9.InvalidJobException;
import tecgraf.openbus.DRMAA.v1_9.JobInfo;
import tecgraf.openbus.DRMAA.v1_9.JobState;
import tecgraf.openbus.DRMAA.v1_9.Session;
import tecgraf.openbus.DRMAA.v1_9.SessionHelper;
import tecgraf.openbus.opendreams.v1_9.FinalizationType;
import tecgraf.openbus.opendreams.v1_9.OpenDreamsJobInfo;
import tecgraf.openbus.opendreams.v1_9.OpenDreamsJobInfoImpl;
import tecgraf.openbus.opendreams.v1_9.ProgressData;
import tecgraf.openbus.opendreams.v1_9.ProjectNotFoundException;

/**
 * Implementa a idl <code>IOpenDreams</code>.
 * 
 * Recupera a sesso ativa associada a um determinado usurio e a um projeto
 * csbase. Se a sesso ainda no existir, cria a sesso.
 * 
 * @author Tecgraf PUC-Rio
 */
public class OpenDreams extends tecgraf.openbus.opendreams.v1_9.IOpenDreamsPOA {
  /**
   * Referncia para a sesso do opendreams
   */
  private Session session;
  /**
   * Referncia para o gerenciador das sesses
   */
  private SessionManager sessionManagement;
  /**
   * No definido.
   */
  public static final String UNDEFINED = "indefinido";

  /**
   * Constri o objeto que implementa a interface <code>IOpenDremas</code>.
   * 
   */
  public OpenDreams() {
    sessionManagement = new SessionManager();
  }

  /**
   * Obtm a sesso passando apenas o nome do projeto. O owner do projeto  o
   * usurio da credencial. {@inheritDoc}
   */
  @Override
  public Session getSession(String projectName) throws AuthorizationException,
    InternalException, ProjectNotFoundException {
    org.omg.CORBA.Object obj;
    String userId = UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreams: getSession (" + projectName + ")");
      checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      checkProject(projectName, userId); // Verifica se o projeto existe
      POA poa = OpenBusService.getInstance().getRootPOA();
      obj = poa.servant_to_reference(sessionManagement.getSession(userId,
        projectName));
      session = SessionHelper.narrow(obj);
    }
    catch (AuthorizationException e) {
      String msg = "Falha na tentativa de obter uma sesso do usurio " + userId
        + " para o projeto " + projectName + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ProjectNotFoundException e) {
      String msg = "Falha na tentativa de obter uma sesso do usurio " + userId
        + " para o projeto " + projectName + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg = "Erro ao obter a sesso do usurio " + userId
        + " para o projeto " + projectName;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
    return session;
  }

  /**
   * Obtm a sesso passando o nome do projeto e o usurio owner do projeto.
   * {@inheritDoc}
   * 
   * @return
   */
  @Override
  public Session getSessionByProjUserId(String projectName, String userName)
    throws AuthorizationException, InternalException, ProjectNotFoundException {
    org.omg.CORBA.Object obj;
    String userId = UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreams: getSessionByProjUserId (" + projectName
        + ", " + userName + ")");
      checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      checkProject(projectName, userName); // Verifica se o projeto existe
      checkProjectPermission(projectName, userName); // Verifica se o usurio da credencial tem acesso ao projeto
      POA poa = OpenBusService.getInstance().getRootPOA();
      obj = poa.servant_to_reference(sessionManagement.getSession(userName,
        projectName));
      session = SessionHelper.narrow(obj);
    }
    catch (AuthorizationException e) {
      String msg = "Falha na tentativa de obter uma sesso do usurio " + userId
        + " para o projeto " + projectName + " do usurio " + userName + ": "
        + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ProjectNotFoundException e) {
      String msg = "Falha na tentativa de obter uma sesso do usurio " + userId
        + " para o projeto " + projectName + " do usurio " + userName + ": "
        + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg = "Erro ao obter a sesso do usurio " + userId
        + " para o projeto " + projectName + " do usurio " + userName;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
    return session;
  }

  /**
   * Cria uma mensagem formatada com o texto de uma mensagem junto com
   * informaes mais detalhadas sobre a prpria exceo.
   * 
   * @param e a exceo que ocorreu
   * @param msg uma mensagem descritiva do contexto da exceo
   * @return um texto formatado
   */
  public static String formatMessage(Throwable e, String msg) {
    StringBuffer message = new StringBuffer(msg);
    message.append(": " + e.getClass().getName() + ": " + e.getMessage());
    message.append("\nCaused by:\n");
    StackTraceElement[] stackTrace = e.getStackTrace();
    for (int i = 0; i < stackTrace.length; i++) {
      message.append(stackTrace[i].toString() + "\n");
    }
    return message.toString();
  }

  /**
   * Verifica se o usurio  cadastrado no CSBase.
   * 
   * @param userId identificador do usurio
   * @throws AuthorizationException caso no exista um usurio com o
   *         identificador
   */
  public static void checkUser(String userId) throws AuthorizationException {
    if (AdministrationService.getInstance().getUser(userId) == null) {
      throw new AuthorizationException("O usurio " + userId + " no existe.");
    }
  }

  /**
   * Verifica se o usurio possui um projeto cujo nome  passado como parmetro.
   * 
   * @param projectName nome do projeto
   * @param userId identificador do usurio
   * @throws ProjectNotFoundException caso no exista um projeto com o
   *         identificador passado como parmetro
   */
  public static void checkProject(String projectName, String userId)
    throws ProjectNotFoundException {
    if (AdministrationService.getInstance().getUser(userId) == null) {
      throw new ProjectNotFoundException(
        "No existe um usurio com identificador " + userId);
    }
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = projectService.getProjectId(userId, projectName);
    if (!projectService.existsProject(projectId)) {
      throw new ProjectNotFoundException("O usurio " + userId
        + " no possui um projeto com nome " + projectName);
    }
  }

  /**
   * Verifica se o usurio da credencial possui acesso um projeto de um outro
   * usurio.
   * 
   * @param projectName nome do projeto
   * @param userId identificador do usurio dono do projeto
   * @throws AuthorizationException caso no exista um usurio com o
   *         identificador
   */
  public static void checkProjectPermission(String projectName, String userId)
    throws AuthorizationException {
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = projectService.getProjectId(userId, projectName);
    try {
      projectService.checkWritePermission(projectId);
    }
    catch (PermissionException e) {
      throw new AuthorizationException("O usurio " + Service.getUser().getId()
        + " no possui acesso de escrita no projeto " + projectName
        + " do usurio " + userId);
    }
  }

  /**
   * Registra no log uma mensagem de warning referente a uma exceo prevista na
   * API da IDL do OpenDreams.
   * 
   * @param msg a mensagem
   * @param exception a exceo
   */
  public static void logWarningException(String msg, UserException exception) {
    try {
      Field messageField = exception.getClass().getField("message");
      Server.logWarningMessage(msg + ": " + messageField.get(exception));
    }
    catch (Exception e) {
      Server.logSevereMessage(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobInfo[] allJobsByProject(String projectName, String userName)
    throws InternalException, ProjectNotFoundException, AuthorizationException {
    String userId = UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreams: getSessionByProjUserId (" + projectName
        + ", " + userName + ")");
      checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      checkProject(projectName, userName); // Verifica se o projeto existe

      ProjectService projectService = ProjectService.getInstance();
      Object projectId = projectService.getProjectId(userName, projectName);

      Set<CommandInfo> commandInfos = CommandPersistenceService.getInstance()
        .getCommandInfos(projectId);
      List<JobInfo> jobInfos = new ArrayList<JobInfo>();
      for (CommandInfo commandInfo : commandInfos) {
        jobInfos.add(getJobInfoByCommandInfo(commandInfo.getId(), commandInfo));
      }
      return jobInfos.toArray(new JobInfo[0]);

    }
    catch (AuthorizationException e) {
      String msg = "Falha na tentativa de obter os jobs do usurio " + userId
        + " para o projeto " + projectName + " do usurio " + userName + ": "
        + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ProjectNotFoundException e) {
      String msg = "Falha na tentativa de obter o projeto " + projectName
        + " do usurio " + userName + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg = "Erro ao obter os jobs do usurio " + userId
        + " para o projeto " + projectName + " do usurio " + userName;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobInfo getJobInfoByJobId(String projectName, String userName,
    String jobId) throws InvalidJobException, InternalException,
    AuthorizationException, ProjectNotFoundException {
    String userId = UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreams: getSessionByProjUserId (" + projectName
        + ", " + userName + ")");
      checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      checkProject(projectName, userName); // Verifica se o projeto existe

      ProjectService projectService = ProjectService.getInstance();
      Object projectId = projectService.getProjectId(userName, projectName);

      CommandInfo commandInfo = CommandPersistenceService.getInstance()
        .getCommandInfo(projectId, jobId);
      if (commandInfo == null) {
        throw new InvalidJobException("No foi encontrado um job com o nome: "
          + jobId);
      }
      return getJobInfoByCommandInfo(jobId, commandInfo);

    }
    catch (AuthorizationException e) {
      String msg = "Falha na tentativa de obter o job " + jobId + " do usurio "
        + userId + " para o projeto " + projectName + " do usurio " + userName
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ProjectNotFoundException e) {
      String msg = "Falha na tentativa de obter o projeto " + projectName
        + " do usurio " + userName + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobException e) {
      String msg = "Falha na tentativa de obter o job " + jobId + " do usurio "
        + userId + " para o projeto " + projectName + " do usurio " + userName
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg = "Erro ao obter o job " + jobId + " do usurio " + userId
        + " para o projeto " + projectName + " do usurio " + userName;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
  }

  /**
   * Obtm um JobInfo atravs de um CommandInfo.
   * 
   * @param jobId identificador do job.
   * @param commandInfo CommandInfo do qual extraimos as informaes para montar
   *        um JobInfo.
   * @return JobInfo com informaes pertinentes ao estado do comando.
   */
  private JobInfo getJobInfoByCommandInfo(String jobId,
    CommandInfo commandInfo) {
    OpenDreamsJobInfo jobInfo = new OpenDreamsJobInfoImpl();
    jobInfo.jobId = jobId;
    jobInfo.hasExited = false;
    jobInfo.wasAborted = false;
    jobInfo.finalizationType = FinalizationType.UNKNOWN;

    CommandStatus status = commandInfo.getStatus();
    if (status == CommandStatus.SCHEDULED) {
      jobInfo.status = JobState.QUEUED_ACTIVE;
    }
    if (status == CommandStatus.INIT || status == CommandStatus.UPLOADING
      || status == CommandStatus.EXECUTING
      || status == CommandStatus.DOWNLOADING) {
      jobInfo.status = JobState.RUNNING;
    }
    if (status == CommandStatus.FINISHED) {
      CommandFinalizationType finnalizationType = commandInfo
        .getFinalizationType();
      if (finnalizationType == CommandFinalizationType.UNKNOWN
        || finnalizationType == CommandFinalizationType.END
        || finnalizationType == CommandFinalizationType.SUCCESS) {
        jobInfo.status = JobState.DONE;
        jobInfo.hasExited = true;
      }
      else if (finnalizationType == CommandFinalizationType.EXECUTION_ERROR
        || finnalizationType == CommandFinalizationType.FAILED
        || finnalizationType == CommandFinalizationType.KILLED
        || finnalizationType == CommandFinalizationType.NO_EXIT_CODE) {
        jobInfo.status = JobState.FAILED;
        jobInfo.hasExited = true;
        if (finnalizationType == CommandFinalizationType.KILLED) {
          jobInfo.wasAborted = true;
        }
      }
    }
    jobInfo.status = JobState.UNDETERMINED;

    if (jobInfo.hasExited) {
      List<String[]> resourceUsage = new ArrayList<String[]>();
      if (commandInfo.getCpuTimeSec() != null) {
        resourceUsage.add(new String[] { "cpu_time", commandInfo.getCpuTimeSec()
          .toString() });
      }
      if (commandInfo.getWallTimeSec() != null) {
        resourceUsage.add(new String[] { "elapsed_time", commandInfo
          .getWallTimeSec().toString() });
      }
      if (commandInfo.getUserTimeSec() != null) {
        resourceUsage.add(new String[] { "user_time", commandInfo
          .getUserTimeSec().toString() });
      }

      CommandFinalizationInfo finalizationInfo = commandInfo
        .getFinalizationInfo();

      if (finalizationInfo == null) {
        jobInfo.exitStatus = JobInfoQueue.NO_EXIT_CODE;
      }
      else {

        if (finalizationInfo.getInfoType() == FinalizationInfoType.EXTENDED) {
          if (finalizationInfo instanceof ExtendedCommandFinalizationInfo) {
            ExtendedCommandFinalizationInfo o =
              (ExtendedCommandFinalizationInfo) finalizationInfo;
            jobInfo.guiltyNodeId = String.valueOf(o.getGuiltyNodeId());
          }
        }
        Integer exitCode = finalizationInfo.getExitCode();
        jobInfo.exitStatus = exitCode == null ? JobInfoQueue.NO_EXIT_CODE
          : exitCode;
        jobInfo.finalizationType = JobInfoQueue.getFinalizationType(
          finalizationInfo.getFinalizationType(), finalizationInfo
            .getFailureCause());
        jobInfo.description = finalizationInfo.getFailureCause() == null
          ? finalizationInfo.getFinalizationType().getDescription()
          : finalizationInfo.getFailureCause().getDescription();
      }

      if (resourceUsage.size() > 0) {
        jobInfo.resourceUsage = resourceUsage.toArray(new String[resourceUsage
          .size()][]);
      }
      jobInfo.exitStatus = finalizationInfo != null ? finalizationInfo
        .getExitCode() : JobInfoQueue.NO_EXIT_CODE;
      jobInfo.hasSignaled = false;
      jobInfo.terminatingSignal = "";
      jobInfo.hasCoreDump = false;
    }

    jobInfo.progressData = new ProgressData();
    return jobInfo;
  }

}
