/*
 * $Id: CommandPropertyParser.java 135841 2012-11-27 19:09:32Z mjulia $
 */
package csbase.server.services.commandpersistenceservice;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import csbase.exception.ParseException;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandStatus;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.logic.FailureFinalizationType;
import csbase.logic.Priority;
import csbase.logic.SimpleCommandFinalizationInfo;
import csbase.logic.algorithms.ExecutionType;

/**
 * <p>
 * Analisador para arquivos que descrevem as propriedades de um comando.
 * </p>
 * 
 * <p>
 * Esta classe  responsvel por ler e escrever arquivos de propridades de
 * comando. O arquivo de propriedades de comando  um {@link Properties arquivo
 * properties padro para Java} com codificao {@value #ENCODING}.
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 */

public final class CommandPropertyParser {
  /**
   * A codificao do arquivo de propriedades.
   */
  private static final String ENCODING = "ISO-8859-1";
  /**
   * A chave da propriedade nmero de execues para a execuo mltipla do
   * comando.
   */
  private static final String PROPERTY_KEY_OF_EXECUTION_COUNT_FOR_MULTIPLE_EXECUTION =
    "execution_count_for_multiple_execution";
  /**
   * A chave da propriedade nmero de execues por sga para a execuo mltipla
   * do comando.
   */
  private static final String PROPERTY_KEY_OF_EXECUTION_COUNT_PER_SGA_FOR_MULTIPLE_EXECUTION =
    "execution_count_per_sga_for_multiple_execution";
  /**
   * A chave da propriedade descrio do comando.
   */
  private static final String PROPERTY_KEY_OF_DESCRIPTION = "description";
  /**
   * A chave da propriedade tipo de execuo do comando.
   */
  private static final String PROPERTY_KEY_OF_EXECUTION_TYPE = "execution_type";
  /**
   * A chave da propriedade dica do algoritmo.
   */
  private static final String PROPERTY_KEY_OF_TIP = "tip";
  /**
   * A chave da propriedade posio global do comando.
   */
  private static final String PROPERTY_KEY_OF_GLOBAL_POSITION =
    "global_position";
  /**
   * A chave da propriedade posio global do comando.
   */
  private static final int PROPERTY_DEFAULT_VALUE_OF_GLOBAL_POSITION = -1;
  /**
   * A chave da propriedade identificador do comando.
   */
  private static final String PROPERTY_KEY_OF_ID = "id";
  /**
   * A chave da propriedade que indica se a execuo do comando  automtica.
   */
  private static final String PROPERTY_KEY_OF_IS_AUTOMATIC = "is_automatic";
  /**
   * A chave da propriedade que indica se um email deve ser enviado ao final da
   * execuo do comando.
   */
  private static final String PROPERTY_KEY_OF_MAIL_AT_END = "mail_at_end";
  /**
   * A chave da propriedade que indica a restrio de plataformas de execuo.
   */
  private static final String PROPERTY_KEY_OF_PLATFORM_FILTER =
    "platform_filter";
  /**
   * A chave da propriedade prioridade do comando.
   */
  private static final String PROPERTY_KEY_OF_PRIORITY = "priority";
  /**
   * A chave da propriedade que indica os nomes dos sgas selecionados para o
   * comando.
   */
  private static final String PROPERTY_KEY_OF_SELECTED_SGAS_NAMES =
    "selected_sgas_names";
  /**
   * O separador usado pela propriedade
   * {@link #PROPERTY_KEY_OF_SELECTED_SGAS_NAMES}.
   */
  private static final String SGAS_NAMES_SEPARATOR = ",";
  /**
   * A chave da propriedade nome do SGA.
   */
  private static final String PROPERTY_KEY_OF_SGA_NAME = "sgaName";
  /**
   * A chave da propriedade estado do comando.
   */
  private static final String PROPERTY_KEY_OF_STATUS = "status";
  /**
   * A chave da propriedade que indica o que levou o comando a terminar.
   */
  private static final String PROPERTY_KEY_OF_FINALIZATION_TYPE =
    "finalization_type";
  /**
   * A chave da propriedade que descreve a causa da falha na finalizao do
   * comando.
   */
  private static final String PROPERTY_KEY_OF_FINALIZATION_FAILURE_TYPE =
    "finalization_failure_cause";
  /**
   * A chave da propriedade data e hora iniciais de execuo do comando.
   */
  private static final String PROPERTY_KEY_OF_SUBMITTED_DATE = "startDate";
  /**
   * A chave da propriedade do tempo total de execuo do processo principal.
   */
  private static final String PROPERTY_KEY_OF_WALL_TIME_SEC = "wallTimeSec";
  /**
   * A chave da propriedade identificador do usurio.
   */
  private static final String PROPERTY_KEY_OF_USER_ID = "userId";
  /**
   * A chave da propriedade cdigo de sada.
   */
  private static final String PROPERTY_KEY_OF_EXIT_CODE = "command_exit_code";
  /** A chave da propriedade informao de finalizao de comando. */
  private static final String PROPERTY_KEY_OF_EXTENDED_FINALIZATION_INFO =
    "extended_finalization_info";
  /**
   * A chave da propriedade n "culpado" pelo erro de execuo (se houver) de um
   * comando do tipo fluxo.
   */
  private static final String PROPERTY_KEY_OF_GUILTY_NODE_ID = "guilty_node_id";
  /** A chave da propriedade tipo de informao de finalizao de comando. */
  private static final String PROPERTY_KEY_OF_FINALIZATION_INFO_TYPE =
    "finalization_info_type";
  /**
   * O separador usado pela propriedade
   * {@link #PROPERTY_KEY_OF_EXTENDED_FINALIZATION_INFO} para dividir as
   * informaes de finalizao individuais dos ns de um fluxo.
   */
  private static final String FINALIZATION_INFO_SEPARATOR = ",";
  /**
   * O separador usado pela propriedade
   * {@link #PROPERTY_KEY_OF_EXTENDED_FINALIZATION_INFO} para dividir os campos
   * da informao de finalizao individual de um n de um fluxo.
   */
  private static final String FINALIZATION_INFO_FIELD_SEPARATOR = ":";

  /**
   * Chaves auxiliares dos mtodos createProperty e saveProperties.
   */
  private static final String PROPERTY_KEY = "key";

  /**
   * Chaves auxiliares dos mtodos createProperty e saveProperties.
   */
  private static final String PROPERTY_VALUE = "value";

  /**
   * Chaves auxiliares dos mtodos createProperty e saveProperties.
   */
  private static final String PROPERTY_COMMENT = "comment";

  /**
   * O properties utilizado para ler o arquivo.
   */
  private Properties properties;

  /**
   * As informaes de comando que esto sendo lidas ou escritas.
   */
  private List<Map<String, String>> propertiesValues;

  /**
   * Cria um analisador para ler ou escrever um arquivo com as propriedades do
   * comando. As propriedades ficam guardadas no analisador.
   */
  public CommandPropertyParser() {
    propertiesValues = new LinkedList<Map<String, String>>();
  }

  /**
   * Obtm uma mensagem no mecanismo de internacionalizao.
   * 
   * @param keySuffix O sufixo da chave (o nome da classe ser pr-fixado ao
   *        sufixo).
   * 
   * @return A mensagem.
   */
  private String getMessage(String keySuffix) {
    return CommandPersistenceService.getInstance().getString(
      getClass().getSimpleName() + "." + keySuffix);
  }

  /**
   * <p>
   * Cria uma tabela associativa que descreve a propriedade de comando.
   * </p>
   * 
   * <p>
   * As chaves deste mapa so: {@value #PROPERTY_KEY}, {@value #PROPERTY_VALUE}
   * e {@value #PROPERTY_COMMENT}. Todas as chaves so obrigatrias. O valor da
   * chave {@value #PROPERTY_KEY}  a chave da propriedade. O valor da chave
   * {@value #PROPERTY_VALUE}  o valor da propriedade. O valor da chave
   * {@value #PROPERTY_COMMENT}  um comentrio para a propriedade.
   * </p>
   * 
   * @param propertyKey A chave da propriedade (No aceita {@code null}).
   * @param propertyValue O valor da propriedade (No aceita {@code null}).
   * @param propertyComment Um comentrio para a propriedade (No aceita
   *        {@code null}).
   * 
   * @return a tabela associativa.
   */
  private Map<String, String> createProperty(String propertyKey,
    String propertyValue, String propertyComment) {
    Map<String, String> property = new HashMap<String, String>();
    property.put(PROPERTY_KEY, propertyKey);
    property.put(PROPERTY_VALUE, propertyValue);
    property.put(PROPERTY_COMMENT, propertyComment);
    return property;
  }

  /**
   * L uma propriedade especfica do tipo inteiro.
   * 
   * @param propertyName a chave da propriedade que se deseja ler.
   * @param defaultValue o valor da propriedade caso ela no esteja no arquivo.
   * @param isMandatory indicador de que a propriedade  obrigatria, ou seja,
   *        se no existir e no tiver valor padro, deve ser gerar um erro.
   * 
   * @return a propriedade.
   * 
   * @throws ParseException Em caso de erro no formato do nmero lido ou se a
   *         propriedade no tiver valor default e no existir.
   */
  private Integer readIntegerProperty(String propertyName,
    Integer defaultValue, boolean isMandatory) throws ParseException {
    String property = properties.getProperty(propertyName);
    if (property == null) {
      if (defaultValue == null && isMandatory) {
        throw new ParseException(String.format(
          getMessage("error_property_not_found"), propertyName));
      }
      return defaultValue;
    }
    try {
      return Integer.parseInt(property);
    }
    catch (NumberFormatException e) {
      String errorMessage =
        String.format(getMessage("error_must_be_integer_property"),
          propertyName, property);
      throw new ParseException(errorMessage);
    }
  }

  /**
   * L uma propriedade obrigatria especfica do tipo inteiro.
   * 
   * @param propertyName a chave da propriedade que se deseja ler.
   * @param defaultValue o valor da propriedade caso ela no esteja no arquivo.
   * 
   * @return a propriedade.
   * 
   * @throws ParseException Em caso de erro no formato do nmero lido ou se a
   *         propriedade no tiver valor default e no existir.
   */
  private Integer readIntegerProperty(String propertyName, Integer defaultValue)
    throws ParseException {
    return readIntegerProperty(propertyName, defaultValue, true);
  }

  /**
   * L uma propriedade especfica do tipo boolean.
   * 
   * @param propertyName a chave da propriedade que se deseja ler.
   * @param defaultValue o valor da propriedade caso ela no esteja no arquivo.
   * @param isMandatory indicador de que a propriedade  obrigatria, ou seja,
   *        se no existir e no tiver valor padro, deve ser gerar um erro.
   * 
   * @return a propriedade.
   * 
   * @throws ParseException Se a propriedade no tiver valor default e no
   *         existir.
   */
  private Boolean readBooleanProperty(String propertyName,
    Boolean defaultValue, boolean isMandatory) throws ParseException {
    String property = properties.getProperty(propertyName);
    if (property == null) {
      if (defaultValue == null && isMandatory) {
        throw new ParseException(String.format(
          getMessage("error_property_not_found"), propertyName));
      }
      return defaultValue;
    }
    return Boolean.parseBoolean(property);
  }

  /**
   * L uma propriedade obrigatria especfica do tipo boolean.
   * 
   * @param propertyName a chave da propriedade que se deseja ler.
   * @param defaultValue o valor da propriedade caso ela no esteja no arquivo.
   * 
   * @return a propriedade.
   * 
   * @throws ParseException Se a propriedade no tiver valor default e no
   *         existir.
   */
  private Boolean readBooleanProperty(String propertyName, Boolean defaultValue)
    throws ParseException {
    return readBooleanProperty(propertyName, defaultValue, true);
  }

  /**
   * L uma propriedade obrigatria especfica do tipo enumerao.
   * 
   * @param <E> Tipo da classe que implementa a enumerao.
   * 
   * @param propertyName a chave da propriedade que se deseja ler.
   * @param enumClass Classe da enumerao.
   * @param defaultValue o valor da propriedade caso ela no esteja no arquivo.
   * 
   * @return a propriedade.
   * 
   * @throws ParseException Se a propriedade no tiver valor default e no
   *         existir.
   */
  private <E extends Enum<E>> E readEnumProperty(String propertyName,
    Class<E> enumClass, E defaultValue) throws ParseException {
    return readEnumProperty(propertyName, enumClass, defaultValue, false);
  }

  /**
   * L uma propriedade especfica do tipo enumerao.
   * 
   * @param <E> Tipo da classe que implementa a enumerao.
   * 
   * @param propertyName a chave da propriedade que se deseja ler.
   * @param enumClass Classe da enumerao.
   * @param defaultValue o valor da propriedade caso ela no esteja no arquivo.
   * @param isMandatory indicador de que a propriedade  obrigatria, ou seja,
   *        se no existir e no tiver valor padro, deve ser gerar um erro.
   * 
   * @return a propriedade.
   * 
   * @throws ParseException Se a propriedade no tiver valor default e no
   *         existir.
   */
  private <E extends Enum<E>> E readEnumProperty(String propertyName,
    Class<E> enumClass, E defaultValue, boolean isMandatory)
    throws ParseException {
    String property = properties.getProperty(propertyName);
    if (property == null) {
      if (defaultValue == null && isMandatory) {
        throw new ParseException(String.format(
          getMessage("error_property_not_found"), propertyName));
      }
      else {
        return defaultValue;
      }
    }
    try {
      return Enum.valueOf(enumClass, property.trim());
    }
    catch (NullPointerException e) {
      String errorMessage =
        String.format(getMessage("error_must_be_enum_property"), propertyName,
          enumClass, property);
      throw new ParseException(errorMessage);
    }
    catch (IllegalArgumentException e) {
      String errorMessage =
        String.format(getMessage("error_must_be_enum_property"), propertyName,
          enumClass, property);
      throw new ParseException(errorMessage);
    }
  }

  /**
   * Cria a propriedade que informa o nmero de execues para a execuo
   * mltipla e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_EXECUTION_COUNT_FOR_MULTIPLE_EXECUTION} 
   * obrigatria quando o tipo de execuo for mltipla e a selao de sgas for
   * automtica. Seu valor  o nmero de nmero de execues que devem ser
   * disparadas na execuo do comando.
   * 
   * @param executionCount O nmero de nmero de execues que devem ser
   *        disparadas na execuo do comando. (No aceita {@code null})
   */
  public void setExecutionCountForMultipleExecution(Integer executionCount) {
    Map<String, String> executionCountProperty =
      createProperty(PROPERTY_KEY_OF_EXECUTION_COUNT_FOR_MULTIPLE_EXECUTION,
        Integer.toString(executionCount),
        getMessage("msg_execution_count_for_multiple_execution_commentary"));
    propertiesValues.add(executionCountProperty);
  }

  /**
   * L o nmero de o nmero de execues para a execuo mltipla. Esta
   * propriedade  obrigatria quando o tipo de execuo for mltipla e a
   * selao de sgas for automtica.
   * 
   * @return o nmero de o nmero de execues para a execuo mltipla.
   * 
   * @throws ParseException Em caso de erro no formato do nmero de execues ou
   *         se a propriedade no existir.
   */
  public int getExecutionCountForMultipleExecution() throws ParseException {
    return readIntegerProperty(
      PROPERTY_KEY_OF_EXECUTION_COUNT_FOR_MULTIPLE_EXECUTION, null);
  }

  /**
   * Cria a propriedade que informa o nmero de execues por sga para a
   * execuo mltipla e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_EXECUTION_COUNT_PER_SGA_FOR_MULTIPLE_EXECUTION} 
   * obrigatria quando o tipo de execuo  mltipla e a seleo de sgas 
   * manual. Seu valor  o nmero de execues por sgas que devem ser disparadas
   * na execuo do comando.
   * 
   * @param executionCount O nmero de execues por sgas que devem ser
   *        disparadas na execuo do comando. (No aceita {@code null})
   */
  public void setExecutionCountPerSGAForMultipleExecution(Integer executionCount) {
    Map<String, String> executionCountProperty =
      createProperty(
        PROPERTY_KEY_OF_EXECUTION_COUNT_PER_SGA_FOR_MULTIPLE_EXECUTION,
        Integer.toString(executionCount),
        getMessage("msg_execution_count_per_sga_for_multiple_execution_commentary"));
    propertiesValues.add(executionCountProperty);
  }

  /**
   * L o nmero de o nmero de execues por sga para a execuo mltipla. Esta
   * propriedade  obrigatria quando o tipo de execuo  mltipla e a seleo
   * de sgas  manual.
   * 
   * @return o nmero de execues por sga para a execuo mltipla.
   * 
   * @throws ParseException Em caso de erro no formato do nmero de execues
   *         por sga ou se a propriedade no existir.
   */
  public int getExecutionCountPerSGAForMultipleExecution()
    throws ParseException {
    return readIntegerProperty(
      PROPERTY_KEY_OF_EXECUTION_COUNT_PER_SGA_FOR_MULTIPLE_EXECUTION, null);
  }

  /**
   * Cria a propriedade que informa a descrio do comando e a anexa  lista de
   * propriedades. A chave {@value #PROPERTY_KEY_OF_DESCRIPTION}  opcional e o
   * seu valor  a descrio do comando.
   * 
   * @param description A descrio do comando. (No aceita {@code null})
   */
  public void setDescription(String description) {
    if (description != null) {
      Map<String, String> descriptionProperty =
        createProperty(PROPERTY_KEY_OF_DESCRIPTION, description,
          getMessage("msg_description_commentary"));
      propertiesValues.add(descriptionProperty);
    }
  }

  /**
   * L a descrio do comando.
   * 
   * @return A descrio do comando. Retorna {@code null} se ele no estiver nas
   *         propriedades do comando.
   */
  public String getDescription() {
    return properties.getProperty(PROPERTY_KEY_OF_DESCRIPTION);
  }

  /**
   * Cria a propriedade que informa o tipo de execuo do comando e a anexa 
   * lista de propriedades. A chave {@value #PROPERTY_KEY_OF_EXECUTION_TYPE} 
   * obrigatria e o seu valor  um nome que representa um {@link ExecutionType}
   * .
   * 
   * @param executionType O tipo de execuo do comando (No aceita {@code null}
   *        ).
   */
  public void setExecutionType(ExecutionType executionType) {
    Map<String, String> executionTypeProperty =
      createProperty(PROPERTY_KEY_OF_EXECUTION_TYPE, executionType.name(),
        getMessage("msg_execution_type_commentary"));
    propertiesValues.add(executionTypeProperty);
  }

  /**
   * L o tipo de execuo do comando.
   * 
   * @return o tipo de execuo do comando ou {@code ExecutionType#SIMPLE} se a
   *         propriedade no existir.
   * 
   * @throws ParseException Em caso de erro no formato do tipo de execuo.
   */
  public ExecutionType getExecutionType() throws ParseException {
    return readEnumProperty(PROPERTY_KEY_OF_EXECUTION_TYPE,
      ExecutionType.class, ExecutionType.SIMPLE);
  }

  /**
   * Cria a propriedade que informa a posio do comando na fila do escalonador
   * e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_GLOBAL_POSITION}  obrigatria e o seu valor  a
   * posio do comando na fila do escalonador.
   * 
   * @param globalPosition A posio do comando na fila do escalonador. (No
   *        aceita {@code null}).
   */
  public void setGlobalPosition(Integer globalPosition) {
    Map<String, String> globalPositionProperty =
      createProperty(PROPERTY_KEY_OF_GLOBAL_POSITION,
        Integer.toString(globalPosition),
        getMessage("msg_global_position_commentary"));
    propertiesValues.add(globalPositionProperty);

  }

  /**
   * L a posio do comando na fila do escalonador.
   * 
   * @return a posio do comando na fila do escalonador ou
   *         {@value #PROPERTY_DEFAULT_VALUE_OF_GLOBAL_POSITION} se a
   *         propriedade no existir.
   * 
   * @throws ParseException Em caso de erro no formato da posio do comando.
   */
  public Integer getGlobalPosition() throws ParseException {
    return readIntegerProperty(PROPERTY_KEY_OF_GLOBAL_POSITION,
      PROPERTY_DEFAULT_VALUE_OF_GLOBAL_POSITION);
  }

  /**
   * Cria a propriedade que informa o identificador do comando e a anexa  lista
   * de propriedades. A chave {@value #PROPERTY_KEY_OF_ID}  obrigatria e o seu
   * valor  o identificador do comando.
   * 
   * @param id O identificador do comando. (No aceita {@code null}).
   */
  public void setId(String id) {
    Map<String, String> idProperty =
      createProperty(PROPERTY_KEY_OF_ID, id, getMessage("msg_id_commentary"));
    propertiesValues.add(idProperty);
  }

  /**
   * L o identificador do comando. Esta propriedade  obrigatria.
   * 
   * @return O identificador do comando.
   * 
   * @throws ParseException Se o identificador do comando no estiver nas
   *         propriedades.
   */
  public String getId() throws ParseException {
    String id = properties.getProperty(PROPERTY_KEY_OF_ID);
    if (id == null) {
      String errorMessage =
        String.format(getMessage("error_property_not_found"),
          PROPERTY_KEY_OF_ID);
      throw new ParseException(errorMessage);
    }
    return id;
  }

  /**
   * Cria a propriedade que informa se a seleo de sgas  automtica e a anexa
   *  lista de propriedades. A chave {@value #PROPERTY_KEY_OF_IS_AUTOMATIC} 
   * obrigatria e o seu valor  verdadeiro caso a seleo seja automtica ou
   * falso caso contrrio.
   * 
   * @param isAutomatic verdadeiro caso a seleo seja automtica ou falso caso
   *        contrrio.
   */
  public void setIsAutomatic(boolean isAutomatic) {
    Map<String, String> isAutomaticProperty =
      createProperty(PROPERTY_KEY_OF_IS_AUTOMATIC,
        Boolean.toString(isAutomatic),
        getMessage("msg_is_automatic_commentary"));
    propertiesValues.add(isAutomaticProperty);
  }

  /**
   * L se a seleo de sgas  automtica.
   * 
   * @return verdadeiro caso a seleo seja automtica ou falso caso contrrio.
   * 
   * @throws ParseException Se a indicao de seleo de sgas no estiver no
   *         formato correto.
   */
  public boolean getIsAutomatic() throws ParseException {
    return readBooleanProperty(PROPERTY_KEY_OF_IS_AUTOMATIC, true);
  }

  /**
   * Cria a propriedade que informa se um email deve ser enviado ao usurio ao
   * final da execuo do comando e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_MAIL_AT_END}  obrigatria e o seu valor 
   * verdadeiro caso o email deva ser enviado ou falso caso contrrio.
   * 
   * @param mailAtEnd verdadeiro caso o email deva ser enviado ou falso caso
   *        contrrio.
   */
  public void setMailAtEnd(boolean mailAtEnd) {
    Map<String, String> mailAtEndProperty =
      createProperty(PROPERTY_KEY_OF_MAIL_AT_END, Boolean.toString(mailAtEnd),
        getMessage("msg_mail_at_end_commentary"));
    propertiesValues.add(mailAtEndProperty);
  }

  /**
   * L se um email deve ser enviado ao usurio ao final da execuo do comando.
   * 
   * @return verdadeiro caso o email deva ser enviado ou falso caso contrrio.
   * 
   * @throws ParseException Se a indicao de envio de email no estiver no
   *         formato correto.
   */
  public boolean getMailAtEnd() throws ParseException {
    return readBooleanProperty(PROPERTY_KEY_OF_MAIL_AT_END, false);
  }

  /**
   * Cria a propriedade que informa uma restrio de plataforma do comando e a
   * anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_PLATFORM_FILTER}  opcional e o seu valor  a
   * restrio de plataforma do comando.
   * 
   * @param platformFilter A restrio de plataforma do comando. (No aceita
   *        {@code null})
   */

  public void setPlatformFilter(String platformFilter) {
    if (platformFilter != null) {
      Map<String, String> platformFilterProperty =
        createProperty(PROPERTY_KEY_OF_PLATFORM_FILTER, platformFilter,
          getMessage("msg_platform_filter_commentary"));
      propertiesValues.add(platformFilterProperty);
    }
  }

  /**
   * L a restrio de plataforma do comando.
   * 
   * @return A restrio de plataforma do comando. Retorna {@code null} se ele
   *         no estiver nas propriedades do comando.
   */
  public String getPlatformFilter() {
    return properties.getProperty(PROPERTY_KEY_OF_PLATFORM_FILTER);
  }

  /**
   * Cria a propriedade que informa a prioridade de escalonamento do comando e a
   * anexa  lista de propriedades. A chave {@value #PROPERTY_KEY_OF_PRIORITY} 
   * obrigatria e o seu valor  um nome que representa um {@link Priority}.
   * 
   * @param priority Um nome que representa um {@link Priority}. (No aceita
   *        {@code null}).
   */
  public void setPriority(Priority priority) {
    if (null == priority) {
      return;
    }
    Map<String, String> priorityProperty =
      createProperty(PROPERTY_KEY_OF_PRIORITY, priority.name(),
        getMessage("msg_priority_commentary"));
    propertiesValues.add(priorityProperty);
  }

  /**
   * L a prioridade de escalonamento do comando.
   * 
   * @return a prioridade de escalonamento do comando ou {@code null} se a
   *         propriedade no existir.
   * 
   * @throws ParseException Em caso de erro no formato da prioridade.
   */
  public Priority getPriority() throws ParseException {
    return readEnumProperty(PROPERTY_KEY_OF_PRIORITY, Priority.class, null,
      false);
  }

  /**
   * Cria a propriedade que informa os sgas selecionados e a anexa  lista de
   * propriedades. A chave {@value #PROPERTY_KEY_OF_SELECTED_SGAS_NAMES} 
   * obrigatria quando a seleo de sgas for manual. Seu valor  uma lista com
   * os sgas selecionados separados por vrgulas.
   * 
   * @param sgasNames A lista de nomes dos sgas. (No aceita {@code null})
   */
  public void setSelectedSGAsNames(List<String> sgasNames) {
    String sgasNamesString = "";
    String sep = "";
    for (String sgaName : sgasNames) {
      sgasNamesString += sep + sgaName;
      sep = SGAS_NAMES_SEPARATOR;
    }
    Map<String, String> sgasNamesProperty =
      createProperty(PROPERTY_KEY_OF_SELECTED_SGAS_NAMES, sgasNamesString,
        getMessage("msg_selected_sgas_names_commentary"));
    propertiesValues.add(sgasNamesProperty);
  }

  /**
   * L um sga selecionado pelo usurio. Esta propriedade  obrigatria quando a
   * seleo de sgas for manual.
   * 
   * @return o sga selecionado pelo usurio.
   * 
   * @throws ParseException Em caso de erro no formato da lista lida ou se a
   *         propriedade no existir.
   */
  public String getSelectedSGAName() throws ParseException {
    List<String> selectedSGAsNames = getSelectedSGAsNames();
    if (selectedSGAsNames.size() != 1) {
      String errorMessage =
        String.format(getMessage("error_wrong_selected_sganame_format"),
          PROPERTY_KEY_OF_SELECTED_SGAS_NAMES, SGAS_NAMES_SEPARATOR);
      throw new ParseException(errorMessage);
    }
    return selectedSGAsNames.get(0);
  }

  /**
   * L os sgas selecionados pelo usurio. Esta propriedade  obrigatria quando
   * a seleo de sgas for manual.
   * 
   * @return a lista de sgas selecionados pelo usurio.
   * 
   * @throws ParseException Em caso de erro no formato da lista lida ou se a
   *         propriedade no existir.
   */
  public List<String> getSelectedSGAsNames() throws ParseException {
    String property =
      properties.getProperty(PROPERTY_KEY_OF_SELECTED_SGAS_NAMES);
    if (property == null) {
      throw new ParseException(String.format(
        getMessage("error_property_not_found"),
        PROPERTY_KEY_OF_SELECTED_SGAS_NAMES));
    }
    List<String> selectedSGAsNames = new ArrayList<String>();
    for (String sgaName : property.split(SGAS_NAMES_SEPARATOR)) {
      selectedSGAsNames.add(sgaName);
    }
    return selectedSGAsNames;
  }

  /**
   * Cria a propriedade que informa o nome do SGA que executou o comando (se
   * existir) e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_SGA_NAME}  opcional. Seu valor  o nome do SGA.
   * 
   * @param sgaName O nome do SGA. (No aceita {@code null}).
   */
  public void setSGAName(String sgaName) {
    if (sgaName != null) {
      Map<String, String> sgaNameProperty =
        createProperty(PROPERTY_KEY_OF_SGA_NAME, sgaName,
          getMessage("msg_sga_name_commentary"));
      propertiesValues.add(sgaNameProperty);
    }
  }

  /**
   * L o nome do SGA que executou o comando.
   * 
   * @return O nome do SGA ou {@code null} se ele no estiver nas propriedades
   *         do comando.
   */
  public String getSGAName() {
    return properties.getProperty(PROPERTY_KEY_OF_SGA_NAME);
  }

  /**
   * Cria a propriedade que informa o estado do comando e a anexa  lista de
   * propriedades. A chave {@value #PROPERTY_KEY_OF_PRIORITY}  obrigatria e o
   * seu valor  um nome que representa um {@link CommandStatus}.
   * 
   * @param status O estado de um comando. (No aceita {@code null}).
   */
  public void setStatus(CommandStatus status) {
    Map<String, String> statusProperty =
      createProperty(PROPERTY_KEY_OF_STATUS, status.name(),
        getMessage("msg_status_commentary"));
    propertiesValues.add(statusProperty);
  }

  /**
   * L a propriedade que informa o estado do comando.
   * 
   * @return o estado do comando ou {@code CommandStatus#FINISHED} se a
   *         propriedade no existir.
   * 
   * @throws ParseException Em caso de erro no formato do estado do comando.
   */
  public CommandStatus getStatus() throws ParseException {
    return readEnumProperty(PROPERTY_KEY_OF_STATUS, CommandStatus.class,
      CommandStatus.FINISHED);
  }

  /**
   * Cria a propriedade que informa o cdigo de sada do comando e a anexa 
   * lista de propriedades. A chave {@value #PROPERTY_KEY_OF_EXIT_CODE} 
   * opcional e o seu valor  um inteiro.
   * 
   * @param exitCode O cdigo de sada do comando. (No aceita {@code null} ).
   */
  protected void setExitCode(Integer exitCode) {
    if (exitCode != null) {
      Map<String, String> exitCodeProperty =
        createProperty(PROPERTY_KEY_OF_EXIT_CODE, String.valueOf(exitCode),
          getMessage("msg_exit_code_commentary"));
      propertiesValues.add(exitCodeProperty);
    }
  }

  /**
   * Cria as propriedades que informam sobre o estado de finalizao do comando
   * e as anexa  lista de propriedades. As propriedades dependem do tipo de
   * informao de finalizao disponvel para o comando.
   * 
   * @param finalizationInfo Informao de finalizao do comando. (No aceita
   *        {@code null} ).
   */
  public void setFinalizationInfo(CommandFinalizationInfo finalizationInfo) {
    if (finalizationInfo != null) {
      FinalizationInfoType infoType = finalizationInfo.getInfoType();
      if (infoType == FinalizationInfoType.EXTENDED) {
        setExtendedFinalizationInfo((ExtendedCommandFinalizationInfo) finalizationInfo);
      }
      else {
        setSimpleFinalizationInfo(finalizationInfo);
      }
    }
  }

  /**
   * Cria as propriedades que informam sobre o estado de finalizao do comando
   * e as anexa  lista de propriedades. Utilizadas para armazenar informaes
   * simples de um comando ({@code FinalizationInfoType#SIMPLE}).
   * 
   * @param finalizationInfo Informao simples de finalizao do comando. (No
   *        aceita {@code null} ).
   */
  protected void setSimpleFinalizationInfo(
    CommandFinalizationInfo finalizationInfo) {
    setFinalizationInfoType(finalizationInfo.getInfoType());
    setExitCode(finalizationInfo.getExitCode());
    setFinalizationType(finalizationInfo.getFinalizationType());
    setFailureCause(finalizationInfo.getFailureCause());
  }

  /**
   * Cria as propriedades que informam sobre o estado de finalizao do comando
   * e as anexa  lista de propriedades. Utilizadas para armazenar informaes
   * consolidadas de fluxos e seus ns em um comando. (
   * {@code FinalizationInfoType#EXTENDED}).
   * 
   * @param finalizationInfo Informao composta de finalizao do comando. (No
   *        aceita {@code null} ).
   */
  protected void setExtendedFinalizationInfo(
    ExtendedCommandFinalizationInfo finalizationInfo) {
    // Salva as informaes consolidadas do fluxo
    setSimpleFinalizationInfo(finalizationInfo);
    setGuiltyNodeId(finalizationInfo.getGuiltyNodeId());
    // Salva a informao de cada n
    Map<Integer, CommandFinalizationInfo> infos =
      finalizationInfo.getFinalizationInfos();
    if (infos != null && !infos.isEmpty()) {
      StringBuffer buffer = new StringBuffer();
      for (Entry<Integer, CommandFinalizationInfo> entry : infos.entrySet()) {
        CommandFinalizationInfo info = entry.getValue();
        buffer.append(entry.getKey());
        buffer.append(FINALIZATION_INFO_FIELD_SEPARATOR);
        Integer exitCode = info.getExitCode();
        if (exitCode != null) {
          buffer.append(exitCode);
        }
        buffer.append(FINALIZATION_INFO_FIELD_SEPARATOR);
        buffer.append(info.getFinalizationType());
        buffer.append(FINALIZATION_INFO_SEPARATOR);
      }
      if (buffer.length() > 0) {
        buffer.deleteCharAt(buffer.length() - 1); // remove o ltimo separador
        Map<String, String> exitCodeProperty =
          createProperty(PROPERTY_KEY_OF_EXTENDED_FINALIZATION_INFO,
            buffer.toString(), getMessage("msg_extended_finalization_info"));
        propertiesValues.add(exitCodeProperty);
      }
    }
  }

  /**
   * Cria a propriedade que informa o tipo de informao de finalizao
   * disponvel para o comando e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_FINALIZATION_INFO_TYPE}  obrigatria e o seu
   * valor  um nome que representa um {@link FinalizationInfoType} .
   * 
   * @param infoType O tipo de informao de finalizao de um comando. (No
   *        aceita {@code null}).
   */
  protected void setFinalizationInfoType(FinalizationInfoType infoType) {
    if (infoType != null) {
      Map<String, String> infoTypeProperty =
        createProperty(PROPERTY_KEY_OF_FINALIZATION_INFO_TYPE, infoType.name(),
          getMessage("msg_finalization_info_type_commentary"));
      propertiesValues.add(infoTypeProperty);
    }
  }

  /**
   * L a propriedade com o tipo de informao de finalizao disponvel para o
   * comando.
   * 
   * @return O tipo de informao ou {@code FinalizationInfoType#SIMPLE} se ele
   *         no est nas propriedades do comando.
   * 
   * @throws ParseException Em caso de erro no formato do tipo de informao.
   */
  public FinalizationInfoType getFinalizationInfoType() throws ParseException {
    return readEnumProperty(PROPERTY_KEY_OF_FINALIZATION_INFO_TYPE,
      FinalizationInfoType.class, FinalizationInfoType.SIMPLE);
  }

  /**
   * Cria a propriedade que identifica o n responsvel por erro na execuo do
   * fluxo (se houver). A chave {@value #PROPERTY_KEY_OF_GUILTY_NODE_ID} 
   * opcional e o seu valor  um inteiro que representa o id do n.
   * 
   * @param nodeId identificador do n responsvel por erro na execuo do
   *        fluxo, se houver. (No aceita {@code null}).
   */
  protected void setGuiltyNodeId(Integer nodeId) {
    if (nodeId != null) {
      Map<String, String> errorNodeIdProperty =
        createProperty(PROPERTY_KEY_OF_GUILTY_NODE_ID, String.valueOf(nodeId),
          getMessage("msg_guilty_node_id"));
      propertiesValues.add(errorNodeIdProperty);
    }
  }

  /**
   * L o identificador do n responsvel por erro na execuo do fluxo (se
   * houver).
   * 
   * @return O identificador do n ou {@code null} se ele no est nas
   *         propriedades do comando.
   * 
   * @throws ParseException Em caso de erro no formato do cdigo.
   */
  protected Integer getExitCode() throws ParseException {
    return readIntegerProperty(PROPERTY_KEY_OF_EXIT_CODE, null, false);
  }

  /**
   * L o cdigo de sada do comando.
   * 
   * @return O cdigo ou {@code null} se ele no est nas propriedades do
   *         comando.
   * 
   * @throws ParseException Em caso de erro no formato do cdigo.
   */
  protected Integer getGuiltyNodeId() throws ParseException {
    return readIntegerProperty(PROPERTY_KEY_OF_GUILTY_NODE_ID, null, false);
  }

  /**
   * L as propriedades que informam o estado finalizao do comando.
   * 
   * @return As informaes de finalizao do comando.
   * 
   * @throws ParseException Em caso de erro no formato das propriedades.
   */
  public CommandFinalizationInfo getFinalizationInfo() throws ParseException {
    FinalizationInfoType finalizationInfoType = getFinalizationInfoType();
    if (finalizationInfoType == FinalizationInfoType.EXTENDED) {
      return getExtendedFinalizationInfo();
    }
    else {
      return getSimpleFinalizationInfo();
    }
  }

  /**
   * L as propriedades que informam o estado finalizao do comando. Utilizadas
   * para ler informaes simples de um comando (
   * {@code FinalizationInfoType#SIMPLE} ).
   * 
   * @return informaes simples de finalizao do comando.
   * @throws ParseException Em caso de erro no formato das propriedades.
   */
  protected CommandFinalizationInfo getSimpleFinalizationInfo()
    throws ParseException {
    return new SimpleCommandFinalizationInfo(getFinalizationType(),
      getFailureCause(), getExitCode());
  }

  /**
   * L as propriedades que informam o estado finalizao do comando. Utilizadas
   * para ler informaes consolidadas de um fluxo e seus ns em um comando (
   * {@code FinalizationInfoType#EXTENDED} ).
   * 
   * @return informaes compostas de finalizao do comando.
   * @throws ParseException Em caso de erro no formato das propriedades.
   */
  protected ExtendedCommandFinalizationInfo getExtendedFinalizationInfo()
    throws ParseException {
    CommandFinalizationInfo simpleFinalizationInfo =
      getSimpleFinalizationInfo();
    ExtendedCommandFinalizationInfo extendedInfo =
      new ExtendedCommandFinalizationInfo(simpleFinalizationInfo);
    extendedInfo.setGuiltyNodeId(getGuiltyNodeId());
    String extendedInfoProperty =
      properties.getProperty(PROPERTY_KEY_OF_EXTENDED_FINALIZATION_INFO);
    if (extendedInfoProperty != null) {
      try {
        String[] exitCodes = extendedInfoProperty.split(",");
        for (String entry : exitCodes) {
          String[] splittedEntry = entry.split(":");
          if (splittedEntry.length == 3) {
            Integer id = Integer.parseInt(splittedEntry[0]);
            Integer exitCode = null;
            if (!splittedEntry[1].trim().isEmpty()) {
              exitCode = Integer.parseInt(splittedEntry[1]);
            }
            CommandFinalizationType type = CommandFinalizationType.UNKNOWN;
            if (!splittedEntry[2].trim().isEmpty()) {
              type =
                Enum.valueOf(CommandFinalizationType.class,
                  splittedEntry[2].trim());
            }
            SimpleCommandFinalizationInfo info =
              new SimpleCommandFinalizationInfo(type, exitCode);
            extendedInfo.setFinalizationInfoForNode(info, id);
          }
          else {
            String errorMessage =
              String
                .format(
                  getMessage("error_must_be_extended_finalization_info_property"),
                  PROPERTY_KEY_OF_EXTENDED_FINALIZATION_INFO,
                  extendedInfoProperty);
            throw new ParseException(errorMessage);
          }
        }
      }
      catch (NullPointerException e) {
      }
      catch (IllegalArgumentException e) {
        String errorMessage =
          String.format(
            getMessage("error_must_be_extended_finalization_info_property"),
            PROPERTY_KEY_OF_EXTENDED_FINALIZATION_INFO, extendedInfoProperty);
        throw new ParseException(errorMessage);
      }
    }
    return extendedInfo;
  }

  /**
   * Cria a propriedade que informa a causa da falha na finalizao do comando e
   * a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_FINALIZATION_FAILURE_TYPE}  opcional eo seu valor
   *  um nome que representa um {@link FailureFinalizationType}.
   * 
   * @param failureCause O que levou o comando a falhar (No aceita {@code null}
   *        ).
   */
  public void setFailureCause(FailureFinalizationType failureCause) {
    if (failureCause != null) {
      Map<String, String> descriptionProperty =
        createProperty(PROPERTY_KEY_OF_FINALIZATION_FAILURE_TYPE,
          failureCause.name(),
          getMessage("msg_finalization_failure_commentary"));
      propertiesValues.add(descriptionProperty);
    }
  }

  /**
   * Cria a propriedade que informa o que levou o comando a terminar e a anexa 
   * lista de propriedades. A chave {@value #PROPERTY_KEY_OF_FINALIZATION_TYPE}
   *  obrigatria e o seu valor  um nome que representa um
   * {@link CommandFinalizationType}.
   * 
   * @param finalizationType O que levou o comando a terminar. (No aceita
   *        {@code null}).
   */
  public void setFinalizationType(CommandFinalizationType finalizationType) {
    Map<String, String> statusProperty =
      createProperty(PROPERTY_KEY_OF_FINALIZATION_TYPE,
        finalizationType.name(), getMessage("msg_finalization_type_commentary"));
    propertiesValues.add(statusProperty);
  }

  /**
   * L a propriedade que informa o que levou o comando a terminar.
   * 
   * @return o comando a terminar ou {@code CommandFinalizationType#UNKNOWN} se
   *         a propriedade no existir.
   * 
   * @throws ParseException Em caso de erro no formato do estado do comando.
   */
  protected CommandFinalizationType getFinalizationType() throws ParseException {
    return readEnumProperty(PROPERTY_KEY_OF_FINALIZATION_TYPE,
      CommandFinalizationType.class, CommandFinalizationType.UNKNOWN);
  }

  /**
   * L a propriedade que informa a causa da falha na finalizao do comando.
   * 
   * @return a causa do comando terminar com falha ou
   *         {@code FailureFinalizationType#UNKNOWN} se a propriedade no
   *         existir.
   * @throws ParseException Em caso de erro no formato do estado do comando.
   */
  protected FailureFinalizationType getFailureCause() throws ParseException {
    return readEnumProperty(PROPERTY_KEY_OF_FINALIZATION_FAILURE_TYPE,
      FailureFinalizationType.class, null);
  }

  /**
   * Cria a propriedade que informa a data e hora em que o comando foi submetido
   * para execuo e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_SUBMITTED_DATE}  obrigatria e o seu valor  um
   * {@link Date}.
   * 
   * @param submittedDate A data em que o comando foi submetido. (No aceita
   *        {@code null}).
   */
  public void setSubmittedDate(Date submittedDate) {
    if (submittedDate != null) {
      long startTime = submittedDate.getTime();
      Map<String, String> submittedDateProperty =
        createProperty(PROPERTY_KEY_OF_SUBMITTED_DATE,
          Long.toString(submittedDate.getTime()),
          String.format(getMessage("msg_submitted_date_commentary"), startTime));
      propertiesValues.add(submittedDateProperty);
    }
  }

  /**
   * L a data em que o comando foi iniciado.
   * 
   * @return A data ou {@code null} se ela no est nas propriedades do comando.
   * 
   * @throws ParseException Em caso de erro no formato da data.
   */
  public Date getSubmittedDate() throws ParseException {
    String startDateProperty =
      properties.getProperty(PROPERTY_KEY_OF_SUBMITTED_DATE);
    if (startDateProperty != null) {
      try {
        return new Date(Long.parseLong(startDateProperty));
      }
      catch (NumberFormatException e) {
        String errorMessage =
          String.format(getMessage("error_must_be_integer_property"),
            PROPERTY_KEY_OF_SUBMITTED_DATE, startDateProperty);
        throw new ParseException(errorMessage);
      }
    }
    return null;
  }

  /**
   * Cria a propriedade que informa o tempo total de execuo do processo
   * principal e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_WALL_TIME_SEC}  obrigatria e o seu valor  um
   * {@link Integer}.
   * 
   * @param wallTimeSec tempo total de execuo do processo principal. (No
   *        aceita {@code null}).
   */
  public void setWallTimeSec(Integer wallTimeSec) {
    if (wallTimeSec != null) {
      Map<String, String> wallTimeSecProperty =
        createProperty(PROPERTY_KEY_OF_WALL_TIME_SEC,
          Integer.toString(wallTimeSec), getMessage("msg_wall_time_commentary"));
      propertiesValues.add(wallTimeSecProperty);
    }
  }

  /**
   * L o tempo total de execuo do processo principal.
   * 
   * @return O tempo ou {@code null} se ela no est nas propriedades do
   *         comando.
   * 
   * @throws ParseException Em caso de erro no formato da data.
   */
  public Integer getWallTimeSec() throws ParseException {
    String wallTimeSecProperty =
      properties.getProperty(PROPERTY_KEY_OF_WALL_TIME_SEC);
    if (wallTimeSecProperty != null) {
      try {
        return Integer.valueOf(wallTimeSecProperty);
      }
      catch (NumberFormatException e) {
        String errorMessage =
          String.format(getMessage("error_must_be_integer_property"),
            PROPERTY_KEY_OF_WALL_TIME_SEC, wallTimeSecProperty);
        throw new ParseException(errorMessage);
      }
    }
    return null;
  }

  /**
   * Cria a propriedade que informa a descrio do algoritmo e a anexa  lista
   * de propriedades. A chave {@value #PROPERTY_KEY_OF_TIP}  opcional e o seu
   * valor  a descrio do algoritmo.
   * 
   * @param tip A descrio do algoritmo. (No aceita {@code null})
   */
  public void setTip(String tip) {
    if (tip != null) {
      Map<String, String> tipProperty =
        createProperty(PROPERTY_KEY_OF_TIP, tip,
          getMessage("msg_tip_commentary"));
      propertiesValues.add(tipProperty);
    }
  }

  /**
   * L a descrio do algoritmo do comando.
   * 
   * @return A descrio do algoritmo do comando. Retorna {@code null} se ele
   *         no estiver nas propriedades do comando.
   */
  public String getTip() {
    return properties.getProperty(PROPERTY_KEY_OF_TIP);
  }

  /**
   * Cria a propriedade que informa o identificador do usurio que submeteu o
   * comando para execuo e a anexa  lista de propriedades. A chave
   * {@value #PROPERTY_KEY_OF_USER_ID}  obrigatria e o seu valor  o
   * identificador do usurio.
   * 
   * @param userId O identificador do usurio. (No aceita {@code null}).
   */
  public void setUserId(Object userId) {
    Map<String, String> userIdProperty =
      createProperty(PROPERTY_KEY_OF_USER_ID, (String) userId,
        getMessage("msg_user_id_commentary"));
    propertiesValues.add(userIdProperty);
  }

  /**
   * L o identificador do usurio que executou o comando.
   * 
   * @return O identificador do usurio ou {@code null} se ele no estiver nas
   *         propriedades do comando.
   */
  public String getUserId() {
    return properties.getProperty(PROPERTY_KEY_OF_USER_ID);
  }

  /**
   * Salva uma lista de propriedades no stream de sada.
   * 
   * @param outputStream O stream de sada (No aceita {@code null}).
   * 
   * @throws IOException Em caso de erro de ES ao escrever no stream.
   */
  public void saveProperties(OutputStream outputStream) throws IOException {
    for (Map<String, String> property : propertiesValues) {
      String propertyComment = property.get(PROPERTY_COMMENT);
      String propertyKey = property.get(PROPERTY_KEY);
      String propertyValue = property.get(PROPERTY_VALUE);
      String[] propertyCommentLines = propertyComment.split("\n");
      for (String propertyCommentLine : propertyCommentLines) {
        saveText(outputStream, String.format("# %s\n", propertyCommentLine));
      }
      saveText(outputStream,
        String.format("%s = %s\n", propertyKey, propertyValue));
    }
    outputStream.flush();
  }

  /**
   * Carrega uma lista de propriedades do stream de entrada.
   * 
   * @param inputStream O stream de entrada (No aceita {@code null}).
   * 
   * @throws IOException Em caso de erro de ES ao ler do stream.
   */
  public void loadProperties(InputStream inputStream) throws IOException {
    properties = new Properties();
    properties.load(inputStream);
  }

  /**
   * Salva um texto qualquer no stream de sada.
   * 
   * @param outputStream O stream de sada (No aceita {@code null}).
   * @param text O texto (No aceita {@code null}).
   * 
   * @throws IOException Em caso de erro de ES ao escrever no stream.
   */
  private void saveText(OutputStream outputStream, String text)
    throws IOException {
    byte[] bytes = text.getBytes(ENCODING);
    outputStream.write(bytes);
  }

}
