/**
 * $Id: Command.java 169784 2015-11-13 18:42:35Z isabella $
 */
package csbase.server.services.sgaservice;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;

import csbase.exception.OperationFailureException;
import csbase.exception.ServiceFailureException;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.logic.FailureFinalizationType;
import csbase.logic.SGAFile;
import csbase.logic.SGAInfo;
import csbase.logic.SimpleCommandFinalizationInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.CommandScript;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.FlowCommandLineContext;
import csbase.logic.algorithms.flows.FlowNode;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.flows.configurator.Input;
import csbase.logic.algorithms.flows.configurator.Node;
import csbase.logic.algorithms.flows.configurator.Output;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.eventlogservice.EventLogService;
import csbase.server.services.projectservice.ProjectService;
import sgaidl.CompletedCommandInfo;
import sgaidl.ExecutionInfo;
import sgaidl.ProcessInfo;
import sgaidl.ProcessState;
import sgaidl.RunningCommandInfo;
import sgaidl.SGACommand;
import sgaidl.SGAServer;
import tecgraf.javautils.core.io.FileUtils;

/**
 * A classe <code>Command</code> representa um comando que est sendo executado
 * no gerenciador de uma mquina hospedeira (ou servidor SGA).
 */
public class Command implements Serializable {

  /**
   * Arquivos e diretrios gerados pelo comando. Estes arquivos so previamente
   * criados pelo SSI e sinalizados como "em construo". Essa sinalizao 
   * removida no trmino do comando.
   */
  private SGAFile[] commandOutputFiles = null;
  /** Informaes oriundas do SGA a respeito do comand (this) em execuo. */
  private RunningCommandInfo info = null;
  /** Informaes para a execuo do comando. */
  private CommandInfo cmdInfo;
  /** Indica se as informaes do comando esto atualizadas. */
  private boolean newInfo = false;
  /**
   * Indica a data e hora em que as informaes do comando foram atualizadas
   * pela ltima vez.
   */
  private long lastInfoUpdate = 0;

  /** Servio de gerenciamento de SGAs. */
  private transient SGAService service = null;
  /** Caminho do diretrio do projeto deste comando. */
  private String[] projectPath = null;
  /** Caminho do repositrio do algoritmos deste comando. */
  private String[] algorithmRootPath = null;
  /** Caminho do repositrio de projetos deste comando. */
  private String[] projectRootPath = null;

  /** Indica a data e hora que o comando foi criado */
  private long startTimeMillis = 0;
  /** SGA que est executando o comando. */
  private transient SGA sga = null;
  /** Referncia remota para o comando em execuo. */
  private transient SGACommand remoteReference;
  /** Texto de comando executado no SGA. */
  private String command = null;
  /**
   * Contador de numero de updates (para fazer freqencia de update
   * incremental).
   */
  private long nupdates = 0;
  /** prazo de validade das informaes do comando. */
  private long expirationInfoDelay;

  //CSFS related stuff
  /**
   * Eventualmente, o comando pode ser executado sobre um sandbox no SGA. Isso
   * acontece quando a maquina de execucao nao possui acesso NFS aos dados e
   * binarios, e um sandbox deve ser criado.
   */
  private String[] sandbox = null; // o sandbox eh um diretorio que deve ser
  // montado usando o FileUtils
  /**
   * Lista de arquivos com as respectivas datas de modificacao, usado para saber
   * o estado do sandbox antes da execucao do comando
   */
  private Map<String[], Long> sandboxSnapshot = null;

  /**
   * SGA
   */
  private SGAInfo node;

  /**
   * Nome do n do SGA que deve executar o comando
   */
  private String hostName;

  /**
   * Indica se o comando deve ser executado como um script.
   */
  private boolean executeAsScript;

  /**
   * Indica se informaes da linha de comando devem ser logadas.
   */
  private boolean logCommandLineInfo;

  /**
   * Limpa as informaes que ficam invlidas quando o comando termina.
   */
  private void eraseFields() {
    setRemoteReference(null);
    this.info = null;
    this.newInfo = false;
  }

  /**
   * Obtm o identificador do projeto no qual o comando  executado.
   *
   * @return O identificador do projeto.
   */
  public Object getProjectId() {
    return cmdInfo.getProjectId();
  }

  /**
   * Retorna um array com o caminho do projeto, a partir da raiz do sistema
   * (exemplo: {"project","admin","proj1"}).
   *
   * @return Array com o caminho do projeto. Retorna <code>null</code> caso o
   *         projeto no seja encontrado.
   */
  String[] getProjectPath() {
    if (!existsProject()) {
      return null;
    }
    ProjectService projectService = ProjectService.getInstance();
    return projectService.getProjectPath(getProjectId());
  }

  /**
   * Atualiza o estado corrente do Comando.
   *
   * @throws ServerException se houver alguma exceo for detectada!
   */
  public synchronized void update() throws ServerException {
    /*
     * Se a informao ainda  nova, i.e. o cliente no coletou, no faz update.
     * Deve verificar se tem referncia para no logar a exceo vrias vezes.
     */
    if (newInfo || (!isAlive())) {
      return;
    }
    try {
      // Busca informaes do comando
      RunningCommandInfo gotInfo = remoteReference.getRunningCommandInfo();
      if (gotInfo != null) {
        setNewInfo(gotInfo);
      }
      ++nupdates;
    }
    catch (Exception e) {
      setRemoteReference(null);
      throw new ServerException(
        "Falha de update do comando:\n" + "\tId: " + getId() + "\n", e);
    }
  }

  /**
   * Atribui informaes sobre o comando vindas do SGA.
   *
   * @param gotInfo informaes sobre o comando.
   */
  private synchronized void setNewInfo(RunningCommandInfo gotInfo) {
    info = gotInfo;
    newInfo = true;
    lastInfoUpdate = System.currentTimeMillis();
  }

  /**
   * Informa quantas vezes o comando foi atualizado Usado para fazer freqencia
   * de update incremental.
   *
   * @return o nmero de vezes que o comando foi atualizado.
   */
  public long getNUpdates() {
    return nupdates;
  }

  /**
   * Cria os arquivos e diretrios gerados pelo comando na rea de projetos, com
   * estado "em construo".
   *
   * @param outputFileCollection os arquivos e diretrios de sada do comando
   *
   * @return o conjunto de arquivos e diretrios criados
   */
  private SGAFile[] createCommandOutputs(
    Collection<FileURLValue> outputFileCollection) {
    if ((outputFileCollection == null) || (outputFileCollection.size() == 0)) {
      return null;
    }
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = getProjectId();
    int numFiles = outputFileCollection.size();
    SGAFile[] commandFiles = new SGAFile[numFiles];
    Iterator<FileURLValue> outputFileIterator = outputFileCollection.iterator();
    for (int i = 0; i < numFiles; i++) {
      FileURLValue file = outputFileIterator.next();
      String[] filePath = file.getPathAsArray();
      String fileType = file.getType();
      try {
        if (!projectService.existsFile(projectId, filePath)) {
          String[] dirPath = new String[filePath.length - 1];
          for (int j = 0; j < filePath.length - 1; j++) {
            dirPath[j] = filePath[j];
          }
          String fileName = filePath[filePath.length - 1];
          projectService.createFile(projectId, dirPath, fileName, fileType);
        }
        projectService.setUnderConstruction(projectId, filePath, true);
        commandFiles[i] = new SGAFile(fileType, filePath);
      }
      catch (Exception e) {
        commandFiles[i] = null;
        Server.logSevereMessage("Falha na criao do arquivo " + file.getPath(),
          e);
      }
    }
    return commandFiles;
  }

  /**
   * <p>
   * Atualiza os comandos utilizando o {@link CommandPersistenceService servio
   * de persistncia de comandos}.
   * </p>
   *
   * <p>
   * O servio de persistncia de comandos pode no estar disponvel em todos os
   * sistemas. Neste caso, este mtodo no faz nada e retorna {@code true}.
   * </p>
   */
  private void updateCommand() {
    try {
      CommandPersistenceService persistenceService =
        CommandPersistenceService.getInstance();
      if (persistenceService == null) {
        return;
      }
      persistenceService.updateCommandInfo(getCommandInfo());
    }
    catch (ServiceFailureException e) {
      Server.logSevereMessage("Ocorreu um erro ao atualizar o comando "
        + getId() + " no servio de persistncia", e);
    }
  }

  /**
   * Atualiza o estado dos arquivos gerados pelo comando, removendo a
   * sinalizao "em construo".
   */
  private void updateCommandOutputFiles() {
    if (commandOutputFiles == null) {
      return;
    }
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = getProjectId();
    for (SGAFile sgaFile : commandOutputFiles) {
      if (sgaFile != null) {
        try {
          projectService.setUnderConstruction(projectId, sgaFile.getFileName(),
            false);
        }
        catch (Exception e) {
          Server.logSevereMessage("Falha na atualizao do estado do arquivo "
            + sgaFile.getFileNameForSeparator('/'), e);
        }
      }
    }
  }

  /**
   * Trmino do comando.
   *
   * @param cmdInfo dados sobre o comando completado.
   *
   * @return <code>true</code> se a finalizao for bem sucedida.
   */
  public boolean finish(CompletedCommandInfo cmdInfo) {
    String msg =
      "Comando terminado.\n" + "Dados da execuo de (" + getId() + ")\n";
    if ((cmdInfo == null) || (cmdInfo.elapsedTimeSec == -1)) {
      msg = msg + "\tTempo decorrido: <No informado>;\n";
    }
    else {
      msg =
        msg + "\tTempo decorrido: " + cmdInfo.elapsedTimeSec + " segundo(s);\n";
    }
    if ((cmdInfo == null) || (cmdInfo.userTimeSec == -1)) {
      msg = msg + "\tTempo de usurio: <No informado>;\n";
    }
    else {
      msg =
        msg + "\tTempo de usurio: " + cmdInfo.userTimeSec + " segundo(s);\n";
    }
    if ((cmdInfo == null) || (cmdInfo.cpuTimeSec == -1)) {
      msg = msg + "\tTempo de CPU: <No informado>.\n";
    }
    else {
      msg = msg + "\tTempo de CPU: " + cmdInfo.cpuTimeSec + " segundo(s);\n";
    }
    Server.logInfoMessage(msg);
    setRemoteReference(null);
    setStatus(CommandStatus.DOWNLOADING);
    boolean success = service.clearExecutionEnvironment(this);
    if (success) {
      if (existsProject()) {
        updateCommandOutputFiles();
      }
    }
    CommandFinalizationInfo info = readCommandFinalizationInfo();
    setFinished(info);
    // No est claro qual a funo do mtodo a seguir:
    eraseFields();
    return success;
  }

  /**
   * Informa se o projeto ainda existe.
   *
   * @return Verdadeiro se o projeto ainda existe, falso caso contrrio.
   */
  public boolean existsProject() {
    ProjectService projectService = ProjectService.getInstance();
    return projectService.existsProject(getProjectId());
  }

  /**
   * Obtm as informaes de finalizao do comando.
   *
   * @return as informaes de finalizao do comando.
   */
  private CommandFinalizationInfo readCommandFinalizationInfo() {
    AlgorithmConfigurator configurator = getCommandConfigurator();
    if (configurator != null && configurator.hasExitCode()) {
      if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
        return readFlowFinalizationInfo(
          (FlowAlgorithmConfigurator) configurator);
      }
      return readSimpleFinalizationInfo(configurator);
    }
    return new SimpleCommandFinalizationInfo(CommandFinalizationType.END);
  }

  /**
   * Obtm as informaes de finalizao de comandos simples.
   *
   * @param configurator O configurador do comando.
   * @return as informaes.
   */
  private CommandFinalizationInfo readSimpleFinalizationInfo(
    AlgorithmConfigurator configurator) {
    Integer exitCode = null;
    CommandFinalizationType type;
    if (configurator.hasExitCode()) {
      exitCode = readIntegerFromFile(configurator.getExitCodeLogFile());
      if (exitCode != null) {
        if (exitCode != 0) {
          type = CommandFinalizationType.EXECUTION_ERROR;
        }
        else {
          type = CommandFinalizationType.SUCCESS;
        }
      }
      else {
        type = CommandFinalizationType.NO_EXIT_CODE;
      }
    }
    else {
      type = CommandFinalizationType.END;
    }
    return new SimpleCommandFinalizationInfo(type, exitCode);
  }

  /**
   * Obtm as informaes de finalizao de comandos tipo fluxo.
   *
   * @param configurator O configurador do comando.
   * @return as informaes.
   */
  private CommandFinalizationInfo readFlowFinalizationInfo(
    FlowAlgorithmConfigurator configurator) {
    if (configurator.canBeRunAsSimpleCommand()) {
      FlowNode node = configurator.getFlow().getNodes().iterator().next();
      return readSimpleFinalizationInfo(node.getAlgorithmConfigurator());
    }
    ExtendedCommandFinalizationInfo finalizationInfo =
      new ExtendedCommandFinalizationInfo(CommandFinalizationType.END);
    boolean foundError = false;
    boolean foundAllCodes = true;
    boolean hasAllCodes = true;
    for (FlowNode node : configurator.getFlow().getNodes()) {
      CommandFinalizationInfo nodeInfo =
        readSimpleFinalizationInfo(node.getAlgorithmConfigurator());
      if (node.hasExitCode()) {
        if (nodeInfo
          .getFinalizationType() == CommandFinalizationType.NO_EXIT_CODE) {
          foundAllCodes = false;
        }
        else if (nodeInfo
          .getFinalizationType() == CommandFinalizationType.EXECUTION_ERROR) {
          foundError = true;
        }
      }
      else {
        if (!node.isBypassed()) {
          hasAllCodes = false;
        }
      }
      finalizationInfo.setFinalizationInfoForNode(nodeInfo, node.getId());
    }
    Integer consolidatedExitCode = null;
    CommandFinalizationType consolidatedType;
    if (foundError) {
      FileURLValue guiltyNodeLog = configurator.getGuiltyNodeLog();
      if (guiltyNodeLog != null) {
        Integer guiltyNodeId = readIntegerFromFile(guiltyNodeLog);
        if (guiltyNodeId != null) {
          finalizationInfo.setGuiltyNodeId(guiltyNodeId);
          CommandFinalizationInfo finalizationInfoForNode =
            finalizationInfo.getFinalizationInfoForNode(guiltyNodeId);
          if (finalizationInfoForNode != null) {
            consolidatedExitCode = finalizationInfoForNode.getExitCode();
          }
        }
      }
      consolidatedType = CommandFinalizationType.EXECUTION_ERROR;
    }
    else if (hasAllCodes) {
      if (foundAllCodes) {
        consolidatedExitCode = 0;
        consolidatedType = CommandFinalizationType.SUCCESS;
      }
      else {
        consolidatedType = CommandFinalizationType.NO_EXIT_CODE;
      }
    }
    else {
      consolidatedType = CommandFinalizationType.END;
    }
    finalizationInfo.setExitCode(consolidatedExitCode);
    finalizationInfo.setFinalizationType(consolidatedType);
    return finalizationInfo;
  }

  /**
   * L um nmero inteiro a partir do arquivo especificado.
   *
   * @param file O arquivo.
   * @return O nmero inteiro.
   */
  private Integer readIntegerFromFile(FileURLValue file) {
    if (file == null || !existsProject()) {
      return null;
    }
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = this.getProjectId();
    String[] filePath = file.getPathAsArray();
    if (!projectService.existsFile(projectId, filePath)) {
      return null;
    }
    InputStream inputStream = null;
    try {
      inputStream = projectService.getInputStream(projectId, filePath);
      BufferedReader buffReader =
        new BufferedReader(new InputStreamReader(inputStream));
      String line = buffReader.readLine();
      Integer value = null;
      if (line != null) {
        value = Integer.parseInt(line);
      }
      return value;
    }
    catch (IOException e) {
      Server.logSevereMessage(
        "Ocorreu um erro na tentativa de ler o contedo do arquivo "
          + FileUtils.joinPath(filePath) + " do projeto " + projectId
          + " com o cdigo de sada do comando " + this.getId() + ".\n",
        e);
    }
    catch (NumberFormatException e) {
      Server
        .logSevereMessage("Formatado invlido no cdigo de sada do comando "
          + this.getId() + ".\n", e);
    }
    finally {
      try {
        if (inputStream != null) {
          inputStream.close();
        }
      }
      catch (IOException e) {
        Server.logSevereMessage(e.getMessage(), e);
      }
    }
    return null;
  }

  /**
   * Obtm o configurador do comando.
   *
   * @return O configurador do comando.
   */
  private AlgorithmConfigurator getCommandConfigurator() {
    AlgorithmConfigurator configurator = null;
    try {
      configurator = getCommandInfo().getConfigurator();
    }
    catch (RemoteException e) {
      Server.logSevereMessage(
        "Ocorreu um erro na tentativa de obter o configurador do comando "
          + this.getId() + ".\n",
        e);
    }
    return configurator;
  }

  /**
   * Interrompe o comando. Se houver alguma falha de comunicao, deixa o objeto
   * inalterado.
   *
   * @return {@code true} se o comando foi cancelado ou {@code false} caso
   *         ocorra alguma falha ao enviar o sinal de interrupo para a
   *         referncia remota do comando no SGA.
   */
  public synchronized boolean kill() {
    try {
      remoteReference.kill();
      setFinished(
        new SimpleCommandFinalizationInfo(CommandFinalizationType.KILLED));
      Server.logWarningMessage(
        "Comando interrompido:\n" + "\tId: " + getId() + "\n");
      eraseFields();
      return true;
    }
    catch (Exception e) {
      setRemoteReference(null);
      Server.logSevereMessage("Falha ao cancelar comando " + getId()
        + " do usurio " + Service.getUser().getId() + ": " + e.getMessage(),
        e);
      return false;
    }
  }

  /**
   * Sinaliza uma falha no comando.
   *
   * @param cause a causa da falha
   */
  public synchronized void failed(FailureFinalizationType cause) {
    setFinished(
      new SimpleCommandFinalizationInfo(CommandFinalizationType.FAILED, cause));
    Server
      .logWarningMessage("Comando com falha:\n" + "\tId: " + getId() + "\n");
    eraseFields();
  }

  /**
   * comando perdido.
   */
  public synchronized void lost() {
    setStatus(CommandStatus.DOWNLOADING);
    boolean success = service.clearExecutionEnvironment(this);
    if (success) {
      if (existsProject()) {
        updateCommandOutputFiles();
      }
    }
    setFinished(
      new SimpleCommandFinalizationInfo(CommandFinalizationType.LOST));
    Server.logWarningMessage("Comando perdido:\n" + "\tId: " + getId() + "\n");
    eraseFields();
  }

  /**
   * Descobre se o comando requer envio de mail ao seu trmino.
   *
   * @return este flag booleano
   */
  boolean requireMail() {
    return cmdInfo.isMailAtEnd();
  }

  /**
   * Obtm a lista de endereos que devem receber email de aviso de trmino de
   * execuo do comando, alm do prprio dono do projeto.
   *
   * @return o array com endereos de email
   */
  public String[] getEmailList() {
    return cmdInfo.getEmailList();
  }

  /**
   * Obtm a hora que o comando foi iniciado.
   *
   * @return hora que o comando foi iniciado
   */
  long getStartTime() {
    return this.startTimeMillis;
  }

  /**
   * Obtm a descricao do comando.
   *
   * @return a descricao do comando
   */
  String getDescription() {
    return cmdInfo.getDescription();
  }

  /**
   * Obtm a dica para auxiliar a identificao do algoritmo do comando
   *
   * @return .
   */
  public String getTip() {
    return cmdInfo.getTip();
  }

  /**
   * Obtm o usurio que executou o comando.
   *
   * @return este identificador
   */
  public Object getUserId() {
    return cmdInfo.getUserId();
  }

  /**
   * Obtm o identificador do comando.
   *
   * @return este identificador
   */
  public String getId() {
    return cmdInfo.getId();
  }

  /**
   * Obtm informaes do comando.
   *
   * @return as informaes do comando
   */
  protected CommandInfo getCommandInfo() {
    return this.cmdInfo;
  }

  /**
   * Obtm a String do comando.
   *
   * @return o comando
   */
  public String getCommand() {
    return this.command;
  }

  /**
   * Obtm o estado de execuo do comando.
   *
   * @return .
   */
  public CommandStatus getStatus() {
    return cmdInfo.getStatus();
  }

  /**
   * Obtm as informaes sobre a finalizao do comando.
   *
   * @return as informaes sobre a finalizao do comando.
   */
  public CommandFinalizationInfo getFinalizationInfo() {
    return cmdInfo.getFinalizationInfo();
  }

  /**
   * Altera o estado de execuo do comando.
   *
   * @param cmdStatus o estado corrente.
   * @return <tt>False</tt> indicando que a alterao no foi feita por que o
   *         comando est no estado terminal - {@link CommandStatus#FINISHED} -.
   */
  private boolean setStatus(CommandStatus cmdStatus) {
    if (cmdInfo.setStatus(cmdStatus)) {
      updateCommand();
      // O ideal  migrar para o commandPersistenceService e gerar um
      // arquivo por comando.
      sga.saveCommandTable();
      return true;
    }
    return false;
  }

  /**
   * Altera o estado do comando para {@link CommandStatus#FINISHED terminado} e
   * atribui um valor ao {@link CommandFinalizationInfo modo em que este comando
   * foi finalizado}.
   *
   * @param info informaes sobre a finalizao do comando.
   */
  private void setFinished(CommandFinalizationInfo info) {
    cmdInfo.setFinished(info);
    updateCommand();
    // O ideal  migrar para o commandPersistenceService e gerar um
    // arquivo por comando.
    sga.saveCommandTable();
  }

  /**
   * Atualiza a referncia remota para o comando no SGA. Necessrio quando o SGA
   * saiu do ar e est voltando.
   *
   * @param remoteReference a referncia remota
   */
  public synchronized void setRemoteReference(SGACommand remoteReference) {
    this.remoteReference = remoteReference;
  }

  /**
   * Informa se h conexo com o comando em execuo.
   *
   * @return <code>true</code> caso o comando disponha de uma referncia para o
   *         processo em execuo.
   */
  boolean isAlive() {
    return remoteReference != null;
  }

  /**
   * Retorna o caminho do sandbox na maquina de execucao.
   *
   * @return O caminho eh retornado como um array de string, representando a
   *         sequancia de diretorios dentro da raiz exportada pelo CSFS.
   */
  String[] getSandbox() {
    return sandbox;
  }

  /**
   * Retorna referncia para o SGA que est executando o comando.
   *
   * @return referncia para o SGA que est executando o comando.
   */
  SGA getSGA() {
    return sga;
  }

  /**
   * Ajusta a referncia para o SGA que est executando o comando.
   *
   * @param sga referncia para o SGA que est executando o comando.
   */
  void setSGA(SGA sga) {
    this.sga = sga;
  }

  /**
   * Define o estado inicial da area de trabalho na maquina de execucao. Essa
   * informacao eh util para que a copia dos arquivos alterados seja avaliada e
   * realizada apos a execucao de um comando. Verificar a classe
   * <code>CSFSService</code>.
   *
   * BUG WARNING: O sistema pode ter problemas caso o SSI caia. Quando o comando
   * for concluido, essa mapeamento nao estara disponivel e o sistema nao serah
   * capaz de saber quais arquivos foram modificados. Nesse cenario, eh
   * necesario verificar/implementar um mecanismo de persistencia desse
   * mapeamento, ou forcar a copia de todos os arquivos do sandbox.
   *
   * @param snapshot O estado inicial da rea de trabalho.
   */
  void setSandboxSnapshot(Map<String[], Long> snapshot) {
    this.sandboxSnapshot = snapshot;
  }

  /**
   * Recupera o mapeamento com a informacao sobre o estado inicial dos arquivos
   * (no sandbox).
   *
   * @return mapeamento com a informacao sobre o estado inicial dos arquivos no
   *         sandbox.
   */
  Map<String[], Long> getSandboxSnapshot() {
    return this.sandboxSnapshot;
  }

  /**
   * Define um sandbox temporrio a ser utilizado para a execuo do comando.
   * Substitui os caminhos originalmente preparados para um SGA com acesso 
   * rea de projetos e aos algoritmos via NFS por caminhos dentro do sandbox do
   * CSFS.
   *
   * @param sandbox array representando o caminho do sandbox temporrio criado
   *        para essa execuo (relativo ao diretrio-raiz do CSFS).
   */
  void setSandbox(String[] sandbox) {
    this.sandbox = sandbox;
  }

  /**
   * Define o caminho do diretrio de projeto onde o comando ser executado.
   *
   * @param projectPath o caminho do diretrio raz do csfs
   */
  void setProjectPath(String[] projectPath) {
    this.projectPath = projectPath;
  }

  /**
   * Atribui o caminho para o repositrio de projetos.
   *
   * @param projectRootPath O caminho.
   */
  void setProjectRootPath(String[] projectRootPath) {
    this.projectRootPath = projectRootPath;
  }

  /**
   * Atribui o caminho para o repositrio de algoritmos.
   *
   * @param algorithmRootPath O caminho.
   */
  void setAlgorithmRootPath(String[] algorithmRootPath) {
    this.algorithmRootPath = algorithmRootPath;
  }

  /**
   * Executa o comando remoto. Esse metodo deve ser chamado apos a copia dos
   * binarios e dos dados para a maquina de execucao. Ou no caso onde esses
   * arquivos estao acessiveis via NFS.
   */
  synchronized protected void executeRemoteCommand() {
    String id = getId();
    if (service.getSGACommand(getSGA().getName(), id) == null) {
      return;
    }
    setStatus(CommandStatus.UPLOADING);
    if (!service.setupExecutionEnvironment(this)) {
      sga.failCommand(id,
        FailureFinalizationType.FAILED_SETUP_EXECUTION_ENVIRONMENT);
      return;
    }
    final Set<String> sandboxes = new TreeSet<String>();
    try {
      AlgorithmConfigurator configurator;
      configurator = cmdInfo.getConfigurator();
      CommandLineContext context = createContextForConfigurator();
      sandboxes.add(context.getSandboxDirectory());
      if (isFlow(configurator)) {
        FlowAlgorithmConfigurator flowConfigurator =
          (FlowAlgorithmConfigurator) configurator;
        FlowCommandLineContext flowContext =
          createFlowContextForConfigurator(context, flowConfigurator);
        context = flowContext;
        sandboxes.add(flowContext.getSandboxDirectory());
        Set<Node> nodes = flowConfigurator.getNodes();
        for (Node flowNode : nodes) {
          CommandLineContext nodeContext =
            createContextForConfigurator(flowNode.getId());
          sandboxes.add(nodeContext.getSandboxDirectory());
          Set<String> linkDirs = getLinkDirNames(flowNode, flowContext);
          sandboxes.addAll(linkDirs);
          flowContext.addFlowNodeContext(flowNode.getId(), nodeContext);
        }
      }

      if (existsProject()) {
        /** Cria os arquivos de sada no diretrio de projetos. */
        Collection<FileURLValue> outputFileCollection =
          configurator.getOutputFiles();
        Collection<FileURLValue> outputDirColletion =
          configurator.getOutputDirectories();
        Collection<FileURLValue> outputs = new TreeSet<FileURLValue>();
        outputs.addAll(outputDirColletion);
        outputs.addAll(outputFileCollection);
        this.commandOutputFiles = createCommandOutputs(outputs);
      }

      if (this.executeAsScript) {
        CommandScript[] commandScripts =
          configurator.makeCommandLineAsScript(context);
        for (CommandScript script : commandScripts) {
          saveCommandScript(script);
        }
        this.command = commandScripts[0].makeCommandLine();
        this.service.uploadScripts(this);
      }
      else {
        this.command = configurator.makeCommandLine(context);
      }

      /*
       * Se estiver habilitado o log da linha de comando, utiliza o
       * EventLogService para registrar o que ser executado pelo SGA, incluindo
       * o nome dos algoritmos e parmetros utilizados.
       */
      if (logCommandLineInfo) {
        if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
          FlowAlgorithmConfigurator flowConfigurator =
                  (FlowAlgorithmConfigurator) configurator;
          Set<Node> nodes = flowConfigurator.getNodes();
          // O fluxo com uma s caixinha  executado como um comando simples
          if (nodes.size() == 1) {
            Node node = nodes.iterator().next();
            logCommandLineInfo(id, node.getConfigurator(), "simple");
          } else {
            for (Node flowNode : nodes) {
              AlgorithmConfigurator nodeConfigurator = flowNode.getConfigurator();
              logCommandLineInfo(id, nodeConfigurator, "flow");
            }
          }
        } else {
          logCommandLineInfo(id, configurator, "simple");
        }
      }
      // A linha de comando simples sempre  logada pelo servidor
      Server.logInfoMessage("Linha de comando: " + this.command);
      String hostNameParam = hostName;
      if (hostNameParam == null) {
        // CORBA nao aceita parametro null p/ String
        hostNameParam = "";
      }
      SGAServer sgaRemoteReference = this.sga.getRemoteReference();
      String outputPath = getOutputDirectoryAsString(context);
      String binaryDirectory =
        getBinaryDirectoryAsString(configurator, context);
      String[] sandboxArray = sandboxes.toArray(new String[sandboxes.size()]);
      this.remoteReference = sgaRemoteReference.executeCommand(this.command, id,
        hostNameParam, binaryDirectory, outputPath, sandboxArray);
      if (this.remoteReference == null) {
        sga.failCommand(id, FailureFinalizationType.SGA_EXECUTION_ERROR);
        String errorMessage =
          "Erro inesperado na execuo do comando pelo SGA. O erro est registrado no log do SGA.\n";
        Server.logSevereMessage(
          errorMessage + "\tCausa: " + errorMessage + "\n\tId: " + id + "\n\n");
      }
      else {
        setStatus(CommandStatus.EXECUTING);
        Server.logInfoMessage(String.format(
          "Comando '%s' posto em execuo:\n\tId: %s\n\t(%s)\n\tComando: %s\n",
          getTip(), id, this.sga.getName(), this.command));
      }
    }
    catch (Exception e) {
      sga.failCommand(id,
        FailureFinalizationType.FAILED_SETUP_EXECUTION_ENVIRONMENT);
      String errorMessage =
        "Falha na construo do comando durante a preparao do ambiente de execuo.\n";
      Server.logSevereMessage(errorMessage + "\tCausa: " + errorMessage + ": ["
        + this.command + "]\n\tSGA [" + sga.getName() + "] - id: " + id
        + ".\n\tErro: " + e + ".\n", e);
    }
  }

  /**
   * Loga a linha de comando no EventLog.
   *
   * @param id o identificador do comando.
   * @param configurator o configurador do algoritmo.
   * @param queue a fila de dados a ser gravadas.
   */
  private void logCommandLineInfo(String id, AlgorithmConfigurator configurator, String queue) {
    EventLogService eventLogger = EventLogService.getInstance();
    String queueBaseName = "commands";
    String[] logQueue = new String[] {queueBaseName, queue};
    String[] logInfo = new String[] {id, configurator.getAlgorithmName(),
            makeParameterValueList(configurator)};
    eventLogger.addServerInformation(logQueue, logInfo);
  }

  /**
   * Monta uma string com a lista de parmetros e seus valores de um
   * configurador. Exemplo: "PARAM1=VALOR1 ; PARAM2=VALOR2; PARAM3=VALOR3"
   *
   * @param configurator o configurador.
   * @return a string com a lista de parmetros.
   */
  private String makeParameterValueList(AlgorithmConfigurator configurator) {
    Map<String, String> parameterValues =
      configurator.getParameterValuesByName();
    StringBuilder parameters = new StringBuilder();
    for (Entry<String, String> parameter : parameterValues.entrySet()) {
      parameters.append(parameter.getKey());
      parameters.append("=");
      parameters.append(parameter.getValue());
      parameters.append(" ; ");
    }
    return parameters.toString();
  }

  /**
   * Indica se um determinado configurador deve ser tratado como um fluxo. Mesmo
   * que o configurador seja do tipo {@link FlowAlgorithmConfigurator}, ele s
   * deve ser tratado como um fluxo caso contenha mais de um n. Caso contrrio,
   * ele ser executado como um comando simples.
   *
   * @param configurator O configurador.
   * @return verdadeiro se o configurador deve ser tratado como um fluxo ou
   *         falso, caso contrrio.
   */
  private boolean isFlow(AlgorithmConfigurator configurator) {
    boolean isFlow = false;
    if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
      FlowAlgorithmConfigurator flowAlgorithmConfigurator =
        (FlowAlgorithmConfigurator) configurator;
      Set<Node> nodes = flowAlgorithmConfigurator.getNodes();
      if (nodes.size() > 1) {
        isFlow = true;
      }
    }
    return isFlow;
  }

  /**
   * Cria o contexto para gerao da linha de comando para o configurador de um
   * algoritmo.
   *
   * @return O contexto para gerao da linha de comando do algoritmo.
   */
  private CommandLineContext createContextForConfigurator() {
    return createContextForConfigurator(null);
  }

  /**
   * Obtm a lista de diretrios utilizados em ligaes de um determinado n de
   * um fluxo. Caso no haja nenhuma ligao do n feita atravs de parmetros
   * do tipo diretrio  retornado um conjunto vazio.
   *
   * @param flowNode o n do fluxo.
   * @param context o contexto do fluxo.
   *
   * @return a lista de diretrios.
   */
  private Set<String> getLinkDirNames(Node flowNode,
    FlowCommandLineContext context) {
    Set<String> dirs = new TreeSet<String>();
    for (Output output : flowNode.getOutputs()) {
      Collection<Input> inputs = output.getInputs();
      if (output.isDir() && !inputs.isEmpty()) {
        String linkDir = CommandLineBuilder.makeLinkDirName(flowNode.getId(),
          output.getParameterName());
        String dirPath = FileUtils.joinPath(linkDir, context.getFileSeparator(),
          context.getSandboxDirectory());
        dirs.add(dirPath);
      }
    }
    return dirs;
  }

  /**
   * Cria o contexto de execuo de um fluxo a partir de um contexto simples,
   * adicionando as informaes especficas pra montagem da linha de comando do
   * fluxo. Alm das informaes do contexto simples, o fluxo precisa da lista
   * de pipes e diretrios de ligao utilizados para fazer as ligaes entre as
   * entradas e sadas dos diferentes ns do fluxo na linha de comando. No caso
   * de ligaes feitas via parmetros do tipo arquivos simples so utilizados
   * named-pipes para fazer a conexo, enquanto que ligaes feitas via
   * parmetros do tipo diretrio so feitas com diretrios comuns.
   *
   * @param simpleContext o contexto simples.
   * @param configurator o configurador do fluxo.
   * @return o contexto do fluxo criado.
   */
  private FlowCommandLineContext createFlowContextForConfigurator(
    CommandLineContext simpleContext, FlowAlgorithmConfigurator configurator) {
    char fileSeparator = simpleContext.getFileSeparator();
    /*
     * Named-pipes mapeados pelo nome do parmetro de sada relacionado,
     * mapeados pelo identificador do n
     */
    Map<Integer, Map<String, String>> allFromPipes =
      new HashMap<Integer, Map<String, String>>();
    /*
     * Named-pipes mapeados pelo nome do parmetro de entrada relacionado,
     * mapeados pelo identificador do n
     */
    Map<Integer, Map<String, String>> allToPipes =
      new HashMap<Integer, Map<String, String>>();
    /*
     * Diretrios de ligao mapeados pelo nome do parmetro de entrada ou sada
     * relacionado, mapeados pelo identificador do n. No caso de diretrios,
     * podemos utilizar o mesmo nome tanto para entrada quanto para sada.
     */
    Map<Integer, Map<String, String>> allLinkDirs =
      new HashMap<Integer, Map<String, String>>();
    for (Node flowNode : configurator.getNodes()) {
      for (Output output : flowNode.getOutputs()) {
        for (Input input : output.getInputs()) {
          if (output.isDir()) {
            Map<String, String> outputDirs =
              allLinkDirs.get(output.getNodeId());
            if (outputDirs == null) {
              outputDirs = new HashMap<String, String>();
              allLinkDirs.put(output.getNodeId(), outputDirs);
            }
            Map<String, String> inputDirs = allLinkDirs.get(input.getNodeId());
            if (inputDirs == null) {
              inputDirs = new HashMap<String, String>();
              allLinkDirs.put(input.getNodeId(), inputDirs);
            }
            String linkDir = CommandLineBuilder
              .makeLinkDirName(flowNode.getId(), output.getParameterName());
            String linkDirPath =
              CommandLineBuilder.makePathWithEnvironmentVariable(
                EnvironmentVariable.FLOW_SANDBOX_DIR, linkDir, fileSeparator);
            /*
             * O mesmo diretrio  definido como a sada do parmetro de origem
             * e na entrada do parmetro de destino.
             */
            outputDirs.put(output.getParameterName(), linkDirPath);
            inputDirs.put(input.getParameterName(), linkDirPath);
          }
          else {
            Map<String, String> toPipes = allToPipes.get(input.getNodeId());
            if (toPipes == null) {
              toPipes = new HashMap<String, String>();
              allToPipes.put(input.getNodeId(), toPipes);
            }
            Map<String, String> fromPipes = allFromPipes.get(flowNode.getId());
            if (fromPipes == null) {
              fromPipes = new HashMap<String, String>();
              allFromPipes.put(flowNode.getId(), fromPipes);
            }
            String toPipeName = CommandLineBuilder
              .makeToPipeName(input.getNodeId(), input.getParameterName());
            String fromPipeName = CommandLineBuilder
              .makeFromPipeName(flowNode.getId(), output.getParameterName());
            String toPipePath =
              CommandLineBuilder.makePathWithEnvironmentVariable(
                EnvironmentVariable.FLOW_SANDBOX_DIR, toPipeName,
                fileSeparator);
            String fromPipePath =
              CommandLineBuilder.makePathWithEnvironmentVariable(
                EnvironmentVariable.FLOW_SANDBOX_DIR, fromPipeName,
                fileSeparator);
            /*
             * No caso de arquivos simples,  criado um named-pipe para o
             * arquivo de sada e outro para o arquivo de entrada. Esses pipes
             * sero "conectados" em tempo de execuo pelo flowmonitor.
             */
            fromPipes.put(output.getParameterName(), fromPipePath);
            toPipes.put(input.getParameterName(), toPipePath);
          }
        }
      }
    }
    FlowCommandLineContext context = FlowCommandLineContext
      .createFlowContext(simpleContext, allFromPipes, allToPipes, allLinkDirs);
    return context;
  }

  /**
   * Cria o contexto para gerao da linha de comando para o configurador de um
   * algoritmo pertencente a um fluxo.
   *
   * @param nodeId O identificador do algoritmo no fluxo.
   * @return O contexto para gerao da linha de comando do algoritmo.
   */
  private CommandLineContext createContextForConfigurator(Integer nodeId) {
    String id = this.getId();
    char fileSeparator = node.getFileSeparator();

    String projectDirectory =
      FileUtils.joinPath(fileSeparator, getProjectDirectory());
    String sandboxDirectory =
      FileUtils.joinPath(fileSeparator, getSandboxDirectory(nodeId));
    /*
     * Os caminhos para o diretrio de projetos, algoritmos e raiz do sistema
     * so *sempre* absolutos, obtidos do SGA que ir executar o comando.
     */
    String projRootPath =
      FileUtils.joinPath(true, fileSeparator, getProjectRootDirectory());
    String algoRootPath =
      FileUtils.joinPath(true, fileSeparator, getAlgorithmRootDirectory());
    String sandboxRootPath =
      FileUtils.joinPath(true, fileSeparator, getSandboxRootDirectory());
    String commandPersistencyPath =
      FileUtils.joinPath(true, fileSeparator, getCommandPersistencyDirectory());

    CommandLineContext context = new CommandLineContext(id,
      node.getPlatformId(), fileSeparator, projectDirectory, algoRootPath,
      projRootPath, sandboxRootPath, sandboxDirectory, commandPersistencyPath,
      executeAsScript, nodeId, cmdInfo.getClientHostName());
    return context;
  }

  /**
   * Salva o script com a linha de comando do algoritmo.
   *
   * @param script O script a ser salvo.
   *
   * @throws OperationFailureException se no for possvel salvar o script.
   */
  private void saveCommandScript(CommandScript script)
    throws OperationFailureException {
    CommandPersistenceService persistenceService =
      CommandPersistenceService.getInstance();
    persistenceService.saveCommandScript(script.getFileName(), getId(),
      getProjectId(), script.getScriptContent());
  }

  /**
   * Monta o caminho para o diretrio de sada de logs do algoritmo.
   *
   * @param context O contexto para gerao de linha de comando do algoritmo.
   * @return O caminho para o diretrio de sado do algoritmo.
   */
  private String getOutputDirectoryAsString(CommandLineContext context) {
    char separatorChar = context.getFileSeparator();
    String logDir = FileUtils.joinPath(separatorChar,
      CommandPersistenceService.getInstance().getLogDirectory(getId()));
    return FileUtils.joinPath(separatorChar, context.getProjectRootDirectory(),
      context.getProjectDirectory(), logDir);
  }

  /**
   * Monta o caminho para o diretrio de binrios do algoritmo.
   *
   * @param configurator O configurador do algoritmo.
   * @param context O contexto para gerao de linha de comando do algoritmo.
   * @return O caminho para o diretrio de binrios do algoritmo.
   */
  private String getBinaryDirectoryAsString(AlgorithmConfigurator configurator,
    CommandLineContext context) {
    char fileSeparator = context.getFileSeparator();
    String binaryPath;
    Set<String> binaryDirectorySet =
      configurator.getBinaryDirectories(node.getPlatformId(), fileSeparator);
    if (!isFlow(configurator)) {
      binaryPath =
        FileUtils.joinPath(fileSeparator, context.getAlgorithmRootDirectory(),
          binaryDirectorySet.iterator().next());
    }
    else {
      // CORBA nao aceita parametro null p/ String
      binaryPath = "./";
    }
    return binaryPath;
  }

  /**
   * Monta o caminho para a sandbox de execuo de um algoritmo.
   *
   * @param nodeId O identificador do algoritmo no fluxo (se for o caso).
   * @return O caminho para a sandbox de execuo.
   */
  private String[] getSandboxDirectory(Integer nodeId) {
    String sandboxDir = FileUtils.fixDirectoryName(getId());
    if (nodeId != null) {
      return new String[] { sandboxDir, sandboxDir + "_" + nodeId };
    }
    return new String[] { sandboxDir };
  }

  /**
   * Monta o caminho para o diretrio de persistncia do comando.
   *
   * @return O caminho para o diretrio de persistncia.
   */
  private String[] getCommandPersistencyDirectory() {
    return SGAService.combinePaths(getProjectRootDirectory(),
      getProjectDirectory(), this.cmdInfo.getPersistencyPath());
  }

  /**
   * Obtm o caminho para o repositrio de algoritmos.
   *
   * @return O caminho para o repositrio de algoritmos.
   */
  private String[] getAlgorithmRootDirectory() {
    if (algorithmRootPath == null) {
      return node.getAlgorithmRootDirectory();
    }
    return this.algorithmRootPath;
  }

  /**
   * Obtm o caminho para o repositrio de projetos.
   *
   * @return O caminho para o repositrio de projetos.
   */
  private String[] getProjectRootDirectory() {
    if (this.projectRootPath == null) {
      return node.getProjectRootDirectory();
    }
    return this.projectRootPath;
  }

  /**
   * Obtm o caminho para o diretrio base de sandboxes.
   *
   * @return O caminho para o diretrio base de sandboxes.
   */
  private String[] getSandboxRootDirectory() {
    return node.getSandboxRootDirectory();
  }

  /**
   * Obtm o caminho para o diretrio do projeto.
   *
   * @return O caminho para o diretrio do projeto.
   */
  private String[] getProjectDirectory() {
    if (this.projectPath == null) {
      return getProjectPath();
    }
    return this.projectPath;
  }

  /**
   * Refaz a referncia para o servio de gerncia dos SGAs.
   *
   * @param sgaService servio de gerncia dos SGAs.
   */
  void setService(SGAService sgaService) {
    this.service = sgaService;
  }

  /**
   * Constri a representao de um comando.
   *
   * @param hostName Nome do n do SGA que deve executar o comando
   * @param sga representao local do SGA da mquina hospedeira
   * @param cmdInfo informaes sobre o comando a ser executado.
   * @param expirationInfoDelay prazo de validade das informaes do comando.
   * @param asScript Indica se o comando deve ser executado como um script.
   * @param logCommandLineInfo Indica se informaes da linha de comando devem
   *        ser logadas.
   *
   * @throws ProjectNotFoundException Em caso de falha na criao do comando.
   */
  public Command(String hostName, SGA sga, CommandInfo cmdInfo,
    long expirationInfoDelay, boolean asScript, boolean logCommandLineInfo)
      throws ProjectNotFoundException {
    this.cmdInfo = cmdInfo;
    // XXX - [CSBASE-567] Rever uso de paths como strings ou arrays.
    // (ver SGAService -- sandbox)
    this.sga = sga;
    this.hostName = hostName;
    this.node =
      hostName == null ? this.sga.getInfo(0) : this.sga.getInfo(hostName);
    this.expirationInfoDelay = expirationInfoDelay;
    this.executeAsScript = asScript;
    this.logCommandLineInfo = logCommandLineInfo;
    // Adicionado somente para persistncia
    Object projectId = cmdInfo.getProjectId();
    if (projectId != null) {
      if (!existsProject()) {
        String message =
          String.format("O projeto %s no foi encontrado.", projectId);
        throw new ProjectNotFoundException(message);
      }
    }
    this.service = SGAService.getInstance();
    this.startTimeMillis = Calendar.getInstance().getTimeInMillis();
    setStatus(CommandStatus.INIT);

    Server.logInfoMessage("Pedido de comando:\n\tUsurio: "
      + cmdInfo.getUserId() + "\n\tId: " + cmdInfo.getId());
  }

  /**
   * Obtm o node que ir executar o comando.
   *
   * @return .
   */
  public SGAInfo getNode() {
    return node;
  }

  /**
   * Obtm o nome do projeto no qual o comando foi executado.
   *
   * @return o nome do projeto no qual o comando foi executado.
   */
  public String getProjectName() {
    ProjectService projectService = ProjectService.getInstance();
    return projectService.getProjectName(getProjectId());
  }

  /**
   * <p>
   * Cria as informaes da monitorao do comando.
   * </p>
   *
   * <p>
   * ATENO: A sincronizao deste mtodo garante que as informaes so da
   * mesma instncia de RunningCommandInfo.
   * </p>
   *
   * @return comando com as ltimas informaes obtidas
   */
  public synchronized CommandInfo createCommandInfo() {
    // Indica que a informao j foi lida pelo cliente e ativa o update.
    newInfo = false;
    if ((System.currentTimeMillis() - lastInfoUpdate) > expirationInfoDelay
      || info == null) {
      if (cmdInfo.getStatus() == CommandStatus.EXECUTING && !isAlive()) {
        cmdInfo.invalidateDynamicFields();
      }
      return cmdInfo;
    }
    // As informaes ainda so vlidas, ento confio na cache.
    ProcessInfo[] infos = info.processData;
    if (infos.length == 0) {
      cmdInfo.validateDynamicFields();
      return cmdInfo;
    }

    boolean queued = false;
    /*
     * Se pelo menos um dos processos do comando est em espera na fila do
     * escalonador (escalonador externo, no o Scheduler), o comando fica
     * marcado com enfileirado.
     *
     * ver CommandInfo#isQueued
     */
    for (ProcessInfo processInfo1 : infos) {
      ProcessState state = processInfo1.state;
      if (state.value() == ProcessState._WAITING) {
        queued = true;
        break;
      }
    }
    cmdInfo.setQueued(queued);

    // Calcula o percentual total de uso de CPU por todos os processos do
    // comando.
    Double cpuPerc = 0.0;
    for (ProcessInfo processInfo : infos) {
      if (processInfo.CPUPerc < 0) {
        cpuPerc = null;
        break;
      }
      cpuPerc += processInfo.CPUPerc;
    }
    cmdInfo.setCpuPerc(cpuPerc);

    // Calcula o total de memria RAM (em MB) utilizada por todos os
    // processos do comando.
    Double ramMB = 0.0;
    for (ProcessInfo processInfo : infos) {
      if (processInfo.memoryRamSizeMb < 0) {
        ramMB = null;
        break;
      }
      ramMB += processInfo.memoryRamSizeMb;
    }
    cmdInfo.setRAMMemoryMB(ramMB);
    // Calcula o percentual de memria RAM utilizada por todos os
    // processos do comando.
    Double memoryRAMPerc = null;
    if (ramMB != null) {
      // [CSBASE-567]
      // Calculando o total de memria do SGA (considerando caso de ser
      // cluster)
      int sgaRAMMB = 0;
      for (SGAInfo sgaInfo : sga.getAllInfo()) {
        sgaRAMMB += sgaInfo.getRAMMemoryInfoMb();
      }
      memoryRAMPerc = (sgaRAMMB == 0) ? null : 100 * (ramMB / sgaRAMMB);
    }
    cmdInfo.setRAMMemoryPerc(memoryRAMPerc);
    // Calcula o total de memria Swap (em MB) utilizada por todos os
    // processos do comando.
    Double swapMB = 0.0;
    for (ProcessInfo processInfo : infos) {
      if (processInfo.memorySwapSizeMb < 0) {
        swapMB = null;
        break;
      }
      swapMB += processInfo.memorySwapSizeMb;
    }
    cmdInfo.setSwapMemoryMB(swapMB);
    // Calcula o percentual de memria swap utilizada por todos os
    // processos do comando.
    Double memorySwapPerc = null;
    if (swapMB != null) {
      // [CSBASE-567]
      // Calculando o total de memria do SGA (considerando caso de ser
      // cluster)
      int sgaSwapMB = 0;
      for (SGAInfo sgaInfo : sga.getAllInfo()) {
        sgaSwapMB += sgaInfo.getSwapMemoryInfoMb();
      }
      memorySwapPerc = (sgaSwapMB == 0) ? null : 100 * (swapMB / sgaSwapMB);
    }
    cmdInfo.setSwapMemoryPerc(memorySwapPerc);

    Integer wallTimeSec =
      (infos[0].wallTimeSec < 0) ? null : infos[0].wallTimeSec;
    cmdInfo.setWallTimeSec(wallTimeSec);
    Double userTimeSec = 0.0;
    for (ProcessInfo processInfo : infos) {
      if (processInfo.userTimeSec < 0) {
        userTimeSec = null;
        break;
      }
      userTimeSec += processInfo.userTimeSec;
    }
    cmdInfo.setUserTimeSec(userTimeSec);

    Double systemTimeSec = 0.0;
    for (ProcessInfo processInfo : infos) {
      if (processInfo.systemTimeSec < 0) {
        systemTimeSec = null;
        break;
      }
      systemTimeSec += processInfo.systemTimeSec;
    }
    cmdInfo.setSystemTimeSec(systemTimeSec);

    Double virtualMemorySizeMB = 0.0;
    for (ProcessInfo processInfo : infos) {
      if (processInfo.virtualMemorySizeMB < 0) {
        virtualMemorySizeMB = null;
        break;
      }
      virtualMemorySizeMB += processInfo.virtualMemorySizeMB;
    }
    cmdInfo.setVirtualMemorySizeMB(virtualMemorySizeMB);

    Double bytesInKB = 0.0;
    if (infos[0].bytesInKB < 0) {
      bytesInKB = null;
    }
    else {
      bytesInKB = infos[0].bytesInKB;
    }
    cmdInfo.setBytesInKB(bytesInKB);

    Double bytesOutKB = 0.0;
    if (infos[0].bytesOutKB < 0) {
      bytesOutKB = null;
    }
    else {
      bytesOutKB = infos[0].bytesOutKB;
    }
    cmdInfo.setBytesOutKB(bytesOutKB);

    Double diskBytesReadKB = 0.0;
    if (infos[0].diskBytesReadKB < 0) {
      diskBytesReadKB = null;
    }
    else {
      diskBytesReadKB = infos[0].diskBytesReadKB;
    }
    cmdInfo.setDiskBytesReadKB(diskBytesReadKB);

    Double diskBytesWriteKB = 0.0;
    if (infos[0].diskBytesWriteKB < 0) {
      diskBytesWriteKB = null;
    }
    else {
      diskBytesWriteKB = infos[0].diskBytesWriteKB;
    }
    cmdInfo.setDiskBytesWriteKB(diskBytesWriteKB);

    Set<String> nodes = new HashSet<String>();
    for (ProcessInfo inf : infos) {
      if (inf.execHost != null) {
        nodes.add(inf.execHost);
      }
    }
    String executionNodeDescription = "";
    if (sga.isCluster()) {
      String sep = "";
      for (String node : nodes) {
        executionNodeDescription += sep + node;
        sep = " | ";
      }
    }
    cmdInfo.setExecutionNodeDescription(executionNodeDescription);

    Map<String, String> specificData = new HashMap<String, String>();
    for (ExecutionInfo execInfo : info.executionData) {
      specificData.put(execInfo.key, execInfo.value);
    }
    cmdInfo.setSpecificData(specificData);
    cmdInfo.validateDynamicFields();
    return cmdInfo;
  }

}
