/*
 * SGA.java
 * 
 * $Author: cviana $ $Revision: 181607 $ - $Date: 2007-10-25 13:54:22 -0300
 * (Thu, 25 Oct 2007) $
 */
package csbase.server.services.sgaservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.lang.reflect.Field;
import java.rmi.RemoteException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import sgaidl.Address;
import sgaidl.ByteOrder;
import sgaidl.CType;
import sgaidl.Capacity;
import sgaidl.DynamicNodeInfo;
import sgaidl.SGACommand;
import sgaidl.SGADynamicInfo;
import sgaidl.SGANetCapacity;
import sgaidl.SGAPath;
import sgaidl.SGAServer;
import sgaidl.StaticNodeInfo;
import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.OperationFailureException;
import csbase.logic.CapacityType;
import csbase.logic.ClientSGAFile;
import csbase.logic.CommandInfo;
import csbase.logic.FailureFinalizationType;
import csbase.logic.SGAInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.remote.EventLogServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.eventlogservice.EventLogService;

/**
 * A classe <code>SGA</code> representa o gerenciador de uma mquina hospedeira
 * (ou servidor), capaz de atender  solicitao de execuo remota de
 * algoritmos. Alm de comandar a execuo remota de um algoritmo, um objeto SGA
 * permite a consulta  configurao da mquina hospedeira (<i>hostname</i>,
 * plataforma de execuo, nmero de processadores, ...) e a seu estado corrente
 * (carga no processador, memria livre disponvel).
 * 
 * @author Ana Moura, Andr Clinio e Cristina Ururahy
 */
class SGA {

  /** Nome do arquivo de persistncia para comandos em execuo. */
  private final static String EXECUTION_COMMANDS_FILENAME_SUFFIX =
    "-execution-commands.dat";

  /** Constante para converso para milisegundos */
  private static final int MILLIS = 1000;

  /** Constante que indica a ausncia de determinado dado do histrico */
  private static final int NO_DATA = -1;

  /** Tabela de comandos em execuo no SGA. */
  private Hashtable<Object, Command> commandTable = null;

  /** Servio que gerencia este SGA. */
  private SGAService sgaService = null;

  /**
   * Informaes de cada um dos ns do SGA. Podem ser serializadas para o
   * cliente.
   */
  private SGAInfo[] info = null;

  /** Nome que representa o SGA */
  private String name = null;

  /** Mquina que deve ser contactada para transferncias via CSFS. */
  private String csfsHost;

  /** Porta da mquina que deve ser contactada para transferncias via CSFS. */
  private int csfsPort;

  /** Diretrio que deve ser usado para transferncias via CSFS. */
  private String[] csfsRootDir;

  /** Indica se o SGA tem acesso ao disco. */
  private boolean hasDiskAccess;

  /** Representa as informaes dos jobs em execuo no SGA. */
  private String jobsInfo;

  /** Indica se o SGA  um cluster. */
  private boolean isCluster;

  /** Indica se o SGA est apto a executar comandos. */
  private boolean enabled;

  /** Um watchdog verifica se o SGA est atualizando seus dados. */
  /*
   * Quando o SGA atualiza seus dados ele faz uma marca (patWatchDog) e de
   * tempos em tempos o SGAService retira essa marca (resetWatchDog). Quando o
   * SGAService vai retirar a marca e ela no est l, significa que o SGA no
   * atualizou seus dados. Neste caso, o SGAService considera que o SGA est
   * inacessvel.
   */
  private boolean watchdog;

  /** Tabela de requisitos do SGA. */
  private Hashtable<String, Boolean> requirements = null;

  /** Referncia remota para o gerenciador da mquina */
  private SGAServer remoteReference;

  /** Constante indicativa de restart */
  private static final int RESTART = 1;

  /** Constante indicativa de shutdown */
  private static final int SHUTDOWN = 2;

  /** Indica se o SGA executa o benchmark de rede */
  private Integer execBench = null;

  /**
   * Flag indicativo de sada da thread de aquisio de dados dos SGAs.
   * */
  private boolean exitAcquisitionDataThread = false;

  /**
   * Tabela com o mapeamento entre as capacidades declaradas na idl corba e as
   * capacidades da enumerao Java
   */
  private static HashMap<Integer, CapacityType> capacityMap = null;

  /** Executor que usa uma thread nica para submisso de comandos. */
  private final ExecutorService commandExecutor;

  /**
   * Mtodo de registro de um evento para o EventLog (servidor).
   * 
   * @param queue um identificador da fila no EventLog.
   * @param info a informao (<b>simples</b>) a ser registrada
   * @see EventLogServiceInterface
   */
  private void logEvent(final String[] queue, final String info) {
    final EventLogService eventService = EventLogService.getInstance();
    final int N = queue.length;
    final String[] reQueue = new String[N + 1];
    reQueue[0] = "sga";
    for (int i = 0; i < N; i++) {
      reQueue[i + 1] = queue[i];
    }
    try {
      eventService.addServerInformation(reQueue, new String[] { info });
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha de log de eventos ", e);
    }
  }

  /**
   * Atualiza a configurao esttica da mquina hospedeira.
   * 
   * @param staticNodesInfo informaes estticas dos ns
   */
  private void setStaticData(StaticNodeInfo[] staticNodesInfo) {
    String sgaName = this.name;
    SGAInfo[] infoTmp = new SGAInfo[staticNodesInfo.length];
    if (infoTmp.length > 1) {
      Server.logInfoMessage("Recebeu dados estticos dos ns de: " + sgaName);
    }
    for (int i = 0; i < infoTmp.length; i++) {
      Server.logInfoMessage("Recebeu dados estticos de: "
        + staticNodesInfo[i].name);
      char fileSeparator = staticNodesInfo[i].file_separator.charAt(0);
      String[] projectRootDirectory =
        FileUtils.splitPath(staticNodesInfo[i].project_root_directory,
          fileSeparator);
      String[] algorithmRootDirectory =
        FileUtils.splitPath(staticNodesInfo[i].algorithm_root_directory,
          fileSeparator);
      String[] sandboxRootDirectory =
        FileUtils.splitPath(staticNodesInfo[i].sandbox_root_directory,
          fileSeparator);
      infoTmp[i] =
        new SGAInfo(
          staticNodesInfo[i].name,
          staticNodesInfo[i].platform_id,
          staticNodesInfo[i].num_processors,
          staticNodesInfo[i].memory_ram_info_mb,
          staticNodesInfo[i].memory_swap_info_mb,
          staticNodesInfo[i].clock_speed_mhz,
          fileSeparator,
          projectRootDirectory,
          algorithmRootDirectory,
          sandboxRootDirectory,
          (staticNodesInfo[i].byte_order == ByteOrder.LITTLE_ENDIAN) ? SGAInfo.LITTLE_ENDIAN
            : SGAInfo.BIG_ENDIAN);

      for (String requirement : staticNodesInfo[i].requirements) {
        infoTmp[i].addRequirement(requirement);
      }
      // Adiciona todas as capacidades medidas pelos benchmarks em uma tabela
      for (Capacity capacity : staticNodesInfo[i].capacities) {
        CapacityType capType = capacityMap.get(capacity.type.value());
        long capValue = capacity.value;
        infoTmp[i].addCapacities(capType, capValue);
      }
    }
    info = infoTmp;
  }

  /**
   * Configura as propriedades usadas para transferncia de arquivos via CSFS.
   * 
   * @throws ServerException se houver falha.
   */
  private void setCSFSProperties() throws ServerException {
    /* Obtm dados do CSFS */
    try {
      Address address = remoteReference.getCSFSAddress();
      csfsHost = address.host;
      if (csfsHost.length() == 0) {
        csfsHost = null;
      }
      csfsPort = address.port;
      Server.logInfoMessage("Obtendo o endereo do CSFS para " + this.name
        + ": " + csfsHost + ":" + csfsPort);
    }
    catch (Exception e) {
      setAlive(false);
      throw new ServerException("Falha na obteno do endereo do CSFS de : "
        + this.name, e);
    }
    try {
      String csfsRoot = remoteReference.getCSFSRootDir();
      if (csfsRoot.length() == 0) {
        this.csfsRootDir = null;
      }
      else {
        this.csfsRootDir = FileUtils.splitPath(csfsRoot);
      }
      Server.logInfoMessage("Obtendo o CSFSRootDir para " + this.name + " : "
        + csfsRoot);
    }
    catch (Exception e) {
      setAlive(false);
      throw new ServerException("Falha na obteno do CSFSRootDir de : "
        + this.name, e);
    }
  }

  /**
   * Inicializa a taxa de transferncia em rede do SGA. Quando a coleta da taxa
   * de transferncia em rede deixar de ser iniciada pelo servidor, essa
   * informao passar a ser informada atravs dos dados dinmicos dos SGAs e
   * esse mtodo deixar de existir.
   * 
   * @param transferRate Taxa de transferncia do SGA.
   */
  public void setTransferRate(long transferRate) {
    info[0].setTransferRate(transferRate);
  }

  /**
   * Obtm o estado corrente da mquina hospedeira.
   * 
   * @param dynamicInfo Informaes dinmicas do SGA.
   */
  public void updateSGAInfo(SGADynamicInfo dynamicInfo) {
    DynamicNodeInfo[] dynamicNodesInfo = dynamicInfo.nodesInfo;
    hasDiskAccess = dynamicInfo.hasDiskAccess;
    jobsInfo = dynamicInfo.jobsInfo;
    sgaService.logSGAMessage("Recebeu dados dinmicos: " + getName());

    for (int i = 0; i < dynamicNodesInfo.length; i++) {
      SGAInfo myinfo = info[i];
      DynamicNodeInfo data = dynamicNodesInfo[i];

      if ((data.memory_ram_free_perc < 0) && (data.memory_swap_free_perc < 0)
        && (data.load_avg_perc.loadAvg1min < 0)) {
        myinfo.setAlive(false);
      }
      else {
        myinfo.setAlive(true);
      }
      myinfo.setRAMFreeMemory(data.memory_ram_free_perc);
      myinfo.setSwapFreeMemory(data.memory_swap_free_perc);
      myinfo.setCPULoad(data.load_avg_perc.loadAvg1min,
        data.load_avg_perc.loadAvg5min, data.load_avg_perc.loadAvg15min);
      myinfo.setNumberOfJobs(data.number_of_jobs);
    }
    sgaService.logSGAInfo(info, "(Recebido do SGA)");
  }

  /**
   * Obtm a mquina que deve ser contactada para transferncias via CSFS.
   * 
   * @return mquina para transferncias via CSFS.
   */
  public String getCSFSHost() {
    return this.csfsHost;
  }

  /**
   * Obtm a porta da mquina que deve ser contactada para transferncias via
   * CSFS.
   * 
   * @return porta da mquina para transferncias via CSFS.
   */
  public int getCSFSPort() {
    return this.csfsPort;
  }

  /**
   * Obtm o diretrio que deve ser usado para transferncias via CSFS.
   * 
   * @return mquina para transferncias via CSFS.
   */
  public String[] getCSFSRootDir() {
    return this.csfsRootDir;
  }

  /**
   * Retorna lista com os arquivos filhos.
   * 
   * @param path - path pai.
   * @return lista com os arquivos filhos.
   */
  public List<ClientSGAFile> getChildren(String path) {
    List<ClientSGAFile> children = new ArrayList<ClientSGAFile>();

    try {
      for (sgaidl.SGAPath pathInfo : remoteReference.getPaths(path)) {
        ClientSGAFile file = buildClientSGAFile(pathInfo);
        if (file != null) {
          children.add(file);
        }
      }
    }
    catch (Exception e) {
      String f = "Erro ao obter as informaes (sga: %s, path: %s, erro: %s)";
      Server.logSevereMessage(String.format(f, name, path, e.getMessage()));
    }

    return children;
  }

  /**
   * Retorna {@link ClientSGAFile} equivalente ao dado path.
   * 
   * @param path path.
   * @return {@link ClientSGAFile} equivalente ao dado path.
   */
  public ClientSGAFile getFile(String path) {
    if (path == null) {
      return null;
    }

    try {
      SGAPath pathInfo = remoteReference.getPath(path);
      return buildClientSGAFile(pathInfo);
    }
    catch (Exception e) {
      String f = "Erro ao obter as informaes (sga: %s, path: %s, erro: %s)";
      Server.logSevereMessage(String.format(f, name, path, e.getMessage()));
      return null;
    }
  }

  /**
   * Constri um {@link ClientSGAFile} a partir de um path do sga.
   * 
   * @param pathInfo objeto que representa o path do sga.
   * @return {@link ClientSGAFile} equivalente ao path.
   */
  private ClientSGAFile buildClientSGAFile(SGAPath pathInfo) {
    if (!pathInfo.exists) {
      return null;
    }
    String separator = String.valueOf(info[0].getFileSeparator());

    ClientSGAFile file = new ClientSGAFile(name, pathInfo.path);
    file.setSeparator(separator);
    file.setDir(pathInfo.isDir);
    file.setSymbolicLink(pathInfo.isSymbolicLink);
    file.setCanRead(pathInfo.readable);
    file.setCanWrite(pathInfo.writable);
    file.setCanExecute(pathInfo.executable);
    file.setSize(((long) pathInfo.sizeKB) * 1024);
    return file;
  }

  /**
   * Retorna as informaes de um n do SGA.
   * 
   * @param index o ndice do n desejado (se for passado zero,  retornado o
   *        dado do n principal).
   * @return um objeto do tipo <code>SGAInfo</code> com as informaes.
   */
  final SGAInfo getInfo(final int index) {
    return info[index];
  }

  /**
   * Retorna as informaes de um n do SGA
   * 
   * @param hostname nome do n do SGA
   * @return informaes de um n do SGA
   */
  final SGAInfo getInfo(String hostname) {
    if (hostname == null) {
      return null;
    }
    for (SGAInfo node : info) {
      if (hostname.equals(node.getHostName())) {
        return node;
      }
    }
    return null;
  }

  /**
   * Retorna todas as informaes do SGA
   * 
   * @return um array de informaes.
   */
  final SGAInfo[] getAllInfo() {
    return info;
  }

  /**
   * Retorna o numero de maquinas do SGA
   * 
   * @return o nmero de mquinas.
   */
  final int getNumNodes() {
    return info.length;
  }

  /**
   * Notificao que o SSI entrou em shutdown.
   */
  void stop() {
    final String sgaName = getName();
    exitAcquisitionDataThread = true;
    Server.logInfoMessage("Terminando:" + sgaName);
    sgaService = null;
  }

  /**
   * Faz log de evento simples.
   * 
   * @param eventName nome do evento a ser sinalizado no arquivo.
   */
  private void logTriggeredEvent(final String eventName) {
    final String sgaName = getName();
    final String[] queue = { sgaName, "events" };
    logEvent(queue, eventName);
  }

  /**
   * Desativa o SGA.
   * 
   * @throws ServerException se houver falha na conexo com o SGA.
   */
  synchronized void shutdownSGA() throws ServerException {
    logTriggeredEvent("shutdown");
    final String sgaName = getName();
    Server.logWarningMessage("Shutdown para:" + sgaName);
    resetWatchDog();
    try {
      remoteReference.kill(SHUTDOWN);
    }
    catch (Exception e) {
      throw new ServerException("Falha na desativao do SGA " + this.name, e);
    }
    finally {
      setAlive(false);
    }
  }

  /**
   * Relanamento do SGA.
   * 
   * @throws ServerException se houver falha na conexo com o SGA.
   */
  final synchronized void restartSGA() throws ServerException {
    logTriggeredEvent("restart");
    final String sgaName = getName();
    Server.logWarningMessage("Restart para:" + sgaName);
    try {
      remoteReference.kill(RESTART);
    }
    catch (Exception e) {
      throw new ServerException("Falha na reativao do SGA " + this.name, e);
    }
    finally {
      setAlive(false);
    }
  }

  /**
   * Verifica se o SGA est apto a executar comandos.
   * 
   * @return true se o SGA pode executar comandos, false caso contrrio
   */
  final boolean getEnabled() {
    return enabled;
  }

  /**
   * Atualiza disponibilidade do SGA para a execu??o de comandos.
   * 
   * @param enabled true se o SGA pode executar comandos, false caso contrrio
   */
  final void setEnabled(boolean enabled) {
    this.enabled = enabled;
  }

  /**
   * Obtm o nome do SGA.
   * 
   * @return nome do SGA.
   */
  final String getName() {
    return this.name;
  }

  /**
   * Obtm a plataforma do SGA.
   * 
   * @return identificao da plataforma
   */
  final String getPlatformId() {
    return info[0].getPlatformId(); // XXX
  }

  /**
   * Verifica se uma plataforma  suportada pelo SGA.
   * 
   * @param platform o vetor de plataformas de pesquisa.
   * 
   * @return true se a plataforma for suportada, false caso contrrio.
   */
  final boolean isPlatformSupported(final Vector<String> platform) {
    for (SGAInfo node : info) {
      final String nodePlatform = node.getPlatformId();
      if (platform.contains(nodePlatform)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm o nome de um n de uma plataforma especfica
   * 
   * @param plats o vetor de plataformas.
   * 
   * @return identificao da plataforma
   */
  final String getHostOfPlatform(final Vector<String> plats) {
    for (SGAInfo node : info) {
      final String platform = node.getPlatformId();
      if (plats.contains(platform)) {
        return node.getHostName();
      }
    }
    return null;
  }

  /**
   * Obtm as capacidades dos sgas medidas atravs dos benchmarks.
   * 
   * @param capacityType Tipo de capacidade a ser obtida.
   * @return O valor da capacidade.
   */
  final long getCapacity(CapacityType capacityType) {
    return info[0].getCapacity(capacityType);
  }

  /**
   * Verifica se o SGA atualizou seus dados.
   * 
   * @return true se o SGA atualizou seus dados
   */
  final boolean getWatchDog() {
    return watchdog;
  }

  /**
   * Indica que o SGA atualizou seus dados.
   */
  final void patWatchDog() {
    watchdog = true;
  }

  /**
   * Indica que o SGA no atualizou seus dados.
   */
  final void resetWatchDog() {
    watchdog = false;
  }

  /**
   * Verifica se o SGA est acessvel.
   * 
   * @return true se o SGA est acessvel, false caso contrrio
   */
  final boolean isAlive() {
    return remoteReference != null;
  }

  /**
   * Atualiza acessibilidade do SGA. Isto invalida (zera) as informaes
   * dinmicas se o SGA estiver inacessvel.
   * 
   * @param alive true se o SGA estiver acessvel, false caso contrrio.
   */
  final void setAlive(final boolean alive) {
    if (alive) {
      return;
    }
    remoteReference = null;
    for (SGAInfo node : info) {
      node.setAlive(false);
    }
  }

  /**
   * Verifica se o SGA tem acesso a disco.
   * 
   * @return true se o SGA tem acesso, false caso contrrio
   */
  final boolean hasDiskAccess() {
    return hasDiskAccess;
  }

  /**
   * Informao dos jobs em execuo no SGA.
   * 
   * @return informao dos jobs em execuo no SGA.
   */
  final String getJobsInfo() {
    return jobsInfo;
  }

  /**
   * Indica se o SGA est apto a executar algum comando: est vivo, habilitado e
   * tem acesso ao disco.
   * 
   * @return true se o SGA est apto, false caso contrrio
   */
  public boolean mayExecuteCommand() {
    return (isAlive() && getEnabled() && hasDiskAccess());
  }

  /**
   * Verifica se o SGA  um cluster.
   * 
   * @return true se o SGA  um cluster, false se o SGA no  um cluster
   */
  final boolean isCluster() {
    return isCluster;
  }

  /**
   * Verifica (remotamente) se o SGA est acessvel.
   * 
   * @return true se o SGA est acessvel, false caso contrrio.
   */
  final synchronized boolean ping() {
    try {
      remoteReference.isAlive();
      setAlive(true);
    }
    catch (Exception e) {
      setAlive(false);
    }
    return isAlive();
  }

  /**
   * Obtm a referncia remota do SGA.
   * 
   * @return referncia remota
   */
  final SGAServer getRemoteReference() {
    return remoteReference;
  }

  /**
   * Armazena a (nova) referncia remota do SGA.
   * 
   * @param reference nova referncia remota do SGA
   */
  final synchronized void setRemoteReference(final SGAServer reference) {
    remoteReference = reference;
  }

  /**
   * Acumula dados de CPU e memria para gravao em arquivo.
   * 
   * @see SGAService
   * @see EventLogServiceInterface
   */
  final synchronized void logAuditEvents() {
    final DecimalFormatSymbols dfs = new DecimalFormatSymbols();
    dfs.setDecimalSeparator('.');
    final DecimalFormat df = new DecimalFormat("000.000");
    df.setDecimalFormatSymbols(dfs);
    final int numNodes = info.length;

    final String sgaName = getName();
    for (int i = 0; i < numNodes; i++) {
      final String hName = info[i].getHostName();
      final String[] cpuQueue = { sgaName, hName, "cpu-usage" };
      final double cpu = info[i].getCPULoad1();
      logEvent(cpuQueue, df.format(cpu));

      final String[] ramQueue = { sgaName, hName, "ram-usage" };
      final double ram = 100.0 - info[i].getRAMFreeMemory();
      logEvent(ramQueue, df.format(ram));

      final String[] swpQueue = { sgaName, hName, "swp-usage" };
      final double swp = 100.0 - info[i].getSwapFreeMemory();
      logEvent(swpQueue, df.format(swp));
    }
  }

  /**
   * Acumula dados do histrico de execuo dos algoritmos em arquivo.
   * 
   * @see SGAService
   * @see EventLogServiceInterface
   */
  final synchronized void logAlgorithmHistoryEvents() {
    final DecimalFormatSymbols dfs = new DecimalFormatSymbols();
    dfs.setDecimalSeparator('.');
    final DecimalFormat df = new DecimalFormat("000.000");
    df.setDecimalFormatSymbols(dfs);
    final int numNodes = info.length;

    final String sgaName = getName();
    for (int i = 0; i < numNodes; i++) {
      Enumeration<Command> eCmds = commandTable.elements();
      while (eCmds.hasMoreElements()) {
        Command cmd = eCmds.nextElement();
        CommandInfo cmdInfo;
        AlgorithmConfigurator algoConfigurator;
        try {
          cmdInfo = cmd.createCommandInfo();
          algoConfigurator = cmdInfo.getConfigurator();

          StringBuffer parametersString = new StringBuffer();
          HashMap<String, String> parameterTable =
            (HashMap<String, String>) algoConfigurator
              .getParameterValuesByName();
          for (String key : parameterTable.keySet()) {
            parametersString.append(key + "=" + parameterTable.get(key) + " ");
          }

          String algorithmName = algoConfigurator.getAlgorithmName();
          String algorithmVersion =
            algoConfigurator.getAlgorithmVersionId().toString();
          String[] historyQueue =
            { sgaName, "historic", algorithmName + "-" + algorithmVersion + "-" };

          final double cpuPerc =
            (cmdInfo.getCpuPerc() == null) ? SGA.NO_DATA : cmdInfo.getCpuPerc();
          final double swapPerc =
            (cmdInfo.getSwapMemoryPerc() == null) ? SGA.NO_DATA : cmdInfo
              .getSwapMemoryPerc();
          final double userTime =
            (cmdInfo.getUserTimeSec() == null) ? SGA.NO_DATA : cmdInfo
              .getUserTimeSec();
          final double systemTime =
            (cmdInfo.getSystemTimeSec() == null) ? SGA.NO_DATA : cmdInfo
              .getSystemTimeSec();
          final double ramMemory =
            (cmdInfo.getRAMMemoryMB() == null) ? SGA.NO_DATA : cmdInfo
              .getRAMMemoryMB();
          final double virtualMemory =
            (cmdInfo.getVirtualMemorySizeMB() == null) ? SGA.NO_DATA : cmdInfo
              .getVirtualMemorySizeMB();
          final double bytesIn =
            (cmdInfo.getBytesInKB() == null) ? SGA.NO_DATA : cmdInfo
              .getBytesInKB();
          final double bytesOut =
            (cmdInfo.getBytesOutKB() == null) ? SGA.NO_DATA : cmdInfo
              .getBytesOutKB();
          final double diskBytesRead =
            (cmdInfo.getDiskBytesReadKB() == null) ? SGA.NO_DATA : cmdInfo
              .getDiskBytesReadKB();
          final double diskBytesWrite =
            (cmdInfo.getDiskBytesWriteKB() == null) ? SGA.NO_DATA : cmdInfo
              .getDiskBytesWriteKB();
          final double cpuLoad = getCPULoad1();
          final double freeRamMemory = getRAMFreeMemory();
          final double cpuCapacity = getCapacity(CapacityType.CPU);
          final double diskReadCapacity = getCapacity(CapacityType.DISK_READ);
          final double diskWriteCapacity = getCapacity(CapacityType.DISK_WRITE);
          final double netCapacity = getCapacity(CapacityType.NET);

          String information =
            cmd.getId() + "; " + cmd.getUserId().toString() + "; "
              + parametersString + "; " + df.format(cpuLoad) + "; "
              + df.format(freeRamMemory) + "; " + df.format(cpuCapacity) + "; "
              + df.format(diskReadCapacity) + "; "
              + df.format(diskWriteCapacity) + "; " + df.format(netCapacity)
              + "; " + df.format(cpuPerc) + "; " + df.format(ramMemory) + "; "
              + df.format(swapPerc) + "; " + df.format(userTime) + "; "
              + df.format(systemTime) + "; " + df.format(virtualMemory) + "; "
              + df.format(bytesIn) + "; " + df.format(bytesOut) + "; "
              + df.format(diskBytesRead) + "; " + df.format(diskBytesWrite);

          logEvent(historyQueue, information);
        }
        catch (RemoteException e) {
          Server.logSevereMessage(
            "Falha na aquisio do configurador de comando!", e);
        }
      }
    }
  }

  /**
   * Obtm a carga na mquina hospedeira no ltimo minuto.
   * 
   * @return carga (percentual) no ltimo minuto.
   */
  final double getCPULoad1() {
    return info[0].getCPULoad1();
  }

  /**
   * Obtm o percentual de memria RAM livre.
   * 
   * @return percentual de memria RAM livre.
   */
  final double getRAMFreeMemory() {
    return 100.0 - info[0].getRAMFreeMemory();
  }

  /**
   * Recuperao de um comando.
   * 
   * @param cmdId identificador do comando
   * @return o objeto que representa o comando
   */
  final Command getCommand(final Object cmdId) {
    return commandTable.get(cmdId);
  }

  /**
   * Recuperao de todos os comandos que esto executando no SGA.
   * 
   * @return um array com os comandos em execuco (objetos Command)
   */
  final Command[] getAllCommands() {
    return commandTable.values().toArray(new Command[0]);
  }

  /**
   * Solicitao de execuo de um comando.
   * 
   * @param hostName n do SGA onde o comando deve ser executado. (somente para
   *        grid).
   * @param cmd informaes sobre o comando a ser executado.
   * @param expirationInfoDelay prazo de validade das informaes do comando.
   * @param asScript Indica se o comando dever ser executado como um script.
   * @param logCommandLineInfo Indica se informaes da linha de comando devem
   *        ser logadas.
   * @throws ProjectNotFoundException em caso de falha ao submeter o comando.
   * 
   */
  synchronized void executeCommand(final String hostName,
    final csbase.logic.CommandInfo cmd, long expirationInfoDelay,
    boolean asScript, boolean logCommandLineInfo)
    throws ProjectNotFoundException {
    final Object userId = Service.getUser().getId();
    final Command command =
      new Command(hostName, this, cmd, expirationInfoDelay, asScript,
        logCommandLineInfo);
    Object commId = command.getId();
    commandTable.put(commId, command);
    saveCommandTable();
    /*
     * Aps a criacao do comando, uma Thread  iniciada para solicitar que o
     * ambiente de execucao seja verificado/criado, em seguida o comando remoto
     *  disparado
     */
    commandExecutor.submit(new Runnable() {
      @Override
      public void run() {
        try {
          Service.setUserId(userId);
          command.executeRemoteCommand();
        }
        finally {
          Service.setUserId(null);
        }
      }
    });
  }

  /**
   * Grava (serializa) a tabela de comandos em arquivo.
   */
  protected void saveCommandTable() {
    ObjectOutput output = null;
    try {
      output =
        new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(
          getBackupFileName())));
      output.writeObject(commandTable);
    }
    catch (OperationFailureException ex) {
      Server.logSevereMessage(
        "Erro na validao do diretrio de persistncia de comandos.", ex);
    }
    catch (IOException ex) {
      Server.logSevereMessage("Erro ao gravar a tabela de comandos.", ex);
    }
    finally {
      try {
        if (output != null) {
          output.close();
        }
      }
      catch (IOException ex) {
        Server.logSevereMessage(
          "Erro ao fechar stream de gravao da tabela de histrico.", ex);
      }
    }
  }

  /**
   * Recuperao de comando
   * 
   * @param cmdId Identificador do comando.
   * @param cmdReference referncia remota do comando.
   * @return um indicativo de recuperao correta do comando.
   */
  final synchronized boolean commandRetrieved(final Object cmdId,
    final SGACommand cmdReference) {
    final Command command = getCommand(cmdId);
    if (command != null) {
      command.setRemoteReference(cmdReference);
      return true;
    }
    Server.logSevereMessage("Impossvel recuperar comando (" + cmdId + ")");
    return false;
  }

  /**
   * Finalizao de comando pelo usurio (kill)
   * 
   * @param cmdId identificador do comando.
   * @return {@code true} se o comando foi cancelado ou se no existe. Retorna
   *         {@code false} se ocorreu uma falha no cancelamento e ele no foi
   *         feito.
   */
  final synchronized boolean killCommand(final Object cmdId) {
    final Command command = getCommand(cmdId);
    if (command != null) {
      if (!command.kill()) {
        return false;
      }
      commandTable.remove(cmdId);
      saveCommandTable();
    }
    else {
      Server.logSevereMessage("Impossvel finalizar comando (" + cmdId + ")!");
    }
    return true;
  }

  /**
   * Finalizao de comando pelo administrador (kill). Remove o objeto do
   * comando mesmo que tenha ocorrido erro. Deve ser usado no caso em que no h
   * mais possibilidade do comando estar executando. Por exemplo, ocorreu
   * problema de hardware na mquina.
   * 
   * @param cmdId identificador do comando.
   */
  final synchronized void killCommandAnyway(final Object cmdId) {
    final Command command = getCommand(cmdId);
    if (command == null) {
      return;
    }
    command.kill();
    commandTable.remove(cmdId);
    saveCommandTable();
  }

  /**
   * Finalizao de um comando com falha.
   * 
   * @param cmdId identificador do comando
   * @param cause causa da falha no comando
   */
  final synchronized void failCommand(final Object cmdId,
    FailureFinalizationType cause) {
    final Command command = getCommand(cmdId);
    if (command != null) {
      commandTable.remove(cmdId);
      command.failed(cause);
      saveCommandTable();
      sgaService.handleCommandInitFailure(command);
    }
    else {
      Server.logSevereMessage("Comando nao encontrado (" + cmdId + ")!");
    }
  }

  /**
   * Finalizao de um comando.
   * 
   * @param cmdId identificador do comando
   */
  final synchronized void lostCommand(final String cmdId) {
    final Command command = getCommand(cmdId);
    if (command != null) {
      /*
       * primeiro remove o comando da tabela para no causar inconsistncia.
       */
      commandTable.remove(cmdId);
      command.lost();
      saveCommandTable();
    }
    else {
      Server.logSevereMessage("Comando nao encontrado (" + cmdId + ")!");
    }
  }

  /**
   * Finalizao de um comando.
   * 
   * @param cmdId identificador do comando
   */
  final synchronized void finishCommand(final String cmdId) {
    if (commandTable.remove(cmdId) != null) {
      saveCommandTable();
    }
    else {
      Server.logSevereMessage("Impossvel finalizar comando (" + cmdId + ")!");
    }
  }

  /**
   * Atualiza a representao de um SGA quando ocorre um "re-registro" (o SGA
   * havia cado e se registra novamente).
   * 
   * @param reference referncia remota para o SGA.
   * @param staticNodesInfo informaes estticas dos ns.
   * 
   * @throws ServerException se houver problemas com as propriedades do CSFS.
   */
  final void updateSGA(final SGAServer reference,
    StaticNodeInfo[] staticNodesInfo) throws ServerException {
    logTriggeredEvent("re-register");
    this.remoteReference = reference;
    this.requirements = new Hashtable<String, Boolean>();
    setStaticData(staticNodesInfo);
    this.watchdog = true;
    this.enabled = true;
    this.hasDiskAccess = true;
    this.isCluster = (info.length > 1) ? true : false;
    final Enumeration<Command> eCmds = commandTable.elements();
    while (eCmds.hasMoreElements()) {
      final Command cmd = eCmds.nextElement();
      cmd.setRemoteReference(null);
    }
    setCSFSProperties();
  }

  /**
   * Verifica (remotamente) se o SGA atende a um requisito. Faz uma cache de
   * requisitos e s busca no servidor a primeira vez.
   * 
   * @param requirement o requisito que se deseja verificar.
   * 
   * @return true se o SGA atende ao requisito, false caso contrrio
   * 
   * @throws ServerException se houver falha na conexo com o SGA.
   */
  final boolean meetsRequirement(final String requirement)
    throws ServerException {
    Boolean req = requirements.get(requirement);
    if (req == null) {
      boolean meets = false;
      try {
        meets = remoteReference.meetsRequirement(requirement);
      }
      catch (Exception e) {
        setAlive(false);
        throw new ServerException("Falha na recuperao do requisito "
          + requirement + " junto ao SGA " + this.name, e);
      }
      req = new Boolean(meets);
      requirements.put(requirement, req);
    }
    return req.booleanValue();
  }

  /**
   * Verifica (remotamente) se o SGA executa o benchmark de rede durante a
   * inicializao.
   * 
   * @return 0 se o SGA deve calcular a capacidade de transferncia. 1 se o SGA
   *         apresenta grande capacidade de transferncia. 2 se o SGA apresenta
   *         pequena capacidade de transferncia
   * 
   * @throws ServerException se houver falha na conexo com o SGA.
   */
  final int execNetBench() throws ServerException {
    if (execBench == null) {
      try {
        if (remoteReference.execNetBench() == SGANetCapacity.ALL) {
          execBench = new Integer(SGAInfo.ALL_CAPACITY);
        }
        else {
          if (remoteReference.execNetBench() == SGANetCapacity.NO) {
            execBench = new Integer(SGAInfo.NO_CAPACITY);
          }
          else {
            execBench = new Integer(SGAInfo.CALC_CAPACITY);
          }
        }
      }
      catch (Exception e) {
        setAlive(false);
        throw new ServerException(
          "Falha na recuperao do requisito benchmark "
            + "de rede junto ao SGA " + this.name, e);
      }
    }
    return execBench.intValue();
  }

  /**
   * Preenche uma tabela que mapeia um tipo de capacidade encontrado na idl do
   * SGA ao tipo de capacidade definida na classe Java CapacityType. Essa tabela
   * deve ser preenchida uma nica vez visto que se um novo tipo de capacidade
   * for adicionada/removida dos SGAs, novos stubs devem ser gerados.
   * 
   * @return Mapa que relaciona uma capacidade definida na idl  capacidade Java
   *         do SGA.
   */
  private HashMap<Integer, CapacityType> fillCapacityMap() {
    capacityMap = new HashMap<Integer, CapacityType>();

    CType fieldEx = CType.CPU;
    for (CapacityType type : CapacityType.values()) {
      try {
        Field field = fieldEx.getClass().getField(type.toString());
        CType cType = (CType) field.get(field);
        capacityMap.put(Integer.valueOf(cType.value()), type);
      }
      catch (NoSuchFieldException e) {
        Server.logSevereMessage("Tipo de capacidade " + type.toString()
          + " invlida.\n" + "Erro: " + e);
        return null;
      }
      catch (IllegalAccessException e) {
        Server.logSevereMessage("Capacidade " + type.toString()
          + " inacessvel.\n" + "Erro: " + e);
        return null;
      }
    }
    return capacityMap;
  }

  /**
   * Constri a representao de um SGA.
   * 
   * @param srv o servio de SGAs.
   * @param sgaName nome do SGA
   * @param reference referncia remota para o SGA
   * @param staticNodesInfo informaes estticas dos ns
   * @throws ServerException se no  possvel obter a configurao do SGA
   */
  SGA(final SGAService srv, final String sgaName, final SGAServer reference,
    StaticNodeInfo[] staticNodesInfo) throws ServerException {
    this.remoteReference = reference;
    this.sgaService = srv;
    this.requirements = new Hashtable<String, Boolean>();
    this.name = sgaName;
    // Antes de chamar setStaticData  preciso mapear as capacidades na idl 
    // para a classe java atravs do mtodo fillCapacityMap.
    if (capacityMap == null) {
      capacityMap = fillCapacityMap();
    }
    setStaticData(staticNodesInfo);
    logTriggeredEvent("register");
    this.watchdog = true;
    this.enabled = true;
    this.isCluster = (info.length > 1) ? true : false;
    this.hasDiskAccess = true;
    this.jobsInfo = null;
    loadCommandTable();
    setCSFSProperties();
    createCommandDataAcquisitionThread();
    // Cria um executor que usa uma thread nica para submisso de comandos.
    this.commandExecutor = Executors.newSingleThreadExecutor();
  }

  /**
   * Atualiza o estado dos comandos. Precisa atualizar os comandos mesmo que o
   * SGA esteja sem conexo, caso contrrio, os dados dos comandos ficam
   * invlidos.
   * 
   * @return o nmero de vezes que os comandos foram atualizados.
   */
  private long updateCommands() {
    long minupdate = Long.MAX_VALUE;
    Command[] sgaCmds = getAllCommands();
    for (Command cmd : sgaCmds) {
      try {
        if (cmd.isAlive()) {
          cmd.update();
          long nupdates = cmd.getNUpdates();
          minupdate = Math.min(minupdate, nupdates);
        }
      }
      catch (ServerException e) {
        String message = e.getMessage();
        Server.logSevereMessage(message, e);
      }
    }
    return minupdate;
  }

  /**
   * Criao da thread que faz a aquisio dos dados dos Comandos nos SGAs
   */
  private void createCommandDataAcquisitionThread() {
    /*
     * Faz um update de freqencia incremental de STEP_SECONDS segundos, at
     * atingir o tempo definido em updateSleepTime. O disparo de um novo
     * comando, faz a freqencia do update voltar para STEP_SECONDS. Isso faz
     * com que os primeiros updates de comandos ocorram com uma frequncia
     * maior, para dar um feedback mais rpido para o usurio.
     */
    final int STEP_SECONDS = 2;
    final int sgaCommandUpdateInterval = sgaService.getCommandsUpdateInterval();
    final long updateSleepTime = sgaCommandUpdateInterval * MILLIS;
    Thread acquisitionCommandDataThread = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          while (!exitAcquisitionDataThread) {
            long stepSleepTime = Long.MAX_VALUE;
            long ntimes = updateCommands();
            if (ntimes != Long.MAX_VALUE) {
              stepSleepTime = (ntimes + 1) * STEP_SECONDS * MILLIS; // Estoura long?
            }
            stepSleepTime = Math.min(stepSleepTime, updateSleepTime);
            try {
              Thread.sleep(stepSleepTime);
            }
            catch (InterruptedException ie) {
              Server
                .logWarningMessage("Aquisio de dados de Comandos SGA interrompido!");
            }
          }
          Server
            .logWarningMessage("Finalizando thread de aquisio de dados de comandos...");
        }
        catch (Throwable t) {
          String msg =
            (new Date()) + " - Erro na aquisio de dados de comandos:";
          System.err.println(msg);
          t.printStackTrace(System.err);
          Server.logSevereMessage(msg, t);
        }
      }
    });
    exitAcquisitionDataThread = false;
    acquisitionCommandDataThread.setName(this.getClass().getSimpleName() + "::"
      + "AcquisitionCommandDataThread");
    acquisitionCommandDataThread.start();
  }

  /**
   * Cria uma tabela vazia e a persiste.
   */
  private void createNewCommandTable() {
    this.commandTable = new Hashtable<Object, Command>();
    saveCommandTable();
  }

  /**
   * Carrega a tabela de comandos, normalmente persistida em arquivo. Caso o
   * arquivo no seja encontrado, uma tabela vazia  persistida.
   */
  @SuppressWarnings("unchecked")
  private void loadCommandTable() {
    Hashtable<Object, Command> table = null;
    ObjectInput input = null;
    try {
      File file = new File(getBackupFileName());
      if (!file.exists()) {
        createNewCommandTable();
        return;
      }
      input =
        new ObjectInputStream(
          new BufferedInputStream(new FileInputStream(file)));
      table = (Hashtable<Object, Command>) input.readObject();
      try {
        for (Command command : table.values()) {
          Service.setUserId(command.getUserId());
          command.setService(sgaService);
          command.setSGA(this);
        }
      }
      finally {
        Service.setUserId(null);
      }
      this.commandTable = table;
    }
    catch (OperationFailureException ex) {
      Server.logSevereMessage(
        "Erro na validao do diretrio de persistncia de comandos.", ex);
      createNewCommandTable();
      return;
    }
    catch (IOException ex) {
      Server.logSevereMessage("Erro ao ler a tabela de comandos.", ex);
      createNewCommandTable();
      return;
    }
    catch (ClassNotFoundException ex) {
      Server.logSevereMessage(
        "No foi encontrada nenhuma classe no arquivo especificado.", ex);
      createNewCommandTable();
      return;
    }
    finally {
      try {
        if (input != null) {
          input.close();
        }
      }
      catch (IOException ex) {
        Server.logSevereMessage(
          "Erro ao fechar stream de leitura da tabela de histrico.", ex);
        createNewCommandTable();
        return;
      }
    }
  }

  /**
   * Retorna o nome do arquivo para backup de comandos em execuo.<br>
   * Formato: &lt;diretrio de comandos&gt;+&lt;nome do sga&gt;+&lt;sufixo&gt;.
   * 
   * @return nome do arquivo para backup de comandos em execuo.
   * 
   * @throws OperationFailureException em caso de erro na validao do diretrio
   *         de persistncia no sistema de arquivos.
   */
  private String getBackupFileName() throws OperationFailureException {
    return sgaService.getCommandsPersistencyDirectoryName() + File.separator
      + name + EXECUTION_COMMANDS_FILENAME_SUFFIX;
  }

  /**
   * Verifica se o caminho especificado existe no sistema de arquivos sobre o
   * qual o SGA atua.
   * 
   * @param path caminho relativo ao diretrio de execuo do SGA ou absoluto
   *        (contanto que seja um caminho do sistema de arquivos do SGA e no do
   *        servidor).
   * 
   * @return <code>true</code> caso o caminho seja encontrado.
   * 
   * @throws ServerException se houver falha no acesso ao SGA
   */
  boolean isDir(String path) throws ServerException {
    try {
      SGAPath sgaPath = remoteReference.getPath(path);
      return sgaPath.isDir;
    }
    catch (Exception e) {
      setAlive(false);
      throw new ServerException("Falha na verificao do acesso ao caminho '"
        + path + "' junto ao SGA " + this.name, e);
    }
  }

  /**
   * Indica se o SGA utiliza algum mecanismo de transferncia de arquivos.
   *
   * @return true se utiliza algum mecanismo de transferncia de arquivos e
   * false se no utiliza.
   */
  public boolean hasDataTransferMechanism() {
    return getCSFSHost() != null;
  }
}
