package csbase.server.services.repositoryservice;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.FileChannel;
import java.util.List;

/**
 * Interface que define um arquivo de repositrio. Os arquivos de um repositrio
 * s so acessveis via um repositrio do tipo {@link IRepository}. Alm disso,
 * essa interface abstrai como  a representao fsica de um arquivo de
 * repositrio.
 *
 * @see IRepository
 * @see RepositoryService
 *
 * @author Tecgraf/PUC-Rio
 */
public interface IRepositoryFile extends Closeable {

  /**
   * Nome do arquivo.
   *
   * @return nome.
   */
  String getName();

  /**
   * Lista de arquivos deste diretrio.
   *
   * @return lista de arquivos.
   */
  List<IRepositoryFile> getChildren();

  /**
   * Lista arquivo filho dado o caminho.
   *
   * @param path caminho.
   *
   * @return arquivo filho.
   */
  IRepositoryFile getChild(String... path);

  /**
   * Verifica se o arquivo existe.
   *
   * @return true se o arquivo existe, false caso contrrio.
   */
  boolean exists();

  /**
   * Data/hora da ltima modificao do arquivo.
   *
   * @return data da ltima modificao.
   */
  long getModificationDate();

  /**
   * Diretrio-pai
   *
   * @return diretrio que contem este arquivo.
   */
  IRepositoryFile getParent();

  /**
   * Obtm o caminho do arquivo relativo ao repositrio.
   *
   * @return o caminho do arquivo relativo ao repositrio.
   */
  String getPath();

  /**
   * Obtm o caminho completo para o arquivo.
   * 
   * @return o caminho completo para o arquivo.
   */
  String getFullPath();

  /**
   * Tamanho do arquivo em bytes.
   *
   * @return tamanho do arquivo.
   */
  long size();

  /**
   * Verifica se  diretrio.
   *
   * @return true se for diretrio, false caso contrrio.
   */
  boolean isDirectory();

  /**
   * Verifica se  um arquivo normal;
   *
   * @return true se for um arquivo normal, false caso contrrio.
   */
  boolean isRegularFile();

  /**
   * Verifica se  um arquivo oculto.
   *
   * @return true se for um arquivo oculto, false caso contrrio.
   */
  boolean isHidden();

  /**
   * Verifica se  um link simblico.
   *
   * @return true se for um link simblico, false caso contrrio.
   */
  boolean isSymbolicLink();

  /**
   * Verifica a permisso de leitura.
   *
   * @return true se for possvel fazer a leitura, false caso contrrio.
   */
  boolean isReadable();

  /**
   * Verifica a permisso de escrita.
   *
   * @return true se for possvel fazer a escrita, false caso contrrio.
   */
  boolean isWritable();

  /**
   * Verifica a permisso de execuo.
   *
   * @return true se for possvel executar, false caso contrrio.
   */
  boolean isExecutable();

  /**
   * Bloqueia o arquivo.
   *
   * TODO Esse mtodo de lock ainda est em construo, a ideia  receber um
   * parmetro que especifica o tipo de lock (exclusivo, compartilhado, leitura,
   * escrita, etc)
   *
   * @return true se o arquivo for bloqueado, false caso contrrio.
   */
  boolean lock();

  /**
   * Libera o bloqueio do arquivo.
   *
   * TODO Esse mtodo de lock ainda est em construo, a ideia  receber um
   * parmetro que especifica o tipo de lock (exclusivo, compartilhado, leitura,
   * escrita, etc)
   *
   * @return true se o arquivo foi liberado, false caso contrrio.
   */
  boolean release();

  /**
   * Verifica se o arquivo est bloqueado.
   *
   * TODO Esse mtodo de lock ainda est em construo, a ideia  receber um
   * parmetro que especifica o tipo de lock (exclusivo, compartilhado, leitura,
   * escrita, etc)
   *
   * @return true se o arquivo estiver bloqueado, false caso contrrio.
   */
  boolean isLocked();

  /**
   * Fluxo de entrada para leitura do arquivo.
   *
   * @return fluxo de entrada para leitura do arquivo.
   * @throws IOException Caso ocorra erro de IO.
   */
  InputStream getInputStream() throws IOException;

  /**
   * Fluxo de sada para escrita no arquivo.
   *
   * @return fluxo de sada para escrita no arquivo.
   * @throws IOException caso ocorra erro de IO.
   */
  OutputStream getOutputStream() throws IOException;

  /**
   * Obtm um canal para leitura e escrita no arquivo.
   *
   * @param mode modo de acesso ao arquivo.
   * @return Canal para leitura e escrita no arquivo.
   * @throws IOException em caso de falha na operao.
   */
  FileChannel getFileChannel(String mode) throws IOException;

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws IOException;
}
