package csbase.server.services.repositoryservice;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.util.LinkedList;
import java.util.List;

import csbase.server.FileSystem;
import csbase.server.Server;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Arquivo localizado fisicamente na mquina do servidor.
 *
 * NOTA: Essa classe  de visibilidade restrita ao pacote, pois s o
 * {@link RepositoryService} pode manipular suas instncias.
 *
 * @see LocalRepository
 *
 * @author Tecgraf/PUC-Rio
 */
class LocalFile implements IRepositoryFile {

  /** Arquivo. */
  private File file;

  /** Arquivo para acesso aleatrio. */
  private RandomAccessFile raf;

  /** Repositrio. */
  private LocalRepository repository;

  /** Representa o path relativo ao repositrio */
  private String relativePath;

  /**
   * Construtor.
   *
   * @param file arquivo.
   * @param repository repositrio.
   * @param relativePath caminho relativo ao repositrio.
   */
  LocalFile(File file, LocalRepository repository, String relativePath) {
    if (file == null) {
      throw new IllegalArgumentException("file no pode ser nulo.");
    }
    if (repository == null) {
      throw new IllegalArgumentException("repository no pode ser nulo.");
    }
    this.file = file;
    this.repository = repository;
    this.relativePath = relativePath;
  }

  /** {@inheritDoc} */
  @Override
  public String getName() {
    return file.getName();
  }

  /** {@inheritDoc} */
  @Override
  public List<IRepositoryFile> getChildren() {
    if (isRegularFile()) {
      return null;
    }

    List<IRepositoryFile> result = new LinkedList<IRepositoryFile>();
    for (File child : file.listFiles()) {
      String childPath = FileUtils.joinPath(File.separatorChar, relativePath,
        child.getName());
      result.add(new LocalFile(child, repository, childPath));
    }
    return result;
  }

  /** {@inheritDoc} */
  @Override
  public IRepositoryFile getChild(String... path) {
    String filePath = FileUtils.joinPath(File.separatorChar, path);
    filePath = FileUtils.joinPath(File.separatorChar, getPath(), filePath);
    return repository.getFile(filePath);
  }

  /** {@inheritDoc} */
  @Override
  public boolean exists() {
    return file.exists();
  }

  /** {@inheritDoc} */
  @Override
  public long getModificationDate() {
    return file.lastModified();
  }

  /** {@inheritDoc} */
  @Override
  public IRepositoryFile getParent() {
    File parentFile = file.getParentFile();
    if (parentFile == null) {
      return null;
    }
    if (parentFile.getPath().equals(repository.getURI())) {
      return null;
    }
    return new LocalFile(parentFile, repository, FileUtils.getFilePath(
      relativePath));
  }

  /** {@inheritDoc} */
  @Override
  public String getPath() {
    return relativePath;
  }

  /** {@inheritDoc} */
  @Override
  public long size() {
    return file.length();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isRegularFile() {
    return file.isFile();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isHidden() {
    return file.isHidden();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isSymbolicLink() {
    return false;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isReadable() {
    return file.canRead();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isWritable() {
    return file.canWrite();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isExecutable() {
    return file.canExecute();
  }

  /** {@inheritDoc} */
  @Override
  public boolean lock() {
    return repository.lock(this);
  }

  /** {@inheritDoc} */
  @Override
  public boolean release() {
    return repository.release(this);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isLocked() {
    return repository.isLocked(this);
  }

  /** {@inheritDoc} */
  @Override
  public InputStream getInputStream() throws IOException {
    return new FileInputStream(file);
  }

  /** {@inheritDoc} */
  @Override
  public OutputStream getOutputStream() throws IOException {
    if (isLocked()) {
      throw new IOException(String.format("Arquivo bloqueado: %s", this));
    }
    return new FileOutputStream(file);
  }

  /** {@inheritDoc} */
  @Override
  public FileChannel getFileChannel(String mode) throws IOException {
    if (raf == null) {
      raf = new RandomAccessFile(file, mode);
    }
    return raf.getChannel();
  }

  /** {@inheritDoc} */
  @Override
  public void close() throws IOException {
    if (raf != null) {
      raf.close();
    }
    raf = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return file.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getFullPath() {
    return file.getAbsolutePath();
  }

  /**
   * Copia um arquivo regular. Faz a cpia do arquivo.
   *
   * @param newFile a cpia do arquivo.
   * @return true se foi feita com sucesso, false caso contrrio.
   */
  boolean copyRegularFile(File newFile) {
    /* Copia o arquivo */
    if (!FileSystem.copyFile(file, newFile)) {
      String errMsg = "ServerProjectFile:copyFile: erro na cpia fsica de "
        + file.getAbsolutePath() + " para " + newFile.getAbsolutePath();
      Server.logWarningMessage(errMsg);
      return false;
    }
    return true;
  }
}
