package csbase.server.services.csfsservice;

import java.lang.reflect.Field;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.omg.CORBA.ORB;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.OperationFailureException;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csfs.remote.FileAlreadyExistsException;
import csfs.remote.FileNotFoundException;
import csfs.remote.FileServer;
import csfs.remote.FileServerHelper;
import csfs.remote.InvalidPathException;
import csfs.remote.InvalidStateException;
import csfs.remote.NotDirectoryException;
import csfs.remote.NotFileException;
import csfs.remote.RemoteFile;

/**
 *
 * Servio para cpia de arquivos utilizando o CSFS Daemon.
 *
 * O CSFS Daemon exporta uma interface remota que d acesso ao sistema de
 * arquivos local da mquina que executa o servidor. Essa interface pode ser
 * usada para copiar arquivos entre diversas mquinas e para permitir o
 * gerenciamento dos mesmos.
 *
 * No caso do CSFSService, temos a cpia de arquivos entre o CSFS Daemon que
 * exporta a rea de projetos (e o repositrio de algoritmos) e o CSFS Daemon
 * que gerencia a area local nas mquinas de execuo. Esse procedimento de
 * cpia  realizado pela comunicao direta entre esses dois CSFS Daemons.
 *
 * O CSFS Daemon  definido em CORBA.
 *
 * @author Marcelo Nery dos Santos
 *
 */
public class CSFSService extends Service {

  /**
   * Mount point para onde  mapeado o repositrio de projetos.
   */
  private static final String[] CFSF_PROJECT_MOUNT_POINT =
    new String[] { "project_csfs" };

  /**
   * Mount point para onde  mapeado o repositrio de algoritmos.
   */
  private static final String[] CFSF_ALGORITHM_MOUNT_POINT =
    new String[] { "algorithms_csfs" };

  /** Uma instncia para o ORB */
  private ORB orb;

  /**
   * O formato do corbaloc utilizado pelo ORB (remoto) do CSFS Daemon. Esse
   * valor  dependente do ORB (fornecedor/versao) e pode ser configurado pelo
   * arquivo de configurao do servio.
   */
  private String corbalocPattern;

  /** A porta onde o objeto remoto  iniciado. */
  private int csfsPort;

  /** Uma referncia para o servidor de arquivos que acessa a rea do SSI. */
  private String csfsHost;

  /** Uma referncia para o servidor de arquivos que acessa a rea do SSI. */
  private FileServer mainFileServer;

  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "CSFSService";

  /** O tipo de mtodo de transferncia de arquivos a ser utilizado */
  private String transferMethod;

  /**
   * Verifica se o servio est habilitado/disponvel. Se no estiver, lanca uma
   * exceo.
   *
   * @throws ServerException caso o servio no esteja disponvel.
   */
  private void checkActive() throws ServerException {
    if (!isActive()) {
      throw new ServerException("Servio CSFS no disponvel");
    }
  }

  /**
   * Retorna a instncia do servio.
   *
   * @return o servio.
   */
  public static CSFSService getInstance() {
    return (CSFSService) getInstance(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    // Esse servico nao possui observadores.
    return false;
  }

  /**
   * Constri a instncia do servio.
   *
   * @throws ServerException em caso de erro na criao d servio.
   */
  @SuppressWarnings("unused")
  public static void createService() throws ServerException {
    new CSFSService();
  }

  /**
   * Construtor
   *
   * @throws ServerException em caso de erro na construo do servio.
   */
  protected CSFSService() throws ServerException {
    super(SERVICE_NAME);
    this.setEnabled(getBooleanProperty("ENABLED"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    this.corbalocPattern = getStringProperty("CSFS_CORBALOC_PATTERN");
    this.csfsPort = getIntProperty("CSFS_PORT");
    this.csfsHost = getStringProperty("CSFS_HOST");
    final int retrySleepTime = getIntProperty("RETRY_SLEEP_TIME");
    this.transferMethod = getStringProperty("TRANSFER_METHOD");

    /*
     * O sistema foi desenvolvido e testado com o uso do JacORB. As classes
     * contidas no jar do csfs possuem as classes que foram compiladas com o
     * compilador de IDL do mesmo. Assim, configuramos a mquina virtual Java
     * para utilizar o JacORB.
     */
    final Properties props = new Properties();
    props.setProperty("org.omg.CORBA.ORBClass", "org.jacorb.orb.ORB");
    props.setProperty("org.omg.CORBA.ORBSingletonClass",
      "org.jacorb.orb.ORBSingleton");

    //Propriedade preventiva para evitar bug de no recuperao aps queda.
    props.setProperty(
      "jacorb.connection.client.disconnect_after_systemexception", "false");

    try {
      orb = ORB.init((String[]) null, props);
    }
    catch (Exception exc) {
      Server.logSevereMessage("Falha na inicializao do orb.", exc);
      throw new ServerException("ORB.init() failed!");
    }

    /*
     * Inicia uma thread para tentar recuperar uma referncia para o objeto
     * remoto que exporta a arvore de arquivos do SSI. A princpio, esse objeto
     * est na mesma maquina do SSI. Isso pode ser alterado para fazer com que a
     * rea de projetos ou o repositrio de algoritmos sejam procurados em outra
     * mquina. A implementao atual supoe que o CSFS Daemon foi iniciado
     * externamente na mesma mquina do SSI.
     *
     * Podemos avaliar o caso onde o proprio CSFSService incorpora o CSFS
     * Deamon. Ou seja, podemos fazer uma implementao que faa com que o
     * initService dispare o servidor principal do CSFS.
     */
    Thread csfsRemoteLocatorThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!isConnected()) {
          try {
            mainFileServer = getFileServer(csfsHost, csfsPort);
            Server.logInfoMessage("Recuperei o mainFileServer " + csfsHost);
            // Remove mount points anteriores
            removeMountPoints();

            final String projectDirProperty =
              getStringProperty("PROJECT_DIR_FROM_ROOT");
            final String algorithmsDirProperty =
              getStringProperty("ALGORITHM_DIR_FROM_ROOT");

            final String[] projectDir = FileUtils.splitPath(projectDirProperty);
            final String[] algorithmsDir =
              FileUtils.splitPath(algorithmsDirProperty);

            addMountPoint(CFSF_PROJECT_MOUNT_POINT, projectDir);
            addMountPoint(CFSF_ALGORITHM_MOUNT_POINT, algorithmsDir);
          }
          catch (ServerException exc) {
            Server
            .logFineMessage("No consegui recuperar o mainFileServer (exc="
              + exc.getMessage() + "). Vou tentar novamente em "
              + retrySleepTime + " ms...");
          }
          catch (OperationFailureException e) {
            Server.logSevereMessage(
              "No foi possvel montar os diretrios exportados pelo CSFS:", e);
          }
          catch (Exception e) {
            Server.logSevereMessage("Exceo ao iniciar o servio CSFS:", e);
          }
          try {
            Thread.sleep(retrySleepTime);
          }
          catch (InterruptedException e) {
            Server.logSevereMessage("InterruptedException", e);
          }
        }
      }

      private RemoteFile addMountPoint(String[] mountPoint, String[] target)
        throws OperationFailureException {
        RemoteFile targetDir;
        try {
          RemoteFile root = mainFileServer.getRoot();
          targetDir = root.getChild(target);
          root.addMountPoint(mountPoint, targetDir);
          return targetDir;
        }
        catch (Exception e) {
          Server.logSevereMessage("No foi possvel montar o diretrio "
            + FileUtils.joinPath(target) + " em "
            + FileUtils.joinPath(mountPoint), e);
          throw new OperationFailureException(e);
        }
      }
    });
    csfsRemoteLocatorThread.setName(this.getClass().getSimpleName() + "::"
      + "CSFSRemoteLocatorThread");
    csfsRemoteLocatorThread.start();
  }

  /**
   * Mtodo invocado no termino do servio. Realiza a shutdown do ORB.
   */

  @Override
  public void shutdownService() throws ServerException {
    if (isConnected()) {
      removeMountPoints();
    }
    orb.shutdown(false);
  }

  /**
   * Obtm a porta utilizada pelo csfs.
   *
   * @return a porta utilizada pelo csfs.
   */
  public int getPort() {
    return csfsPort;
  }

  /**
   * Obtm o host utilizado pelo csfs.
   *
   * @return o host utilizado pelo csfs.
   */
  public String getHost() {
    return csfsHost;
  }

  /**
   * Obtm o CORBALOC utilizado pelo csfs.
   *
   * @return o CORBALOC utilizado pelo csfs.
   */
  public String getCorbalocPattern() {
    return corbalocPattern;
  }

  /**
   * Remove os mount points utilizados pelo CSFS.
   *
   * @throws ServerException em caso de erro.
   */
  private void removeMountPoints() throws ServerException {
    if (checkExistence(csfsHost, csfsPort, CFSF_PROJECT_MOUNT_POINT)) {
      removeMountPoint(CFSF_PROJECT_MOUNT_POINT);
    }

    if (checkExistence(csfsHost, csfsPort, CFSF_ALGORITHM_MOUNT_POINT)) {
      removeMountPoint(CFSF_ALGORITHM_MOUNT_POINT);
    }
  }

  /**
   * Remove o mount point no caminho especificado.
   *
   * @param mountPoint O caminho para o mountPoint.
   */
  private void removeMountPoint(String[] mountPoint) {
    try {
      RemoteFile root = mainFileServer.getRoot();
      root.removeMountPoint(mountPoint);
    }
    catch (Exception e) {
      Server.logWarningMessage("No foi remover o mount point em "
        + FileUtils.joinPath(mountPoint));
    }
  }

  /**
   * Determina se o servio est conectado a um servidor de arquivos CSFS.
   *
   * @return verdadeiro se o servio est conectado ao servidor ou falso, caso
   *         contrrio.
   */
  private boolean isConnected() {
    return this.mainFileServer != null;
  }

  /**
   *
   * Verifica a existncia de um caminho em um determinado servidor.
   *
   * @param host o servidor que deve ser consultado
   * @param port porta para acessar o servidor.
   * @param path o caminho absoluto do arquivo (relativo a raiz exportada pelo
   *        servidor)
   * @return a indicao de existncia
   * @throws ServerException em caso de erro.
   */
  public boolean checkExistence(String host, int port, String[] path)
    throws ServerException {
    checkActive();
    boolean response = false;
    FileServer fs = getFileServer(host, port);
    try {
      fs.getRoot().getChild(path);
      response = true;
    }
    catch (FileNotFoundException exc) {
      // a resposta j foi inicializada como false
    }
    catch (csfs.remote.ServerException exc) {
      Server.logSevereMessage("Error in CSFS subsystem!", exc);
      throw new ServerException("Remote File Server Exception. message="
        + exc.message, exc);
    }
    catch (Exception exc) {
      throw new ServerException("Exception during checkExistence operation."
        + " exc.getClass().getName()=" + exc.getClass().getName() + " file="
        + FileUtils.joinPath(getName(exc)), exc);
    }
    return response;
  }

  /**
   * Retorna o caminho absoluto para o diretrio raiz do CSFS para o servidor
   * especificado.
   *
   * @param host nome do servidor que exporta um sistema de arquivos via CSFS.
   * @param port porta para acessar o servidor.
   *
   * @return diretrio raiz do CSFS.
   *
   * @throws ServerException em caso de erro.
   */
  public String getRootPath(String host, int port) throws ServerException {
    checkActive();
    FileServer fs = getFileServer(host, port);
    try {
      return fs.getProperty("ROOT_DIR");
    }
    catch (csfs.remote.ServerException exc) {
      throw new ServerException("Exception during getRootPath operation."
        + " exc.getClass().getName()=" + exc.getClass().getName() + " file="
        + FileUtils.joinPath(getName(exc)), exc);
    }
  }

  /**
   * Obtm o caminho para o repositrio de projetos visto pelo CSFS.
   *
   * @return o caminho para o repositrio de projetos.
   * @throws ServerException caso o servio no esteja disponvel
   */
  public String[] getCSFSProjectsRootPath() throws ServerException {
    checkActive();
    return CFSF_PROJECT_MOUNT_POINT;
  }

  /**
   * Obtm o caminho para o repositrio de algoritmos visto pelo CSFS.
   *
   * @return o caminho para o repositrio de algoritmos.
   * @throws ServerException caso o servio no esteja disponvel
   */
  public String[] getCSFSAlgorithmsRootPath() throws ServerException {
    checkActive();
    return CFSF_ALGORITHM_MOUNT_POINT;
  }

  /**
   * Recupera uma referncia para um objeto remoto, baseado no servidor
   * especificado.
   *
   * @param host o servidor que exporta um sistema de arquivos pelo CSFS.
   * @param port porta para acessar o servidor.
   * @return a referncia para o objeto remoto.
   *
   * @throws ServerException em caso de erro.
   */
  private FileServer getFileServer(String host, int port)
    throws ServerException {

    /*
     * Optamos por ter uma configurao para todas os servidores do CSFS. Dessa
     * forma, a porta no precisa ser passada como parmetro e a porta padro 
     * sempre utilizada na recuperao do objeto remoto. Esse mtodo pode ser
     * modificado para receber tambm o parmetro da porta.
     */
    String corbaloc =
      MessageFormat.format(corbalocPattern, new Object[] { host, "" + port });
    FileServer response;
    try {
      response = FileServerHelper.narrow(orb.string_to_object(corbaloc));
      response.getName(); // realizando uma chamada remota (sanity-check).
    }
    catch (Exception exc) {
      throw new ServerException(
        "Error contacting remote file server! corbaloc=" + corbaloc + "\n", exc);
    }
    return response;
  }

  /**
   *
   * Obtm um mapeamento entre um arquivo e sua data de modificao para todos
   * os arquivos de uma rvore de diretrios, dentro do servidor especificado.
   * Esse valor  obtido sobre o arquivo remoto, o que evita problemas de
   * sincronismo de relgios em uma rede composta por vrias mquinas.
   *
   * Esse mtodo  a base para permitir que uma rvore de diretrios seja
   * comparada antes e depois da execuo de um algoritmo, fazendo com que a
   * sincronizao entre a area de armazenamento no SGA e a rea de projetos
   * seja feita de uma forma otimizada. Por outro lado, ele pode ter um custo
   * elevado, dado a operao recursiva sobre a rvore remota.
   *
   * @param host o servidor que mantem a rvore
   * @param port porta para acessar o servidor
   * @param path a raiz que deve ser analisada
   * @return mapeamento entre um arquivo e sua data de modificao para todos os
   *         arquivos de uma rvore de diretrios.
   *
   * @throws ServerException em caso de erro.
   */
  public Map<String[], Long> getTimestamps(String host, int port, String[] path)
    throws ServerException {
    checkActive();
    Map<String[], Long> response = new HashMap<String[], Long>();
    try {
      FileServer targetFS = getFileServer(host, port);
      RemoteFile file = lookup(targetFS, path);
      visitRemoteFilesGettingTimestamps(response, file);
    }
    catch (FileNotFoundException exc) {
      throw new ServerException(
        "Unable to get timestamps for an unexistent path. path=" + path
        + " address=" + host + ":" + port, exc);
    }
    return response;
  }

  /**
   * Obtm um mapeamento entre um arquivo e sua data de modificao para todos
   * os arquivos de uma arvore de diretrios, dentro do servidor local.
   *
   * @param path caminho para uma rvore de diretrios.
   *
   * @return mapeamento entre um arquivo e sua data de modificao para todos os
   *         arquivos de uma rvore de diretrios, dentro do servidor local.
   *
   * @throws ServerException em caso de erro.
   */
  public Map<String[], Long> getTimestamps(String[] path)
    throws ServerException {
    return getTimestamps(csfsHost, csfsPort, path);
  }

  /**
   * Implementao recursiva que percorre uma arvore de diretrios. Sempre que
   * um arquivo  localizado, a tabela com a data de modificao  acrescida em
   * um elemento. No caso de diretrios, esse metodo eh chamado recursivamente
   * para cada um dos seus filhos. Ao final da execuo, o mapeamento possui a
   * informao para todos os arquivos.
   *
   * @param info o mapeamento que agrega a informao sobre as datas de
   *        modificao
   * @param root um ponto da rvore, que deve ser percorrido recursivamente
   * @throws ServerException em caso de erro.
   */
  private void visitRemoteFilesGettingTimestamps(Map<String[], Long> info,
    RemoteFile root) throws ServerException {
    try {
      if (!root.isDirectory()) {
        info.put(root.getFullName(), new Long(root.lastModified()));
      }
      else {
        RemoteFile[] children = root.getChildren();
        for (RemoteFile element : children) {
          visitRemoteFilesGettingTimestamps(info, element);
        }
      }
    }
    catch (csfs.remote.ServerException exc) {
      Server.logSevereMessage("Error in CSFS subsystem!", exc);
      throw new ServerException(
        "An error occurred in remote file server. message=" + exc.message, exc);
    }
    catch (Exception exc) {
      throw new ServerException(
        "Exception during visitRemoteFilesGettingTimestamps operation."
          + "exc.getClass().getName()=" + exc.getClass().getName() + " file="
          + FileUtils.joinPath(getName(exc)), exc);
    }
  }

  /**
   * Copia um arquivo da rea de armazenamento principal para uma mquina de
   * execuo.
   *
   * @param host o nome da mquina de execuo
   * @param port porta para acessar a mquina
   * @param sourcePath o caminho absoluto do arquivo na area de armazenamento
   *        (relativo a raiz exportada pelo CSFS).
   * @param targetPath o caminho absoluto do arquivo a ser criado no destino
   *        (relativo a raiz exportada pelo CSFS).
   * @throws ServerException em caso de erro.
   */
  public void copyTo(String host, int port, String[] sourcePath,
    String[] targetPath) throws ServerException {
    checkActive();
    FileServer sgaFileServer = getFileServer(host, port);
    copy(mainFileServer, sourcePath, sgaFileServer, targetPath);
  }

  /**
   * Copia um arquivo de uma mquina de execuo para a rea de armazenamento
   * principal.
   *
   * @param host o nome da mquina de execuo
   * @param port porta para acessar a mquina
   * @param sourcePath o caminho absoluto do arquivo na mquina de execucao
   *        (relativo a raiz exportada pelo CSFS).
   * @param targetPath o caminho absoluto do arquivo a ser criado na rea de
   *        armazenamento principal (relativo a raiz exportada pelo CSFS).
   * @throws ServerException em caso de erro.
   */
  public void copyFrom(String host, int port, String[] sourcePath,
    String[] targetPath) throws ServerException {
    checkActive();
    FileServer sgaFileServer = getFileServer(host, port);
    copy(sgaFileServer, sourcePath, mainFileServer, targetPath);
  }

  /**
   * Realiza a copia de arquivos (Esse mtodo opera recursivamente sobre
   * diretrios).
   *
   * @param sourceFS o servidor (CSFS) de origem.
   * @param sourcePath o caminho absoluto no servidor de origem (relativo a raiz
   *        exportada pelo CSFS).
   * @param targetFS o servidor (CSFS) de destino.
   * @param targetPath o caminho absoluto no servidor de destino (relativo a
   *        raiz exportada pelo CSFS).
   * @throws ServerException em caso de erro.
   */
  private void copy(FileServer sourceFS, String[] sourcePath,
    FileServer targetFS, String[] targetPath) throws ServerException {
    try {
      // a origem deve existir.
      RemoteFile sourceFile = lookup(sourceFS, sourcePath);
      // o destino pode existir previamente (ou nao).
      RemoteFile targetFile =
        lookupOrCreate(targetFS, targetPath, sourceFile.isDirectory());
      if (sourceFile.isDirectory()) {
        copyDirectory(sourceFile, targetFile);
      }
      else {
        copyFile(sourceFile, targetFile);
      }
    }
    catch (csfs.remote.ServerException exc) {
      Server.logSevereMessage("Error in CSFS subsystem!", exc);
      throw new ServerException("Remote File Server Exception. message="
        + exc.message, exc);
    }
    catch (Exception exc) {
      throw new ServerException(
        "Exception during copy operation. exc.getClass().getName()="
          + exc.getClass().getName() + " file="
          + FileUtils.joinPath(getName(exc)) + " source:"
          + FileUtils.joinPath(sourcePath) + " target:"
          + FileUtils.joinPath(targetPath), exc);
    }
  }

  /**
   * Realiza a cpia recursiva de diretrios. Tanto a origem quanto o destino
   * devem apontar explicitamente para um diretrio.
   *
   * Um tratamento peculiar sobre a preexistncia de arquivos e diretorios na
   * rvore remota  realizado. De forma sucinta, arquivos preexistentes no
   * destino sero sobrescritos, desde que a origem tambm seja um arquivo. Caso
   * um diretrio esteja sendo copiado sobre um destino preexistente, caso o
   * destino tambem seja um diretrio, ele receber novos elementos. Caso
   * contrrio, ou seja, quando um arquivo iria sobreescrever um diretrio, uma
   * exceo  lanada.
   *
   * Esses procedimentos foram definidos com base em algumas observaes sobre a
   * coerncia e previsibilidade de alguns algoritmos. Outros tipos de
   * algoritmos talvez possam necessitar de uma outra semntica sobre as
   * operaes de cpia, e esse procedimento podem ser revisto de acordo com o
   * uso do servio.
   *
   * @param sourceFile o diretrio de origem
   * @param targetFile o diretrio de destino
   *
   * @throws NotDirectoryException caso no encontre um diretrio.
   * @throws csfs.remote.ServerException em caso de erro na mquina remota.
   * @throws InvalidStateException caso haja algum problema com um
   *         arquivo/diretrio remoto.
   * @throws InvalidPathException caso o caminho seja invlido conforme
   *         especificao da IDL.
   * @throws FileNotFoundException caso um arquivo/diretrio remoto no seja
   *         encontrado.
   * @throws ServerException em caso de erro na mquina local.
   * @throws FileAlreadyExistsException caso j exista um arquivo no caminho
   *         especificado.
   * @throws NotFileException caso fonte ou destino no sejam arquivos de dados.
   */
  private void copyDirectory(RemoteFile sourceFile, RemoteFile targetFile)
    throws NotDirectoryException, csfs.remote.ServerException,
    InvalidStateException, InvalidPathException, FileNotFoundException,
    ServerException, FileAlreadyExistsException, NotFileException {
    RemoteFile[] children = sourceFile.getChildren();
    for (RemoteFile element : children) {
      if (element.isDirectory()) {
        // caso a origem seja um diretrio, tentaremos criar um diretrio no
        // destino
        RemoteFile targetChildFile;
        try {
          targetChildFile =
            targetFile.createDirectory(new String[] { element.getName() });
        }
        catch (FileAlreadyExistsException e) {
          // Caso um elemento j exista, precisamos ver se  um arquivo, para
          // levantar a exceo.
          targetChildFile =
            targetFile.getChild(new String[] { element.getName() });
          if (!targetChildFile.isDirectory()) {
            throw new ServerException("There is a file in the way: "
              + FileUtils.joinPath(e.name), e);
          }
        }
        // chamamos a funcao recursiva para mais um nvel na rvore
        copyDirectory(element, targetChildFile);
      }
      else {
        // tentamos realizar a copia do arquivos para o diretrio de destino
        copyFile(element, targetFile);
      }
    }
  }

  /**
   * Copia um arquivo. Caso o destino seja um arquivo, ele ser sobrescrito.
   * Caso contrrio, um novo arquivo ser criado no diretrio de destino (com o
   * mesmo nome do arquivo de origem). Caso um elemento j exista, se ele for um
   * arquivo, ser sobrescrito. Se for um diretrio, um conflito entre o nome da
   * origem e o elemento contido no diretrio de destino ser detectado, e uma
   * exceo ser lanada.
   *
   * Para que os biarios possam ser executados no Linux, eles devem possuir a
   * permisso de acesso adequada. Optamos por fazer com que todo arquivo
   * copiado tenha essa permisso, independente do fato do arquivo ser realmente
   * um binrio. Como a versao atual de Java ainda nao implementa essa
   * atribuio, o servidor faz uma chamada externa (via JNI/Runtime.exec())
   * para o "chmod". Caso a plataforma do servidor seja "Win32", definido pela
   * variavel "os.arch" (da classe System), a tentativa de se habilitar a
   * permisso de execuo nao surte nenhum efeito.
   *
   * @param sourceFile o arquivo de origem
   * @param targetFile o diretorio de destino
   *
   * @throws NotDirectoryException caso no encontre um diretrio.
   * @throws csfs.remote.ServerException em caso de erro na mquina remota.
   * @throws InvalidStateException caso haja algum problema com um
   *         arquivo/diretrio remoto.
   * @throws InvalidPathException caso o caminho seja invlido conforme
   *         especificao da IDL.
   * @throws FileNotFoundException caso um arquivo/diretrio remoto no seja
   *         encontrado.
   * @throws ServerException em caso de erro na mquina local.
   * @throws FileAlreadyExistsException caso j exista um arquivo no caminho
   *         especificado.
   * @throws NotFileException caso fonte ou destino no sejam arquivos de dados.
   */
  private void copyFile(RemoteFile sourceFile, RemoteFile targetFile)
    throws csfs.remote.ServerException, InvalidStateException,
    NotDirectoryException, InvalidPathException, FileAlreadyExistsException,
    NotFileException, FileNotFoundException, ServerException {
    RemoteFile targetCopyFile = targetFile;
    if (targetFile.isDirectory()) {
      try {
        targetCopyFile =
          targetFile.createFile(new String[] { sourceFile.getName() });
      }
      catch (FileAlreadyExistsException e) {
        // Caso um elemento jah exista, precisamos ver se eh um diretorio, para
        // levantar a excecao.
        targetCopyFile =
          targetFile.getChild(new String[] { sourceFile.getName() });
        if (targetCopyFile.isDirectory()) {
          throw new ServerException("There is a file in the way: "
            + FileUtils.joinPath(e.name), e);
        }
      }
    }
    sourceFile.copyTo(targetCopyFile, this.transferMethod);
    targetCopyFile.enableExecutionPermission();
  }

  /**
   * Remove uma rvore remota recursivamente no servidor especificado.
   *
   * @param host o servidor que possui a arvore
   * @param port porta para acessar o servidor
   * @param path o caminho absoluto do arquivo (relativo a raiz exportada pelo
   *        CSFS).
   * @throws ServerException em caso de erro.
   */
  public void remove(String host, int port, String[] path)
    throws ServerException {
    checkActive();
    try {
      FileServer targetFS = getFileServer(host, port);
      RemoteFile file = lookup(targetFS, path);
      recursiveRemove(file);
    }
    catch (FileNotFoundException exc) {
      throw new ServerException("A file was not found: "
        + FileUtils.joinPath(exc.name), exc);
    }
  }

  /**
   * Implementao recursiva do mtodo de remoo.
   *
   * @param file a raiz que deve ser removida.
   * @throws ServerException em caso de erro.
   */
  private void recursiveRemove(RemoteFile file) throws ServerException {
    try {
      if (file.isDirectory()) {
        RemoteFile[] children = file.getChildren();
        for (RemoteFile element : children) {
          recursiveRemove(element);
        }
      }
      file.remove();
    }
    catch (csfs.remote.ServerException exc) {
      Server.logSevereMessage("Error in CSFS subsystem!", exc);
      throw new ServerException("Remote File Server Exception. message="
        + exc.message, exc);
    }
    catch (Exception exc) {
      throw new ServerException(
        "Exception during recursiveRemove operation. exc.getClass().getName()="
          + exc.getClass().getName() + " file="
          + FileUtils.joinPath(getName(exc)), exc);
    }
  }

  /**
   * Cria um diretrio em uma mquina.
   *
   * @param host o servidor onde o diretrio deve ser criado
   * @param port porta para acessar o servidor
   * @param dir o caminho absoluto do arquivo (relativo a raiz exportada pelo
   *        CSFS).
   * @throws ServerException em caso de erro.
   */
  public void createDirectory(String host, int port, String[] dir)
    throws ServerException {
    checkActive();
    FileServer targetFS = getFileServer(host, port);
    lookupOrCreate(targetFS, dir, true);
  }

  /**
   * Busca uma referncia para um arquivo remoto.
   *
   * @param fs o servidor de arquivos que mantem o elemento.
   * @param path o caminho absoluto do arquivo (relativo a raiz exportada pelo
   *        CSFS).
   * @return a referencia para o arquivo remoto
   * @throws FileNotFoundException caso no encontre o arquivo.
   * @throws ServerException em caso de erro.
   */
  private RemoteFile lookup(FileServer fs, String[] path)
    throws FileNotFoundException, ServerException {
    if (fs == null) {
      Server.logWarningMessage("Trying to lookup a file in a null file system");
      throw new ServerException(
        "It is not possible to lookup a path in a null file server reference!");
    }
    RemoteFile response;
    try {
      response = fs.getRoot().getChild(path);
    }
    catch (FileNotFoundException exc) {
      throw exc;
    }
    catch (csfs.remote.ServerException exc) {
      Server.logSevereMessage("Error in CSFS subsystem!", exc);
      throw new ServerException("Remote File Server Exception. message="
        + exc.message, exc);
    }
    catch (Exception exc) {
      throw new ServerException(
        "Exception during lookup operation. exc.getClass().getName()="
          + exc.getClass().getName() + " file="
          + FileUtils.joinPath(getName(exc)), exc);
    }
    return response;
  }

  /**
   * Tenta buscar uma referncia para um elemento remoto. Caso no encontre, um
   * novo elemento ser criado no servidor de destino e a referncia ser
   * retornada.
   *
   * @param fs o servidor de destino
   * @param path o caminho absoluto do arquivo (relativo a raiz exportada pelo
   *        CSFS).
   * @param directory indica qual o tipo de elemento deve ser buscado/criado.
   * @return o elemento
   * @throws ServerException em caso de erro.
   */
  private RemoteFile lookupOrCreate(FileServer fs, String[] path,
    boolean directory) throws ServerException {
    RemoteFile response = null;
    try {
      response = lookup(fs, path);
      if (response.isDirectory() != directory) {
        throw new ServerException("type mismatch. path=" + path
          + " response.isDirectory()=" + response.isDirectory() + " expected="
          + directory);
      }
    }
    catch (FileNotFoundException e) {
      try {
        if (directory) {
          response = fs.getRoot().createDirectory(path);
        }
        else {
          response = fs.getRoot().createFile(path);
        }
        // talvez seja possvel um refactoring nesse tratamento de excees.
      }
      catch (csfs.remote.ServerException exc) {
        Server.logSevereMessage("Error in CSFS subsystem!", exc);
        throw new ServerException("Remote File Server Exception. message="
          + exc.message, exc);
      }
      catch (Exception exc) {
        throw new ServerException(
          "Exception during lookupOrCreate operation. exc.getClass().getName()="
            + exc.getClass().getName() + " file="
            + FileUtils.joinPath(getName(exc)), exc);
      }
    }
    catch (csfs.remote.ServerException exc) {
      Server.logSevereMessage("Error in CSFS subsystem!", exc);
      throw new ServerException("Remote File Server Exception. message="
        + exc.message, exc);
    }
    catch (Exception exc) {
      throw new ServerException(
        "Exception during lookupOrCreate operation. exc.getClass().getName()="
          + exc.getClass().getName() + " file="
          + FileUtils.joinPath(getName(exc)), exc);
    }
    return response;
  }

  /**
   * Recupera o nome do arquivo associado a uma exceo (caso disponivel).
   *
   * De acordo com a exceo, a informao sobre o arquivo pode estar disponvel
   * como um membro da classe. O ideal era ter uma superclasse das excees de
   * arquivo, e fazer o catch por ela, em vez de um catch genrico por
   * Exception... Um mecanismo de herana de excees em CORBA no foi
   * localizado, e essa implementao baseada em "reflection" foi usada como
   * alternativa. Na especificacao da IDL, toda exceo de arquivo possui um
   * String[] name indicando o arquivo onde aconteceu o erro.
   *
   * @param exc a exceo.
   * @return o nome.
   */
  private String[] getName(Exception exc) {
    String[] name = null;
    try {
      Field f = exc.getClass().getDeclaredField("name");
      name = (String[]) f.get(exc);
    }
    catch (Exception e) {
      name = new String[] { "unspecified" };
    }
    return name;
  }

}
