package csbase.server.services.opendreamsservice.opendreams.v2_0;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;

import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAPackage.ServantNotActive;
import org.omg.PortableServer.POAPackage.WrongPolicy;

import tecgraf.openbus.DRMAA.v2_0.*;
import tecgraf.openbus.DRMAA.v2_0.UnsupportedOperationException;
import tecgraf.openbus.opendreams.v2_0.FinalizationType;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobInfo;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobTemplate;
import csbase.exception.OperationFailureException;
import csbase.logic.CommandNotification;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.filters.BodyTypeFilter;

/**
 * @author Tecgraf/PUC-Rio
 */
@SuppressWarnings("serial")
public class OpenDreamsJobSession implements JobSessionOperations, Serializable {

	/**
	 * Constante para timeout infinito.
	 */
	public static long TIMEOUT_WAIT_FOREVER = -1;

	/**
	 * Constante para espera sem timeout.
	 */
	public static long TIMEOUT_NO_WAIT = 0;

	/**
	 * Nome do diretrio para persistncia de jobs
	 */
	private static final String JOBS_DIR = "jobs";

	/**
	 * Nome do diretrio para persistncia de job arrays
	 */
	private static final String JOB_ARRAYS_DIR = "job_arrays";

	/**
	 * O identificador da job session.
	 */
	private String sessionName;

	/**
	 * Representa uma instalao especfica de um sistema DRM especfico
	 */
	private String contact;

	/**
	 * O identificador do usurio associado a essa sesso.
	 */
	private String userId;

	/**
	 * Mapa persistente de jobs por jobId.
	 */
	private PersistentMap<String, OpenDreamsJob> jobs;

	/**
	 * Mapa persistente de jobArrays por jobArrayId.
	 */
	private PersistentMap<String, OpenDreamsJobArray> jobArrays;

	/**
	 * Referncia para o objeto atual.
	 */
	private transient JobSession ref;

	/**
	 * O identificador do listener de mensagens de trmino de jobs da sesso.
	 */
	private transient Serializable listenerId;

	/**
	 * Flag que indica se a sesso est aberta.
	 */
	private transient boolean opened = false;

	/**
	 * @param sessionName
	 * @param contact
	 * @param userId
	 */
	public OpenDreamsJobSession(String sessionName, String contact, String userId) {
		this.sessionName = sessionName;
		this.contact = contact;
		this.userId = userId;
		this.jobs = new PersistentMap<String, OpenDreamsJob>(PersistentObject.generatePath(JOBS_DIR, sessionName));
		this.jobArrays = new PersistentMap<String, OpenDreamsJobArray>(
				PersistentObject.generatePath(JOB_ARRAYS_DIR, sessionName));
		this.listenerId = setSessionOnJobTerminateListener();
		this.opened = true;

	}

	/**
	 * Mtodo utilizado para reiniciar servios da sesso, caso seja necessrio.
	 * Chamado em {@link SessionManager #openJobSession(String)}.
	 */
	public void onResume() {
		if (!opened) {
			System.out.println("Session " + sessionName + " being resumed");
			if (listenerId != null) {
				MessageService.getInstance().clearServerMessageListener(listenerId);
			}
			listenerId = setSessionOnJobTerminateListener();
			this.opened = true;
		}
	}

	private void writeObject(ObjectOutputStream out) throws IOException {
		out.defaultWriteObject();
	}

	private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
		in.defaultReadObject();
    this.jobs = new PersistentMap<String, OpenDreamsJob>(PersistentObject.generatePath(JOBS_DIR, sessionName));
    this.jobArrays = new PersistentMap<String, OpenDreamsJobArray>(
      PersistentObject.generatePath(JOB_ARRAYS_DIR, sessionName));
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * This method returns the contact value that was used in the
	 * {@link SessionManager #createJobSession}. This attribute is read-only.
	 */
	@Override
	public String contact() {
		return contact;
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * This method returns the session name, a unique session identifier passed
	 * through {@link SessionManager #createJobSession}. This attribute is
	 * read-only.
	 */
	@Override
	public String sessionName() {
		return sessionName;
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * UNSUPPORTED This method provides the list of valid job category names
	 * which can be used for the {@code jobCategory} attribute in a
	 * {@code JobTemplate} instance
	 */
	@Override
	public String[] jobCategories() {
		return null;
	}

	/**
	 * {@inheritDoc}
	 * 
	 */
	@Override
	public Job[] getJobs(JobInfo filter) {
		OpenDreamsJobInfo filterJobInfo = (OpenDreamsJobInfo) filter;
		ArrayList<Job> filtered = new ArrayList<Job>();
		for (OpenDreamsJob job : jobs.values()) {
			try {

				JobInfo jobInfo = job.getInfo();
				List<Field> fields = new ArrayList<Field>();
				fields = getAllFields(fields, filter.getClass());

				// JobInfo com description igual  "UNSET" retorna todos os jobs
				// da sesso.
				if (match(filter, jobInfo, fields) || filterJobInfo.description.equals("UNSET")) {
					filtered.add(job.corbaObjectReference());
				}
			} catch (OperationFailureException | ServantNotActive | WrongPolicy e) {
				String msg = "Erro ao criar referncias do Job " + job.jobId();
				Server.logSevereMessage(msg, e);
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				String msg = "Erro ao filtrar jobs a partir das informaes " + filter.toString();
				Server.logSevereMessage(msg, e);
				e.printStackTrace();
			}
		}
		return filtered.toArray(new Job[]{});
	}
	private boolean match(JobInfo filter, JobInfo info, List<Field> fields) throws IllegalAccessException {

		for (Field field : fields) {
			if (field.getName().equals("serialVersionUID") || field.getName().equals("_truncatable_ids")
					|| field.getName().equals("allocatedMachines"))
				continue;
			field.setAccessible(true);
			if (field.getType().equals(Integer.TYPE) || field.getType().equals(Integer.class)) {
				Integer integerValue = (Integer) field.get(filter);
				if (!integerValue.equals(SessionManager.UNSET_INTEGER) && !integerValue.equals(0)) {
					if (!integerValue.equals(field.get(info))) {
						return false;
					}
				}
			} else if (field.getType().equals(Long.TYPE) || field.getType().equals(Long.class)) {
				Long longValue = (Long) field.get(filter);
				if (!longValue.equals(SessionManager.UNSET_LONG) && !longValue.equals(0L)) {
					if (!longValue.equals(field.get(info))) {
						return false;
					}
				}
			} else if (field.getType().equals(String.class)) {
				String stringValue = (String) field.get(filter);
				if (!stringValue.equals(SessionManager.UNSET) && !stringValue.isEmpty()) {
					if (!stringValue.equals(field.get(info))) {
						return false;
					}
				}

			} else if (field.getType().equals(String[].class)) {
				String[] stringVectorValue = (String[]) field.get(filter);
				if (stringVectorValue.length > 0) {
					if (!stringVectorValue.equals(field.get(info))) {
						return false;
					}
				}
			} else if (field.getType().equals(String[][].class)) {
				String[][] stringMatrixValue = (String[][]) field.get(filter);
				if (stringMatrixValue.length > 0) {
					if (!stringMatrixValue.equals(field.get(info))) {
						return false;
					}
				}
			} else if (field.getType().equals(JobState.class)) {
				if (!field.get(filter).equals(JobState.UNDETERMINED)) {
					if (!field.get(filter).equals(field.get(info))) {
						return false;
					}
				}

			} else if (field.getType().equals(FinalizationType.class)) {
				if (!field.get(filter).equals(FinalizationType.UNDEFINED)) {
					if (!field.get(filter).equals(field.get(info))) {
						return false;
					}
				}

			} else {
				if (!field.get(filter).equals(field.get(info))) {
					return false;
				}
			}

		}
		return true;
	}

	public static List<Field> getAllFields(List<Field> fields, Class<?> type) {
		fields.addAll(Arrays.asList(type.getDeclaredFields()));

		if (type.getSuperclass() != null) {
			fields = getAllFields(fields, type.getSuperclass());
		}

		return fields;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JobArray getJobArray(String jobArrayId) throws InvalidArgumentException, InternalException {
		try {
			if (jobArrays.containsKey(jobArrayId)) {
				return jobArrays.get(jobArrayId).corbaObjectReference();
			} else {
				throw new InvalidArgumentException("O job array especificado " + jobArrayId + " no foi encontrado.");
			}
		} catch (Throwable e) {
			String msg = "Erro na recuperao do job array " + jobArrayId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		}
	}

	/**
	 * {@inheritDoc}
	 *
	 * @throws DeniedByDrmsException
	 * @throws InternalException
	 * @throws InvalidArgumentException
	 */
	@Override
	public Job runJob(JobTemplate jobTemplate)
			throws DeniedByDrmsException, InvalidArgumentException, InternalException {

		Server.logFineMessage("OpenDreamsJobSession: runJob");

		try {
			userId = OpenBusService.getInstance().initCSBaseAccess();
			SessionManager.checkUser(userId);

			if (jobTemplate instanceof OpenDreamsJobTemplate) {
				OpenDreamsJobTemplate jt = (OpenDreamsJobTemplate) jobTemplate;
				OpenDreamsJob job = new OpenDreamsJob(sessionName, jt);
				job.execute(userId);
				jobs.put(job.jobId(), job);
				return job.corbaObjectReference();
			} else {
				throw new InvalidJobTemplateException("No foi possvel executar o JobTemplate recebido");
			}
		} catch (DeniedByDrmsException e) {
			e.printStackTrace();
			String msg = "Erro de autenticao no CSBASE do usurio " + userId;
			Server.logSevereMessage(msg, e);
			throw e;
		} catch (InvalidJobTemplateException e) {
			String msg = "Falha na submisso do comando pelo usurio " + userId + ": " + e.getMessage();
			Server.logWarningMessage(msg);
			throw new InvalidArgumentException(LogUtils.formatMessage(e, msg));
		} catch (Throwable e) {
			String msg = "Erro na submisso do comando pelo usurio " + userId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} finally {
			OpenBusService.getInstance().finishCSBaseAccess();
		}
	}

	/**
	 * @return a stub object that references the actual server's object
	 * @throws OperationFailureException
	 * @throws ServantNotActive
	 * @throws WrongPolicy
	 */
	public JobSession corbaObjectReference() throws OperationFailureException, ServantNotActive, WrongPolicy {
		POA poa = OpenBusService.getInstance().getRootPOA();
		if (ref == null) {
			JobSessionPOATie tie = new JobSessionPOATie(this, poa);
			org.omg.CORBA.Object obj = poa.servant_to_reference(tie);
			ref = JobSessionHelper.narrow(obj);
		}
		return ref;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JobArray runBulkJobs(JobTemplate jobTemplate, int beginIndex, int endIndex, int step, int maxParallel)
			throws DeniedByDrmsException, InvalidArgumentException, InternalException {

		Server.logFineMessage("OpenDreamsSession: runBulkJobs (" + beginIndex + ", " + endIndex + ", " + step + ")");

		String userId = SessionManager.UNSET;

		try {

			userId = OpenBusService.getInstance().initCSBaseAccess();
			SessionManager.checkUser(userId);
			if (jobTemplate instanceof OpenDreamsJobTemplate) {
				OpenDreamsJobTemplate jt = (OpenDreamsJobTemplate) jobTemplate;
				OpenDreamsJobArray jobArray = new OpenDreamsJobArray(sessionName, userId, jt);
				jobArray.execute(userId, beginIndex, endIndex, step);
				jobArrays.put(jobArray.jobArrayId(), jobArray);
				jobs.putAll(jobArray.jobsMap());
				return jobArray.corbaObjectReference();
			} else {
				throw new InvalidJobTemplateException("No foi possvel executar o JobTemplate recebido");
			}

		} catch (DeniedByDrmsException e) {
			String msg = "Falha na submisso do comando pelo usurio  " + userId + ": " + e.message;
			Server.logWarningMessage(msg);
			throw e;
		} catch (InvalidJobTemplateException e) {
			String msg = "Falha na submisso do comando pelo usurio " + userId + ": " + e.getMessage();
			Server.logWarningMessage(msg);
			throw new InvalidArgumentException(LogUtils.formatMessage(e, msg));
		} catch (InvalidArgumentException e) {
			String msg = "Falha na submisso do comando pelo usurio " + userId + ": " + e.message;
			Server.logWarningMessage(msg);
			throw e;
		} catch (Throwable e) {
			String msg = "Erro na submisso do comando pelo usurio " + userId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} finally {
			OpenBusService.getInstance().finishCSBaseAccess();
		}
	}

	/**
	 * The waitAnyStarted method blocks until any of the jobs referenced in the
	 * jobs parameter entered one of the "Started" states.
	 */
	@Override
	public Job waitAnyStarted(Job[] jobs, long timeout) throws UnsupportedOperationException {
		throw new UnsupportedOperationException("Operao no suportada.");
	}

	/**
	 * The waitAnyTerminated method blocks until any of the jobs referenced in
	 * the jobs parameter entered one of the "Terminated" states.
	 * <p/>
	 * The timeout argument specifies the desired waiting time for the state
	 * change. The constant value INFINITE_TIME MUST be supported to get an
	 * indefinite waiting time. The constant value ZERO_TIME MUST be supported
	 * to express that the method call SHALL return immediately. A number of
	 * seconds can be specified to indicate the maximum waiting time . If the
	 * method call returns because of timeout, an TimeoutException SHALL be
	 * raised. An application waiting for some condition to happen in all jobs
	 * of a set is expected to perform looped calls of these waiting functions.
	 *
	 * @throws InternalException
	 */

	@Override
	public Job waitAnyTerminated(Job[] jobs, long timeout)
			throws TimeoutException, DeniedByDrmsException, InternalException {

		Server.logFineMessage("OpenDreamsJobSession: waitAnyTerminated");

		try {
			userId = OpenBusService.getInstance().initCSBaseAccess();
			SessionManager.checkUser(userId);

			final SynchronousQueue<OpenDreamsJob> anyTerminatedJob = new SynchronousQueue<OpenDreamsJob>();
			Serializable listenerId = setOnJobTerminateListener(jobs, anyTerminatedJob);
			OpenDreamsJob terminated = anyTerminatedJob.poll(timeout, TimeUnit.SECONDS);
			MessageService.getInstance().clearServerMessageListener(listenerId);
			if (terminated != null) {
				return terminated.corbaObjectReference();
			} else {
				throw new TimeoutException("Nenhum job terminou dentro do timeout especificado.");
			}
		} catch (DeniedByDrmsException e) {
			e.printStackTrace();
			String msg = "Erro de autenticao no CSBASE do usurio " + userId;
			Server.logSevereMessage(msg, e);
			throw e;
		} catch (TimeoutException e) {
			e.printStackTrace();
			String msg = "Timeout excedido na espera por finalizao de jobs " + userId;
			Server.logSevereMessage(msg, e);
			throw e;
		} catch (InterruptedException e) {
			e.printStackTrace();
			String msg = "Erro na espera por finalizao de jobs " + userId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} catch (Throwable e) {
			String msg = "Erro na espera por finalizao de jobs " + userId;
			Server.logSevereMessage(msg, e);
			throw new InternalException(LogUtils.formatMessage(e, msg));
		} finally {
			OpenBusService.getInstance().finishCSBaseAccess();
		}
	}

	/**
	 * @param jobsToListen
	 *            Jobs para serem monitorados
	 * @param anyTerminatedJob
	 *            SynchronousQueue para receber o primeiro job terminado.
	 * @return o identificador do listener criado.
	 */
	private Serializable setOnJobTerminateListener(Job[] jobsToListen,
			final SynchronousQueue<OpenDreamsJob> anyTerminatedJob) {

		final ArrayList<String> jobIdsToListen = new ArrayList<String>();
		for (Job job : jobsToListen) {
			jobIdsToListen.add(job.jobId());
		}
		return MessageService.getInstance().setServerMessageListener(new IMessageListener() {
			@Override
			public void onMessagesReceived(Message... messages) throws Exception {
				for (Message message : messages) {
					CommandNotification notification = (CommandNotification) message.getBody();
					String cmdId = (String) notification.getCommandId();

					if (jobIdsToListen.contains(cmdId)) {
						boolean handled = anyTerminatedJob.offer(jobs.get(cmdId));
					}
				}
			}
		}, new BodyTypeFilter(CommandNotification.class));
	}

	/**
	 * Registra um listener de mensagens.
	 *
	 * @return o identificador do listener criado
	 */

	/*
	*
	  *
	 */
	private Serializable setSessionOnJobTerminateListener() {
		return MessageService.getInstance().setServerMessageListener(new IMessageListener() {
			@Override
			public void onMessagesReceived(Message... messages) throws Exception {
				for (Message message : messages) {
					CommandNotification notification = (CommandNotification) message.getBody();
					String cmdId = (String) notification.getCommandId();
					System.out.println(sessionName + " onSessionJobTerminateListener: onMessageReceived " + cmdId);

					try {
						Service.setUserId(userId);
						if (jobs.containsKey(cmdId)) {
							System.out.println(
									"Finishing job " + cmdId + " of user " + userId + " of session " + sessionName);
							jobs.get(cmdId).onFinish(userId, notification);
						} else {
							System.out.println(
									"Job " + cmdId + " of user " + userId + " don't belong to session " + sessionName);

						}
					} catch (Exception e) {
						Server.logSevereMessage("Erro ao finalizar o comando " + cmdId, e);
					} finally {
						Service.setUserId(null);
					}
				}
			}
		}, new BodyTypeFilter(CommandNotification.class));
	}

}
