/*
 * $Id: Server.java 176427 2016-10-04 14:35:28Z fpina $
 */
package csbase.server;

import java.io.File;
import java.io.InputStream;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.SortedSet;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.exception.InitFailureException;
import csbase.logic.User;
import csbase.logic.diagnosticservice.DeploymentInfo;
import csbase.logic.diagnosticservice.PropertyInfo;
import csbase.remote.ServerEntryPoint;
import csbase.server.keystore.CSKeyStore;
import csbase.server.services.messageservice.MessageStoreDAO;
import csbase.util.messages.MessageBroker;
import csbase.util.messages.dao.IMessageStoreDAO;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.FormatUtils;

/**
 * Classe que representa um servidor genrico.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class Server {

  /**
   * Propriedade que define se o servidor pode ser associado a localhost
   * (127.0.*). Deve ser definida como <code>false</code> no ambiente de
   * produo.
   */
  private static final String PROP_LOCALHOST_ALLOWED = "localhostAllowed";

  /**
   * Propriedade que define o locale default do servidor.
   */
  private static final String PROP_SERVER_DEFAULT_LOCALE = "DefaultLocale";

  /**
   * Propriedade que define a propriedade locale default do cliente.
   */
  private static final String PROP_LOCALE = "locale";

  /**
   * Propriedade que define o nome do servidor.
   */
  private static final String PROP_HOST_NAME = "name";

  /**
   * Propriedade que define o endereo IP do servidor.
   */
  private static final String PROP_HOST_ADDR = "hostAddr";

  /**
   * Propriedade que define o endereo IP do servidor.
   */
  private static final String PROP_DEFAULT_CHARSET = "defaultCharset";

  /**
   * Chave usada para criptografar os arquivos de permisses.
   */
  private static final String ADMIN_PASSWORD = "aquaman";

  /**
   * Propriedade que define a verso mnima da JVM suportada pelo servidor.
   */
  private static final String PROP_MIN_SUPPORTED_JVM_VERSION =
    "minimumJVMVersion";

  /**
   * Constante que define o nome padro do arquivo de idiomas.
   */
  private final static String DEFAULT_LAGUANGE_FILE_NAME = "idiom";

  /**
   * Propriedade que define arquivos de idiomas adicionais
   */
  private final static String ADDITIONAL_LANGUAGE_FILE_PROPERTY =
    "additional.language.file";

  /**
   * Mensagem para indicar o locale usado ao iniciar o servidor
   */
  private static final String STARTING_LOCALE = "locale.starting.message";

  /**
   * Instncia do servidor
   */
  private static Server instance = null;

  /**
   * Interface remota do servidor.
   */
  protected Registry registry = null;

  /**
   * Caminho para o arquivo do servidor para sobrescrever propriedades dos
   * servios
   */
  private String serverPropertiesFilePath;

  /**
   * Propriedades definidas pela linha de comando
   */
  private final Properties commandLineProperties;

  /**
   * Propriedades do servidor, carregadas do arquivo Server.properties.
   */
  private final ServerSideProperties serverProperties;

  /**
   * Propriedades do override do servidor (vinda da command-line).
   */
  private final ServerSideProperties systemProperties;

  /**
   * Nome do servidor. Por exemplo: InfoGrid, WebSintesi, MARLIM.
   */
  final private String systemName;

  /**
   * Verso mnima da JVM suportada pelo servidor. Por exemplo: 1.6.0_07.
   */
  final private String minJVMVersion;

  /**
   * A senha da chave privada do servidor.
   */
  private final String privateKeyPassword;

  /**
   * Porta na qual o registry RMI aceita requisies.
   */
  private final int registryPort;

  /**
   * Porta na qual os objetos RMI sero exportados.
   */
  private final int rmiExportPort;

  /**
   * Configura o endereo utilizado para exportar objetos RMI do servidor. *
   */
  private final String serverHostName;

  /**
   * Representa o ponto de entrada do servidor.
   */
  private ServerEntryPoint entryPoint;

  /**
   * Locale default do servidor
   */
  private final Locale defaultLocale;

  /**
   * Bundles de internacionalizao (indexados por objetos locale)
   */
  private final Map<Locale, HierachicalResourceBundle> bundles =
    new Hashtable<Locale, HierachicalResourceBundle>();

  /**
   * Arquivo de sistema (configurao override) do servidor; que pode ser
   * <code>null</code> no caso de serem usadas as configuraes default do
   * sistema.
   */
  private String systemFileName = null;

  /**
   * Endereo IP do servidor, definido pela propriedade {@value #PROP_HOST_ADDR}
   * .
   */
  private String serverHostAddr = null;

  /**
   * Charset default do servidor, definido pela propriedade
   * {@value #PROP_DEFAULT_CHARSET}
   */
  final private Charset systemDefaultCharset;

  /**
   * Charset da mquina aonde roda o servidor.
   */
  final private Charset serverHostCharset;

  /**
   * Timestamp em ms referente  hora de inicializao do servidor.
   * 
   * @see System#currentTimeMillis()
   */
  private long startupTime;

  /**
   * O <i>boker</i> de mensagens.
   */
  private MessageBroker messageBroker;

  /**
   * Mtodo para checagem e criao de diretrios (se necessrio).
   * 
   * @param dirPath o caminho relativo para verificao.
   * 
   * @throws ServerException 1) se o caminho existir e no representar um
   *         diretrio; ou 2) se o caminho no existir e no for possvel criar
   *         um novo diretrio no mesmo.
   */
  public static void checkDirectory(final String dirPath)
    throws ServerException {
    if (dirPath == null) {
      throw new IllegalArgumentException("dirPath == null");
    }
    final File directory = new File(dirPath);
    if (!directory.exists()) {
      if (!directory.mkdirs()) {
        final String err = "Erro ao criar diretrio: " + dirPath + ".";
        throw new ServerException(err);
      }
    }
    else {
      if (!directory.isDirectory()) {
        final String err = "Diretrio invlido: [" + dirPath + "].";
        throw new ServerException(err);
      }
    }
  }

  /**
   * Verifica se comunicao com o registry.
   * 
   * @return true em caso haja comunicao, false caso contrrio.
   */
  private boolean checkRegistry() {
    try {
      registry.list();
      return true;
    }
    catch (final Throwable t) {
    }
    return false;
  }

  /**
   * Cria o repositrio de chaves do servidor.
   */
  private void createKeyStore() {
    /* Propriedade do caminho do repositrio de chaves do servidor */
    final String KS_PATH_PROPERTY = "keyStorePath";

    /* Propriedade da senha de chave do servidor */
    final String KS_PASSWORD_PROPERTY = "keyStorePassword";

    final boolean nullKeyStorePath = isPropertyNull(KS_PATH_PROPERTY);
    if (nullKeyStorePath) {
      final String fmt = "No foi encontrada a propriedade [%s].";
      final String message = String.format(fmt, KS_PATH_PROPERTY);
      Server.logInfoMessage(message);
    }
    else {
      final String keyStorePath = getStringProperty(KS_PATH_PROPERTY);
      final boolean nullKeyStorePassword = isPropertyNull(KS_PASSWORD_PROPERTY);
      if (nullKeyStorePassword) {
        final String fmt = "No foi encontrada a propriedade [%s].";
        final String message = String.format(fmt, KS_PASSWORD_PROPERTY);
        Server.logWarningMessage(message);
        CSKeyStore.createInstance(keyStorePath);
      }
      else {
        final String keyStorePassword = getStringProperty(KS_PASSWORD_PROPERTY);
        CSKeyStore.createInstance(keyStorePath, keyStorePassword);
      }
    }
  }

  /**
   * Cria a porta de entrada para acesso as funcionalidade do servidor, o
   * <code>ServerEntryPointImpl</code>.
   * 
   * @return o <code>ServerEntryPointImpl</code> que  a porta de entrada para
   *         acesso as funcionalidade do servido
   */
  protected ServerEntryPoint createServerEntryPoint() {
    return new ServerEntryPointImpl();
  }

  /**
   * Mtodo responsvel para criao dos servios. A partir deste mtodo, deve
   * ser chamado o mtodo <code>createService</code> de cada servio.
   * 
   * @throws ServerException caso ocorra falha na criao do servio.
   */
  public abstract void createServices() throws ServerException;

  /**
   * Retorna a senha do administrador.
   * 
   * @return string a senha.
   */
  public String getAdminPassword() {
    return ADMIN_PASSWORD;
  }

  /**
   * Obtm o nome do servidor central.
   * 
   * @return o nome do servidor central
   */
  public String getCentralServerName() {
    final String propertyName = "centralServer";
    final boolean isNull = isPropertyNull(propertyName);
    if (isNull) {
      return null;
    }
    final String centralServerName = getStringProperty(propertyName);
    return centralServerName;
  }

  /**
   * Retorna o {@code Locale} default configurado na instalao do servidor
   * CSBase.
   * 
   * @return O {@code Locale} ou {@code null} caso no haja um configurado.
   */
  public Locale getDefaultLocale() {
    return defaultLocale;
  }

  /**
   * Obtm o ponto de entrada do servidor.
   * 
   * @return o ponto de entrada.
   */
  public final ServerEntryPoint getEntryPoint() {
    return this.entryPoint;
  }

  /**
   * Retorna a mensagem de "boas-vindas" do sistema (exibida quando a
   * inicializao foi bem-sucedida). Pode ser redefinido por subclasses que
   * precisem ou queiram exibir outras informaes.
   * 
   * @return mensagem de boas-vindas do sistema
   */
  protected String getGreetingMessage() {
    return "Servidor do " + getSystemName() + " iniciado em: " + FormatUtils
      .format(getStartupTime());
  }

  /**
   * Retorna o nome do servidor, definido pela propriedade
   * {@value #PROP_HOST_NAME}.
   * 
   * @return nome do servidor
   * 
   * @see #getHostAddr()
   */
  public String getHostName() {
    return serverHostName;
  }

  /**
   * Retorna o endereo IP do servidor, definido pela propriedade
   * {@value #PROP_HOST_ADDR}.
   * 
   * @return endereo IP do servidor
   * 
   * @see #getHostName()
   */
  public String getHostAddr() {
    return serverHostAddr;
  }

  /**
   * Retorna o charset default do sistema conforme definido pela propriedade
   * indicadas por: {@link #PROP_DEFAULT_CHARSET}.
   * 
   * @return o charset
   */
  final public Charset getSystemDefaultCharset() {
    return systemDefaultCharset;
  }

  /**
   * Retorna o nome charset default do sistema: {@link #systemDefaultCharset}.
   * 
   * @return o nome
   */
  final public String getSystemDefaultCharsetName() {
    final Charset ch = getSystemDefaultCharset();
    if (ch == null) {
      return null;
    }
    return ch.name();
  }

  /**
   * Retorna o charset default da mquina do servidor:
   * 
   * @return o charset
   */
  final public Charset getServerHostCharset() {
    return serverHostCharset;
  }

  /**
   * Retorna o nome charset da mquina do servidor: {@link #serverHostCharset}.
   * 
   * @return o nome
   */
  final public String getServerHostCharsetName() {
    if (serverHostCharset == null) {
      return null;
    }
    return serverHostCharset.name();
  }

  /**
   * Retorna a porta na qual o registry RMI devem ser encontrado.
   * 
   * @return porta na qual executa o registry RMI.
   */
  public int getRegistryPort() {
    return this.registryPort;
  }

  /**
   * Retorna a porta na qual os objetos RMI devem ser exportados.
   * 
   * @return porta na qual os objetos RMI devem ser exportados.
   */
  public int getRMIExportPort() {
    return this.rmiExportPort;
  }

  /**
   * Obtm a senha da chave privada do servidor. S existir quando o servidor
   * for servidor local.
   * 
   * @return A senha da chave privada ou null, caso o servidor no seja servidor
   *         local ou se a senha no foi informada.
   */
  public String getPrivateKeyPassword() {
    return this.privateKeyPassword;
  }

  /**
   * Retorna o nome verdadeiro da chave com prefixo.
   * 
   * @param key a chave
   * @return a chave com prefixo (se houver)
   */
  final private String getRealPropertyKey(final String key) {
    final String realKey = "Server." + key;
    return realKey;
  }

  /**
   * Indica se o servidor faz override da propriedade de um servio
   * 
   * @param prefixedKey a chave (j prefixada pelo servio)
   * @return indicativo.
   */
  final protected boolean overridesServiceProperty(String prefixedKey) {
    final boolean inServer = serverProperties.hasProperty(prefixedKey);
    final boolean inSystem;
    if (systemProperties != null) {
      inSystem = systemProperties.hasProperty(prefixedKey);
    }
    else {
      inSystem = false;
    }
    return inServer || inSystem;
  }

  /**
   * Faz checagem de propriedades corretamente carregadas.
   */
  private void checkPropertiesLoaded() {
    if (serverProperties == null) {
      final String msg = "Falha interna de implementao do servidor!\n\n"
        + "Propriedades do servidor ainda no devidamente carregadas";
      throw new IllegalStateException(msg);
    }
  }

  /**
   * Recupera uma propriedade da aplicao (string)
   * 
   * @param key a identificao da propriedade
   * @return o valor da propriedade.
   * @throws IllegalStateException se a propriedade no estiver setada.
   * @see ServerSideProperties#getStringProperty(String)
   */
  final protected String getStringProperty(final String key) {
    return _propertyString(key, true);
  }

  /**
   * Recupera uma propriedade da aplicao (string)
   * 
   * @param key a identificao da propriedade
   * @return o valor da propriedade.
   * @throws IllegalStateException se a propriedade no estiver setada.
   * @see ServerSideProperties#getStringProperty(String)
   */
  final String getStringServiceProperty(final String key) {
    return _propertyString(key, false);
  }

  /**
   * Recupera uma propriedade da aplicao (double)
   * 
   * @param key a chave
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  protected final double getDoubleProperty(final String key) {
    return _propertyDouble(key, true);
  }

  /**
   * Recupera uma propriedade da aplicao (double)
   * 
   * @param key a chave
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final double getDoubleServiceProperty(final String key) {
    return _propertyDouble(key, false);
  }

  /**
   * Recupera uma propriedade da aplicao (double)
   * 
   * @param key a chave
   * @param prefix indica que o tag da propriedade deve concatenar
   *        <code>"Server."</code>.
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final private double _propertyDouble(final String key, final boolean prefix) {
    checkPropertiesLoaded();
    final String realKey = prefix ? getRealPropertyKey(key) : key;
    if (systemProperties != null && systemProperties.hasProperty(realKey)) {
      return systemProperties.getDoubleProperty(realKey);
    }
    return serverProperties.getDoubleProperty(realKey);
  }

  /**
   * Recupera uma propriedade da aplicao (string)
   * 
   * @param key a chave
   * @param prefix indica que o tag da propriedade deve concatenar
   *        <code>"Server."</code>.
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final private String _propertyString(final String key, final boolean prefix) {
    checkPropertiesLoaded();
    final String realKey = prefix ? getRealPropertyKey(key) : key;
    if (systemProperties != null && systemProperties.hasProperty(realKey)) {
      return systemProperties.getStringProperty(realKey);
    }
    return serverProperties.getStringProperty(realKey);
  }

  /**
   * Recupera uma propriedade da aplicao (string)
   * 
   * @param key a chave
   * @param prefix indica que o tag da propriedade deve concatenar
   *        <code>"Server."</code>.
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final private List<String> _propertyStringList(final String key,
    final boolean prefix) {
    checkPropertiesLoaded();
    final String realKey = prefix ? getRealPropertyKey(key) : key;
    if (systemProperties != null && systemProperties.hasProperty(realKey.concat(
      ".1"))) {
      return systemProperties.getStringListProperty(realKey);
    }
    return serverProperties.getStringListProperty(realKey);
  }

  /**
   * Recupera uma propriedade da aplicao (int)
   * 
   * @param key a chave
   * @param prefix indica que o tag da propriedade deve concatenar
   *        <code>"Server."</code>.
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final private int _propertyInt(final String key, final boolean prefix) {
    checkPropertiesLoaded();
    final String realKey = prefix ? getRealPropertyKey(key) : key;
    if (systemProperties != null && systemProperties.hasProperty(realKey)) {
      return systemProperties.getIntProperty(realKey);
    }
    return serverProperties.getIntProperty(realKey);
  }

  /**
   * Recupera uma propriedade da aplicao (long)
   * 
   * @param key a chave
   * @param prefix indica que o tag da propriedade deve concatenar
   *        <code>"Server."</code>.
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final private long _propertyLong(final String key, final boolean prefix) {
    checkPropertiesLoaded();
    final String realKey = prefix ? getRealPropertyKey(key) : key;
    if (systemProperties != null && systemProperties.hasProperty(realKey)) {
      return systemProperties.getLongProperty(realKey);
    }
    return serverProperties.getLongProperty(realKey);
  }

  /**
   * Recupera uma propriedade da aplicao (int)
   * 
   * @param key a chave
   * @param prefix indica que o tag da propriedade deve concatenar
   *        <code>"Server."</code>.
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final private boolean _propertyBool(final String key, final boolean prefix) {
    checkPropertiesLoaded();
    final String realKey = prefix ? getRealPropertyKey(key) : key;
    if (systemProperties != null && systemProperties.hasProperty(realKey)) {
      return systemProperties.getBooleanProperty(realKey);
    }
    return serverProperties.getBooleanProperty(realKey);
  }

  /**
   * Recupera uma propriedade da aplicao (int)
   * 
   * @param key a chave
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final public int getIntProperty(final String key) {
    return _propertyInt(key, true);
  }

  /**
   * Recupera uma propriedade da aplicao (int)
   * 
   * @param key a chave
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final int getIntServiceProperty(final String key) {
    return _propertyInt(key, false);
  }

  /**
   * Recupera uma propriedade da aplicao (long)
   * 
   * @param key a chave
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final public long getLongProperty(final String key) {
    return _propertyLong(key, true);
  }

  /**
   * Recupera uma propriedade da aplicao (long)
   * 
   * @param key a chave
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final long getLongServiceProperty(final String key) {
    return _propertyLong(key, false);
  }

  /**
   * Recupera uma propriedade da aplicao (boolean)
   * 
   * @param key a chave
   * 
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final protected boolean getBooleanProperty(final String key) {
    return _propertyBool(key, true);
  }

  /**
   * Recupera uma propriedade da aplicao (int)
   * 
   * @param key a chave
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final boolean getBooleanServiceProperty(final String key) {
    return _propertyBool(key, false);
  }

  /**
   * Recupera uma propriedade da aplicao (lista)
   * 
   * @param key a chave (prefixo)
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final protected List<String> getStringListProperty(final String key) {
    return _propertyStringList(key, true);
  }

  /**
   * Recupera uma propriedade da aplicao (lista)
   * 
   * @param key a chave (prefixo)
   * @return o valor
   * @throws IllegalStateException se a propriedade no estiver setada.
   */
  final List<String> getStringListServiceProperty(final String key) {
    return _propertyStringList(key, false);
  }

  /**
   * Verifica a nulidade de uma propriedade
   * 
   * @param key a identificao da propriedade
   * 
   * @return indicativo
   */
  final protected boolean isPropertyNull(final String key) {
    checkPropertiesLoaded();
    final String realKey = getRealPropertyKey(key);

    // Verificao se a propriedade foi redefinida pelo arquivo system.
    if (systemProperties != null && systemProperties.hasProperty(realKey)) {
      // Se no existir no arquivo sistema, ser levantada exceo!
      return systemProperties.isPropertyNull(realKey);
    }

    // Se no existir no arquivo servidor, ser levantada exceo!
    return serverProperties.isPropertyNull(realKey);
  }

  /**
   * Verifica se h um RMI registry na porta indicada. Caso no haja, tenta
   * rodar um. Em qualquer caso, guarda no campo privado registry a referncia
   * para o registry. Caso uma referncia no possa ser obtida, retorna falso.
   * 
   * @return true em caso de sucesso, false em caso de falha.
   */
  private boolean getRegistry() {
    try {
      registry = LocateRegistry.getRegistry(registryPort);
    }
    catch (final Exception e) {
    }
    if (!checkRegistry()) {
      try {
        registry = LocateRegistry.createRegistry(registryPort);
      }
      catch (final Exception e) {
      }
    }
    return checkRegistry();
  }

  /**
   * Nome do sistema. Por exemplo: InfoGrid, WebSintesi, MARLIM.
   * 
   * @return O nome do sistema
   */
  public String getSystemName() {
    return this.systemName;
  }

  /**
   * Obtm a verso do servidor.
   *
   * @return a verso.
   */
  public abstract String getVersion();

  /**
   * Pega a URL para execuo do cliente applet do sistema.
   * 
   * @return A URL.
   */
  public String getSystemURL() {
    final String propertyName = "systemURL";
    final boolean isNull = isPropertyNull(propertyName);
    if (isNull) {
      return null;
    }
    final String systemURL = getStringProperty(propertyName);
    return systemURL;
  }

  /**
   * Mtodo para inicializao do sistema de log do servidor.
   * 
   * @throws ServerException em caso de falha.
   */
  private void initLog() throws ServerException {
    System.out.println("Inicializando sistema de logs...");
    // Classe do log manager
    final String logManagerClassName = ServiceLogManager.class.getName();
    System.out.println(" - LogManager: " + logManagerClassName);
    final String loggingMgrKey = "java.util.logging.manager";
    System.setProperty(loggingMgrKey, logManagerClassName);
    // Ajustando path para "logging config file"
    final String logCnfKey = "logging.config.file";
    if (isPropertyNull(logCnfKey)) {
      String err = "Propriedade " + logCnfKey + " est nula no servidor!";
      throw new ServerException(err);
    }
    final String logCnfFile = getStringProperty(logCnfKey);
    System.setProperty("java.util.logging.config.file", logCnfFile);
    System.out.println(" - Usando " + logCnfKey + ": " + logCnfFile);

    try {
      java.util.logging.LogManager.getLogManager().reset();
      java.util.logging.LogManager.getLogManager().readConfiguration();
    }
    catch (Exception e1) {
      throw new ServerException(
        "No foi possvel carregar o arquivo de configurao de logs definido em "
          + getRealPropertyKey(logCnfKey), e1);
    }
    createFileHandlerLoggingDirectory();
  }

  /**
   * CSBASE-3135
   * 
   * O FileHandler no  carregado se o diretrio definido em
   * java.util.logging.FileHandler.pattern no existir. O erro somente aparece
   * na primeira vez que algum logger  usado. Nenhuma exceo  lanada e uma
   * mensagem de erro  enviada para stderr. Para contornar este problema,
   * estamos criando o diretrio a partir do valor definido na propriedade.
   * 
   * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6244047
   * 
   * @throws ServerException (checkDirectory)
   */
  private void createFileHandlerLoggingDirectory() throws ServerException {
    String filePattern = java.util.logging.LogManager.getLogManager()
      .getProperty("java.util.logging.FileHandler.pattern");
    if (filePattern == null) {
      return;
    }
    File fileParent = new File(filePattern).getParentFile();
    if (fileParent == null) {
      return;
    }

    // Cria o diretrio
    System.out.println(
      " - Criando diretrio definido em java.util.logging.FileHandler.pattern: "
        + filePattern);
    checkDirectory(generateDirectoryFileName(fileParent.getAbsolutePath()));
  }

  /**
   * Baseado em java.util.logging.FileHandler.generate. Porm, apenas gera o
   * nome do diretrio aplicando o pattern fornecido.
   * 
   * @param pattern a identificao da propriedade
   * 
   *        A pattern consists of a string that includes the following special
   *        components that will be replaced at runtime:
   *        <ul>
   *        <li>"/" the local pathname separator
   *        <li>"%t" the system temporary directory
   *        <li>"%h" the value of the "user.home" system property
   *        <li>"%%" translates to a single percent sign "%"
   *        </ul>
   * 
   *        No vamos dar suporte em nome de diretrio a:
   * 
   *        <ul>
   *        <li>"%g" the generation number to distinguish rotated logs
   *        <li>"%u" a unique number to resolve conflicts
   *        </ul>
   * 
   * @return indicativo
   */
  public static String generateDirectoryFileName(String pattern) {
    String file = null;
    String word = "";
    int pattIndex = 0;
    while (pattIndex < pattern.length()) {
      char firstChar = pattern.charAt(pattIndex);
      pattIndex++;
      switch (firstChar) {
        case '/':
          if (file == null) {
            file = word;
          }
          else {
            file = file + File.separator + word;
          }
          word = "";
          continue;
        case '%':
          char secondChar = 0;
          if (pattIndex < pattern.length()) {
            secondChar = Character.toLowerCase(pattern.charAt(pattIndex));
          }
          switch (secondChar) {
            case 't':
              String tmpDir = System.getProperty("java.io.tmpdir");
              if (tmpDir == null) {
                tmpDir = System.getProperty("user.home");
              }
              file = tmpDir;
              pattIndex++;
              word = "";
              continue;
            case 'h':
              file = System.getProperty("user.home");
              pattIndex++;
              word = "";
              continue;
            case '%':
              word = word + "%";
              pattIndex++;
              continue;
          }
      }
      word = word + firstChar;
    }
    if (word.length() > 0) {
      if (file == null) {
        file = word;
      }
      else {
        file = file + File.separator + word;
      }
    }
    return file;
  }

  /**
   * Verifica se o servidor  servidor central.
   * 
   * @return true, caso seja um servidor central, ou false, caso contrrio.
   */
  public boolean isCentralServer() {
    final String centralServerName = this.getCentralServerName();
    return (centralServerName == null);
  }

  /**
   * Carrega o bundle default do servidor.
   * 
   * @return o locale default.
   */
  private Locale loadAndSetDefaultLocale() {
    final String propertyName = PROP_SERVER_DEFAULT_LOCALE;
    final boolean isNull = isPropertyNull(propertyName);
    final Locale locale;
    if (!isNull) {
      final String localeStr = getStringProperty(propertyName);
      locale = FormatUtils.parseLocale(localeStr);
    }
    else {
      locale = Locale.getDefault();
    }
    Server.logInfoMessage(getString("Server.defaultLocaleMessage", locale));
    return locale;
  }

  /**
   * Carregamento de bundles de internacionalizao do servidor. Este mtodo
   * carrega o arquivo padro do servidor e todos os arquivos adicionais
   * definidos pelas propriedades:
   * 
   * 1. {@link Server#DEFAULT_LAGUANGE_FILE_NAME}<br/>
   * 2. {@link Server#ADDITIONAL_LANGUAGE_FILE_PROPERTY}<br/>
   * 
   * @param locale locale da traduo.
   * @return true se algum arquivo foi carregado, false caso contrrio.
   */
  private boolean loadLanguageBundle(Locale locale) {
    List<String> languageFiles = new ArrayList<String>();
    languageFiles.add(DEFAULT_LAGUANGE_FILE_NAME);
    languageFiles.addAll(getStringListProperty(
      ADDITIONAL_LANGUAGE_FILE_PROPERTY));

    boolean hasBundlesForServer = false;
    HierachicalResourceBundle parent = null;

    for (String filePath : languageFiles) {
      String fileName = String.format("%s_%s.properties", filePath, locale);
      Class<?> thisClass = getClass();
      /*
       * Carrega todos os resources bundle do servidor. Retorna false se nenhum
       * bundles for carregado.
       */
      while (thisClass != null) {
        try (InputStream in = thisClass.getResourceAsStream(fileName)) {
          thisClass = thisClass.getSuperclass();
          if (in == null) {
            continue;
          }
          hasBundlesForServer = true;
          HierachicalResourceBundle bundle = new HierachicalResourceBundle(in);
          bundles.put(locale, bundle);

          String logFormat = "Arquivo de bundle %s carregado.";
          Server.logInfoMessage(String.format(logFormat, fileName));

          if (parent != null) {
            bundle.setParent(parent);
          }
          parent = bundle;
        }
        catch (Exception e) {
          String logFormat = "Falha na leitura de: %s --> %s";
          String msg = e.getMessage();
          Server.logSevereMessage(String.format(logFormat, filePath, msg));

          logFormat = "Falha no bundle de: %s";
          Server.logSevereMessage(String.format(logFormat, locale));
          return hasBundlesForServer;
        }
      }
      ;
    }
    return hasBundlesForServer;
  }

  /**
   * Carrega o arquivo com as propriedades do sistema.
   * 
   * @return as propriedades do servidor.
   * 
   * @throws ServerException se no existir diretrio de propriedades.
   */
  private ServerSideProperties loadAndSetServerProperties()
    throws ServerException {
    final String propDirPath = getPropertiesRootDirectoryName();
    checkDirectory(propDirPath);

    final String fileName = "Server.properties";
    final String filePath = propDirPath + File.separator + fileName;
    System.out.println("Carregando propriedades (servidor) de: " + filePath);
    final ServerSideProperties props = new ServerSideProperties(filePath);
    props.load();
    System.out.println(" - Carga das propriedades terminada");
    serverPropertiesFilePath = filePath;
    return props;
  }

  /**
   * Carrega o arquivo com as propriedades do sistema.
   * 
   * @param overprops as propriedades que sofreram "override" no servidor.
   * @return as propriedades do servidor.
   * 
   * @throws ServerException se no existir diretrio de propriedades.
   */
  private ServerSideProperties loadAndSetSystemProperties(Properties overprops)
    throws ServerException {

    if ((systemFileName == null) && overprops.isEmpty()) {
      System.out.println("Propriedades (sistema) no definidas! "
        + "Usando configurao default...");
      return null;
    }
    final ServerSideProperties props;
    if (systemFileName == null) {
      props = new ServerSideProperties();
    }
    else {
      props = new ServerSideProperties(systemFileName);
      System.out.println("Carregando propriedades (sistema) de: "
        + systemFileName);
      props.load();
      System.out.println(" - Carga das propriedades (sistema) terminada");
    }
    /*
     * Sobrescreve as propriedades do sistema com as passadas na linha de
     * comando.
     */
    for (Object k : overprops.keySet()) {
      final String key = (String) k;
      final String value = overprops.getProperty(key);
      props.overrideProperty(key, value);
      System.out.println("Carregando propriedades (linha de comando): " + key
        + " = " + value);
    }
    System.out.println(
      " - Carga das propriedades (linha de comando) terminada");
    return props;
  }

  /**
   * Obtm o valor associado a uma propriedade, ou <code>null</code> caso esta
   * no tenha sido definida.
   * 
   * @param propertyName nome da propriedade
   * 
   * @return o valor da propriedade ou <code>null</code> caso esta no tenha
   *         sido definida
   */
  private String getStringPropertyOrNull(String propertyName) {
    if (isPropertyNull(propertyName)) {
      Server.logSevereMessage(String.format(
        "A propriedade %s no foi definida.", propertyName));
      return null;
    }
    return this.getStringProperty(propertyName);
  }

  /**
   * Traduz uma determinada chave na sua string correspondente com o locale
   * padro.
   * 
   * @param key chave do bundle.
   * @return String com o texto respectivo a chave.
   */
  public String getString(String key) {
    return getString(key, getDefaultLocale());
  }

  /**
   * Traduz uma determinada chave na sua string correspondente de acordo com o
   * locale desejado.
   * 
   * @param key chave do bundle.
   * @param locale o locale desejado para internacionalizao.
   * @return String com o texto respectivo a chave.
   */
  public String getString(String key, Locale locale) {
    if (key == null) {
      String f = "Chave nula em consulta  internacionalizao: %s";
      logSevereMessage(String.format(f, key));
      return "<<<null-key>>>";
    }
    if (locale == null) {
      String f = "Locale nulo em consulta  internacionalizao da chave: %s";
      logSevereMessage(String.format(f, key));
      return "<<<null-locale>>>";
    }
    if (!bundles.containsKey(locale)) {
      boolean loaded = loadLanguageBundle(locale);
      if (!loaded) {
        return null;
      }
    }
    try {
      ResourceBundle bnd = bundles.get(locale);
      return bnd.getString(key);
    }
    catch (MissingResourceException mre) {
      return null;
    }
  }

  /**
   * Retorna o bundle do servidor para um locale especfico
   * 
   * @param locale
   * @return bundle de um locale
   */
  HierachicalResourceBundle getBundle(Locale locale) {
    if (locale == null) {
      return null;
    }
    if (!bundles.containsKey(locale)) {
      loadLanguageBundle(locale);
    }
    return bundles.get(locale);
  }

  /**
   * Traduz uma determinada chave na sua string formatada inferindo o locale do
   * usurio (thread) que originou a chamada (ser usado um locale padro se
   * houver falha nesta identificao).
   * 
   * @param key chave do bundle.
   * @param objects array de objetos de formatao.
   * @return string com o texto respectivo a chave.
   */
  public String getFormattedString(String key, Object... objects) {
    String txt = getString(key);
    return MessageFormat.format(txt, objects);
  }

  /**
   * Carrega a propriedade que define a porta do registry RMI.
   * 
   * @return a porta do registry
   */
  private int loadRegistryPort() {
    int port = this.getIntProperty("registryPort");
    String fmt = "Servidor utilizando registry RMI na porta %d";
    Server.logInfoMessage(String.format(fmt, port));
    return port;
  }

  /**
   * Carrega o atributo do nome de charset da mquina.
   * 
   * @return o nome
   */
  private Charset loadHostCharset() {
    final Charset charset = Charset.defaultCharset();
    return charset;
  }

  /*
   * /** Carrega locale inicial do servidor.
   * 
   * @return locale / private Locale loadInitialLocale() { final String
   * paramLocaleText = getStringProperty(); final Locale paramLocale =
   * FormatUtils.parseLocale(paramLocaleText); final Locale defaultLocale =
   * Locale.getDefault(); final Locale initLocale = (paramLocale == null ?
   * defaultLocale : paramLocale); return initLocale; }
   */
  /**
   * Faz log se charset no for IANA.
   * 
   * @param charset charset
   */
  private void logOnNotRegisteredCharset(final Charset charset) {
    final boolean registered = charset.isRegistered();
    if (!registered) {
      final String fmt = "Charset-enconding: [%s] no  registrado IANA!";
      final String chName = charset.name();
      final String msg = String.format(fmt, chName);
      Server.logWarningMessage(msg);
    }
  }

  /**
   * Carrega a propriedade de charset.
   * 
   * @return o charset
   * @throws ServerException em caso de falha de carga e ajuste de charset.
   */
  private Charset loadDefaultCharset() throws ServerException {
    final String chName = this.getStringProperty(PROP_DEFAULT_CHARSET);
    if (!Charset.isSupported(chName)) {
      final String fmt = "Charset-enconding no suportado (prop. %s): [%s]";
      final String msg = String.format(fmt, PROP_DEFAULT_CHARSET, chName);
      throw new ServerException(msg);
    }

    final Charset serverCharset = Charset.forName(chName);
    if (serverCharset != null) {
      final String fmt = "Servidor definido com charset-enconding: [%s]";
      final String msg = String.format(fmt, chName);
      Server.logInfoMessage(msg);
    }
    else {
      final String fmt = "Falha de definio de charset-enconding: [%s]";
      final String msg = String.format(fmt, chName);
      throw new ServerException(msg);
    }

    logOnNotRegisteredCharset(serverCharset);
    return serverCharset;
  }

  /**
   * Carrega a propriedade que define a porta dos objetos RMI exportados.
   * 
   * @return a porta a ser usada para exportar os objetos RMI
   */
  private int loadRMIExportPort() {
    // Objetos exportados
    int port = this.getIntProperty("rmiExportPort");
    String fmt = "Servidor exportanto objetos RMI na porta %d";
    Server.logInfoMessage(String.format(fmt, port));
    return port;
  }

  /**
   * Retorna um LogRecord indentificando o ponto no qual foi gerado.
   * 
   * @param level Nvel de log desejado.
   * @param msg Mensagem de log.
   * @param t A exceo gerada, que pode ser null.
   * @param method O mtodo que realizou a chamada ao mtodo de log.
   * @return O LogRecord.
   */
  private static LogRecord getLogRecord(Level level, String msg, Throwable t,
    StackTraceElement method) {
    LogRecord record = new ServerLogRecord(level, msg, method);
    record.setThrown(t);
    record.setLoggerName(method.getClassName());
    record.setSourceClassName(method.getClassName());
    record.setSourceMethodName(method.getMethodName());
    return record;
  }

  /**
   * Gera uma mensagem de log no nvel informado. Na identificao do ponto onde
   * o log foi solicitado, so desconsiderados os mtodos internos dessa classe
   * usados para gerar o log.
   * 
   * @param level Nvel de log da mensagem.
   * @param msg A mensagem de log.
   * @param t A exceo correspondente, que pode ser nula.
   */
  private static void logMessage(Level level, String msg, Throwable t) {
    StackTraceElement[] stack = Thread.currentThread().getStackTrace();
    StackTraceElement method = stack[3];
    Logger logger = Logger.getLogger(method.getClassName());
    logger.log(getLogRecord(level, msg, t, method));
  }

  /**
   * Gera uma mensagem de log de nvel SEVERE.
   * 
   * @param msg A mensagem de log.
   * @param t A pilha correspondente. Pode ser null.
   */
  public static void logSevereMessage(String msg, Throwable t) {
    logMessage(Level.SEVERE, msg, t);
  }

  /**
   * Gera uma mensagem de log de nvel SEVERE.
   * 
   * @param msg A mensagem de log.
   */
  public static void logSevereMessage(String msg) {
    logMessage(Level.SEVERE, msg, null);
  }

  /**
   * Gera uma mensagem de log de nvel WARNING.
   * 
   * @param msg A mensagem de log.
   */
  public static void logWarningMessage(String msg) {
    logMessage(Level.WARNING, msg, null);
  }

  /**
   * Gera uma mensagem de log de nvel INFO.
   * 
   * @param msg A mensagem de log.
   */
  public static void logInfoMessage(String msg) {
    logMessage(Level.INFO, msg, null);
  }

  /**
   * Gera uma mensagem de log de nvel FINE. Esse nvel deve ser usado para
   * mensagens de depurao.
   * 
   * @param msg A mensagem de log.
   */
  public static void logFineMessage(String msg) {
    logMessage(Level.FINE, msg, null);
  }

  /**
   * Cria a thread que  acionada quando o processo do servidor  interrompido.
   */
  private void setShutdownHook() {
    final Thread t = new Thread(new Runnable() {
      @Override
      public synchronized void run() {
        try {
          stopServerHook();
        }
        catch (final Exception e) {
          e.printStackTrace();
        }
      }
    });
    Runtime.getRuntime().addShutdownHook(t);
  }

  /**
   * Finaliza o ambiente de execuo no servidor.
   */
  public void shutdown() {
    Server.logSevereMessage("Shutdown do servidor acionado!");
    System.exit(0);
  }

  /**
   * Inicia o servidor, inicializando todos os servios locais. Caso esteja
   * definido um servidor central,  feito o o login nele como admin. Note que a
   * conexo ao servidor central s pode ser feita aps a inicializao dos
   * servios locais pois  necessria a senha de admin para efetuar o login.
   * Por outro lado, o monitor tem que exitir antes que os servios sejam
   * inicializados para que os observadores possam ser colocados.
   * 
   * @return indicativo.
   */
  public boolean start() {
    try {
      // Grava no log a verso do servidor
      logInfoMessage("Verso do servidor: " + getVersion());

      if (!sanityCheck()) {
        return false;
      }

      if (!this.isCentralServer()) {
        startCentralServerMonitor();
      }
      startupTime = System.currentTimeMillis();
      startServices();
      startRMI();
      logFineMessage("Servios locais ativos.");
      if (!this.isCentralServer()) {
        startCentralServerConnection();
      }
      logFineMessage("Preparando ps-inicializao do servidor.");
      postInitialization();
      logProperties();
      Server.logInfoMessage("Servidor iniciado.");
      Server.logInfoMessage(listServerLibs());
      System.out.println(getGreetingMessage());
      System.out.println(getServerConfig());
      System.out.println(getServerJVMArgs());
      System.out.println(getVersion());
      if (!isJVMVersionSupported()) {
        Server.logSevereMessage("Verso " + DeploymentInfo.JAVA_VERSION
          + " da JVM no  suportada. So suportas as JVMs do mesmo release com verso acima de "
          + minJVMVersion + ".");
        shutdown();
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha no start-up do servidor", e);
      System.err.println("\n");
      System.err.println("[ERRO] - Falha no start-up do servidor!");
      System.err.println("       > " + e.getMessage());
      System.err.println("\n");
      return false;
    }
    return true;
  }

  /**
   * Testa requisitos mnimos para execuo do servidor.
   * 
   * @return <code>true</code> se a execuo pode prosseguir, <code>false</code>
   *         caso o servidor no possa ser executado
   */
  private boolean sanityCheck() {
    boolean localhostAllowed = getBooleanProperty(PROP_LOCALHOST_ALLOWED);
    if (localhostAllowed) {
      // No precisamos verificar nada
      return true;
    }

    // Verificando casos "localhost", "localhost.localdomain" etc.
    if (serverHostName.startsWith("localhost")) {
      System.err.printf("[ERRO] Nome do servidor est como \"localhost\". "
        + "Verifique a propriedade 'Server.%s' do servidor\n",
        PROP_LOCALHOST_ALLOWED);
      return false;

    }
    try {
      InetAddress localHost = InetAddress.getLocalHost();
      String localhostAddr = localHost.getHostAddress();
      if (localhostAddr.startsWith("127.0")) {
        System.err.printf("[ERRO] IP do servidor associado a localhost (%s). "
          + "Verifique a propriedade 'Server.%s' do servidor\n", localhostAddr,
          PROP_LOCALHOST_ALLOWED);
        return false;
      }
      String hostAddr = getHostAddr();
      if (hostAddr.startsWith("127.0")) {
        System.err.printf("[ERRO] propriedade %s associada a localhost (%s).\n"
          + "Verifique a propriedade 'Server.%s' do servidor.\n",
          PROP_HOST_ADDR, hostAddr, PROP_LOCALHOST_ALLOWED);
        return false;
      }
    }
    catch (UnknownHostException e) {
      e.printStackTrace();
      return false;
    }
    return true;
  }

  /**
   * Registra no log do servidor (com prioridade INFO) todas as propriedades,
   * agrupando-as por propriedades redefinidas no System.properties,
   * propriedades apenas no System.properties, e propriedades com valores
   * default.
   */
  private void logProperties() {
    logInfoMessage('\n' + listRuntimeProperties());
  }

  /**
   * Obtm todas as propriedades runtime como uma string.
   * 
   * @return string com todas as propriedades
   */
  public String listRuntimeProperties() {
    Collection<Service> services = Service.getServices().values();
    SortedSet<Object> allProps = new TreeSet<Object>(serverProperties
      .getPropertiesKeys());
    SortedSet<Object> overridenProps = new TreeSet<Object>();
    SortedSet<Object> newProps = new TreeSet<Object>();
    for (Service service : services) {
      allProps.addAll(service.getPropertiesKeys());
    }
    StringBuilder builder = new StringBuilder();
    if (systemProperties != null) {
      for (Object prop : systemProperties.getPropertiesKeys()) {
        if (allProps.contains(prop)) {
          /*
           * propriedade foi redefinida no System.properties
           */
          overridenProps.add(prop);
          allProps.remove(prop);
        }
        else {
          /*
           * propriedade foi definida apenas no System.properties
           */
          newProps.add(prop);
        }
      }
      String title = "\nPropriedades REDEFINIDAS em " + systemFileName;
      buildPropsListOutput(builder, title, overridenProps, systemProperties);
      title = "\nPropriedades APENAS em " + systemFileName;
      buildPropsListOutput(builder, title, newProps, systemProperties);
    }
    builder.append(
      "\nPropriedades com valores default:\n============================\n");
    /*
     * para cada uma das propriedades restantes, procuramos em cada um dos
     * servios pela sua definio default
     */
    for (Object prop : allProps) {
      for (Service service : services) {
        String defaultValue = service.getDefaultPropertyValue((String) prop);
        if (defaultValue != null) {
          builder.append(prop);
          builder.append(" = ");
          builder.append(defaultValue);
          builder.append('\n');
        }
      }
    }
    return builder.toString();
  }

  /**
   * Obtm um mapa com as propriedades runtime do servidor.
   * 
   * @return mapa com as propriedades runtime do servidor
   */
  public Map<String, String> getRuntimeProperties() {
    Map<String, String> result = new HashMap<String, String>();
    /*
     * primeiro pegamos as propriedades definidas para os servios (valores
     * default)
     */
    Collection<Service> services = Service.getServices().values();
    for (Service service : services) {
      result.putAll(service.getPropertiesMap());
    }
    /*
     * agora sobrescrevemos com as propriedades definidas em Server.properties
     */
    result.putAll(serverProperties.getPropertiesMap());
    /*
     * finalmente, sobrescrevemos com as propriedades do System.properties
     */
    if (systemProperties != null) {
      result.putAll(systemProperties.getPropertiesMap());
    }
    return result;
  }

  public Map<String, PropertyInfo> getPropertiesInfo() {
    Map<String, String> runtimeProperties = this.getRuntimeProperties();
    Map<String, PropertyInfo> result = new Hashtable<String, PropertyInfo>();
    Collection<Service> services = Service.getServices().values();
    for (Service service : services) {
      SortedSet<Object> keys = service.getPropertiesKeys();
      for (Object key : keys) {
        String propertyKey = key.toString();
        Map<String, String> serviceProperties = service.getPropertiesMap();
        String serviceValue =
          serviceProperties.containsKey(propertyKey) ? serviceProperties
            .get(propertyKey) : null;
        String serverValue =
          serverProperties.hasProperty(propertyKey) ? serverProperties
            .getStringProperty(propertyKey) : null;
        String systemValue = systemProperties != null && systemProperties
          .hasProperty(propertyKey) ? systemProperties.getStringProperty(
            propertyKey) : null;
        String commandLineValue =
          commandLineProperties != null
            && commandLineProperties.containsKey(propertyKey) ? commandLineProperties
            .getProperty(propertyKey) : null;
        String runtimeValue = runtimeProperties.get(propertyKey);
        PropertyInfo info = new PropertyInfo(propertyKey, runtimeValue,
          serviceValue, serverValue, systemValue, commandLineValue);
        result.put(propertyKey, info);
      }
    }
    return result;
  }

  /**
   * Acrescenta a um builder strings no formato "nome = valor" para um
   * determinado conjunto de nomes de propriedades; os valores so obtidos de um
   * conjunto de propriedades.
   * 
   * @param builder builder
   * @param title ttulo que precede a lista
   * @param propsKeys chaves (nomes) das propriedades
   * @param props conjunto de propriedades. Deve possuir valores para todos os
   *        nomes em <code>propsKeys</code>
   */
  private void buildPropsListOutput(StringBuilder builder, String title,
    SortedSet<Object> propsKeys, ServerSideProperties props) {
    builder.append(title);
    builder.append("\n============================\n");
    for (Object p : propsKeys) {
      builder.append(p);
      builder.append(" = ");
      builder.append(props.getStringProperty((String) p));
      builder.append('\n');
    }
  }

  /**
   * Verifica se a verso da JVM  suportada para a execuo do CSBase.
   * 
   * @return flag indicando se a verso  suportada
   */
  private boolean isJVMVersionSupported() {
    Pattern pattern = Pattern.compile("(\\d+)\\.(\\d+)\\.(\\d+)_?(\\d+)?");
    Matcher curVersionMatcher = pattern.matcher(DeploymentInfo.JAVA_VERSION);
    Matcher minVersionMatcher = pattern.matcher(minJVMVersion);

    if ((curVersionMatcher.matches()) && (minVersionMatcher.matches())) {

      // Major Version (da corrente deve ser igual a mnima)
      if (!curVersionMatcher.group(1).equals(minVersionMatcher.group(1))) {
        return false;
      }

      // Minor Version (da corrente deve ser maior ou igual a mnima)
      int currentMinor = Integer.parseInt(curVersionMatcher.group(2));
      int minimumMinor = Integer.parseInt(minVersionMatcher.group(2));
      if (currentMinor < minimumMinor) {
        return false;
      }
      else if (currentMinor > minimumMinor) {
        return true;
      }

      // major version e minor version so iguais, verificar patch version

      // Patch Version (da corrente deve ser maior ou igual da mnima)
      StringBuilder cur = new StringBuilder();
      for (int i = 3; i <= curVersionMatcher.groupCount(); i++) {
        Integer num = 0;
        if (curVersionMatcher.group(i) != null) {
          num = Integer.parseInt(curVersionMatcher.group(i));
        }
        cur.append(String.format("%02d", num));
      }

      StringBuilder min = new StringBuilder();
      for (int i = 3; i <= minVersionMatcher.groupCount(); i++) {
        Integer num = 0;
        if (minVersionMatcher.group(i) != null) {
          num = Integer.parseInt(minVersionMatcher.group(i));
        }
        min.append(String.format("%02d", num));
      }

      int curInt = Integer.parseInt(cur.toString());
      int minInt = Integer.parseInt(min.toString());

      if (curInt < minInt) {
        return false;
      }
    }
    return true;
  }

  /**
   * Obtm as informaes de configurao do servidor. Por default mostra o IP
   * (Server.hostAddr), nome (Server.hostName) e portas RMI (Server.registryPort
   * e Server.rmiExportPort), mas os sistemas podem redefinir este mtodo para
   * mostrar mais (ou menos) informaes.
   * 
   * @return informaes de configurao do servidor
   */
  protected String getServerConfig() {
    final int regPort = getRegistryPort();
    final int rmiExPort = getRMIExportPort();
    final String rmiRegStr = String.format("%d", regPort);
    final String rmiRegExp = String.format("%d", rmiExPort);

    String sysChText = "???";
    final Charset sysCh = getSystemDefaultCharset();
    if (sysCh != null) {
      sysChText = getSystemDefaultCharsetName();
    }

    String hostChText = "???";
    final String htChName = getServerHostCharsetName();
    if (htChName != null) {
      hostChText = htChName;
    }

    final String[][] cnfInfos = { { "Endereo IP", getHostAddr() }, { "Nome",
        getHostName() }, { "RMI [registry]", rmiRegStr }, { "RMI [export]",
            rmiRegExp }, { "Charset [mquina]", hostChText }, {
                "Charset [servidor/sistema]", sysChText }, };

    String info = "";
    for (String[] cnfInfo : cnfInfos) {
      final String label = cnfInfo[0];
      final String value = cnfInfo[1];
      info += String.format("  -- %-30s: %s\n", label, value);
    }
    info += String.format("  -- %-30s: %s %s %s\n", "Sistema Operacional",
      DeploymentInfo.OS_NAME, DeploymentInfo.OS_ARCH,
      DeploymentInfo.OS_VERSION);
    info += String.format("  -- %-30s: %s %s (%s)\n", "Java",
      DeploymentInfo.JAVA_NAME, DeploymentInfo.JAVA_VERSION,
      DeploymentInfo.JAVA_VENDOR);
    return info;
  }

  /**
   * Lista as bibliotecas utilizadas pelo servidor
   * 
   * @return as bibliotecas carregadas pelo class loader
   */
  private String listServerLibs() {
    StringBuffer libsToShow = new StringBuffer("Bibliotecas carregadas:\n");
    String[] serverLibs = getServerLibs();
    for (String lib : serverLibs) {
      libsToShow.append(lib);
      libsToShow.append("\n");
    }
    return libsToShow.toString();
  }

  /**
   * Obtm as bibliotecas utilizadas pelo servidor
   * 
   * @return as bibliotecas carregadas pelo class loader
   */
  public String[] getServerLibs() {
    StringTokenizer st = new StringTokenizer(System.getProperty(
      "java.class.path"), System.getProperty("path.separator"));
    List<String> serverLibs = new ArrayList<String>();
    while (st.hasMoreTokens()) {
      String token = st.nextToken();
      if (token.endsWith(".jar")) {
        int lIdx = token.lastIndexOf(System.getProperty("file.separator"));
        serverLibs.add(token.substring(lIdx + 1));
      }
    }
    return serverLibs.toArray(new String[0]);
  }

  /**
   * Obtm os argumentos utilizados execuo da JVM
   * 
   * @return argumentos passados para a JVM
   */
  private String getServerJVMArgs() {
    String JVMArgs = "Argumentos da JVM:\n";
    List<String> arguments = ManagementFactory.getRuntimeMXBean()
      .getInputArguments();
    for (String arg : arguments) {
      JVMArgs += "  " + arg + "\n";
    }
    return JVMArgs;
  }

  /**
   * Inicia a comunicao com o servidor central.
   */
  private void startCentralServerConnection() {
    final String centralServerName = getCentralServerName();
    if (centralServerName == null) {
      return;
    }
    LocalServerRemoteMonitor.getInstance().setSystemName(MessageFormat.format(
      "Servidor local {0}", this.getSystemName()));
    Server.logInfoMessage("Iniciando conexo ao servidor central: "
      + centralServerName);
    LocalServerRemoteMonitor.getInstance().start();
    Server.logInfoMessage("Conexo ao servidor central concluda.");
  }

  /**
   * Inicia a classe que ser usada pelo servidor local para monitorar o estado
   * da conexo com um servidor central.
   */
  public abstract void startCentralServerMonitor();

  /**
   * Cria a porta de entrada para acesso as funcionalidade do servidor e a
   * exporta via rmi
   * 
   * @throws ServerException caso ocorra alguma falha na exportao via rmi
   */
  private void startRMI() throws ServerException {
    try {
      this.entryPoint = createServerEntryPoint();
      UnicastRemoteObject.exportObject(this.entryPoint, rmiExportPort);
      registry.rebind(ServerEntryPoint.LOOKUP, this.entryPoint);
    }
    catch (final Exception e) {
      throw new ServerException("Falha no export/bind.", e);
    }
  }

  /**
   * Cria os servios e os inicia em seguida.
   * 
   * @throws ServerException caso ocorra alguma falha na criao ou na
   *         inicializao dos servios
   */
  private void startServices() throws ServerException {
    ServiceManager.init();
    createServices();
    try {
      // Durante a partida, o usurio  o admin.
      Service.setUserId(User.getAdminId());
      ServiceManager.getInstance().initAllServices();
    }
    finally {
      Service.setUserId(null);
    }
  }

  /**
   * Mtodo chamado quando pela thread acionada quando o processo do servidor 
   * interrompido.
   */
  private void stopServerHook() {
    System.out.println("Finalizando o servidor.");
    System.out.println("Acionado stop server hook!");
  }

  /**
   * Retorna a instncia do servidor.
   * 
   * @return a instncia
   */
  public static Server getInstance() {
    if (instance == null) {
      System.out.println("SERVIDOR NO INSTANCIADO!");
    }
    return instance;
  }

  /**
   * Mtodo de consulta ao nome do diretrio-raiz de dados persistidos.
   * 
   * @return uma string com o nome do diretrio-raiz.
   */
  public String getPersistencyRootDirectoryName() {
    final String dirName = getStringProperty("persistency.directory");
    return dirName;
  }

  /**
   * Mtodo de consulta ao nome do diretrio de execuo do servidor.
   * 
   * @return uma string com o nome do diretrio de execuo do servidor.
   */
  public String getRunningDirectoryName() {
    final String dirName = System.getProperty("user.dir");
    return dirName;
  }

  /**
   * Mtodo de consulta ao nome do diretrio-raiz de propriedades.
   * 
   * @return uma string com o nome do diretrio-raiz.
   */
  public static String getPropertiesRootDirectoryName() {
    return FileSystem.PROPERTIES_DIR;
  }

  /**
   * Ponto de entrada da aplicao que inicia o servidor do sistema. Instala um
   * SecurityManager e configura o provedor de objetos para atender ao servidor.
   * 
   * @param args argumentos da linha de comando do servidor.
   * @throws ServerException em caso de falha.
   */
  protected Server(final String[] args) throws ServerException {
    commandLineProperties = parseCommandLineArgs(args);
    /*
     * Define o locale nativo como sendo pt_BR. Isso garante que temos arquivos
     * de bundle necessrios para todos os textos.
     */

    try {
      if (instance != null) {
        final String msg = getString("Server.argumentIgnoredserver");
        throw new ServerException(msg);
      }

      setShutdownHook();

      // Carga das propriedades do servidor
      this.serverProperties = loadAndSetServerProperties();
      this.systemProperties = loadAndSetSystemProperties(commandLineProperties);

      /*
       * Define o locale nativo como sendo pt_BR. Isso garante que temos
       * arquivos de bundle necessrios para todos os textos. /
       * 
       * LNG.setTranslationListener(new TranslationListener() {
       * 
       * @Override public String keyNotFound(String key, String text) {
       * keyNotFoundInAnyIdiom.put(key, text); return text; }
       * 
       * @Override public String keyNotFoundInDefaultLanguage(String key, String
       * text) { keyNotFoundInSelectedIdiom.put(key, text); return text; } });
       */
      // Inicia sistema de log
      initLog();

      // Criao do diretrio de peristncia (se necessrio)
      final String persistDirPath = getPersistencyRootDirectoryName();
      checkDirectory(persistDirPath);

      // Definio de charset default
      this.serverHostCharset = loadHostCharset();
      this.systemDefaultCharset = loadDefaultCharset();

      // Ajuste de locale
      this.defaultLocale = loadAndSetDefaultLocale();

      // Portas RMI
      this.registryPort = loadRegistryPort();
      setupRegistry();
      this.rmiExportPort = loadRMIExportPort();

      // nome do servidor
      this.serverHostName = loadAndSetServerHostName();
      // IP do servidor
      this.serverHostAddr = getStringProperty(PROP_HOST_ADDR);
      // nome do sistema
      this.systemName = getStringPropertyOrNull(PROP_HOST_NAME);
      // verso mnima da JVM suportada pelo servidor
      this.minJVMVersion = getStringProperty(PROP_MIN_SUPPORTED_JVM_VERSION);
      /*
       * Torna os identificadores dos objetos remotos aleatrios. Essa
       * caracterstica  a base do mecanismo de autenticao e autorizao,
       * garantindo que um usurio s consegue fazer chamadas aos proxies
       * criados especificamente para ele.
       */
      System.setProperty("java.rmi.server.randomIDs", "true");

      // Preparando chaves
      this.createKeyStore();
      this.privateKeyPassword = loadAndSetPrivateKeyPassword();

      // Cria o gerente de mensagens.
      this.messageBroker = createMessageBroker();

      instance = this;
    }
    catch (RuntimeException rte) {
      throw new ServerException(rte);
    }
  }

  /**
   * Faz parser dos argumentos
   * 
   * @param args os argumentos
   * @return properties relativas aos argumentos.
   */
  private Properties parseCommandLineArgs(String[] args) {
    final Properties props = new Properties();
    if (args == null) {
      return props;
    }
    final int nargs = args.length;
    int i = 0;
    while (i < nargs) {
      final String arg = args[i];
      if (arg.equals("-p")) {
        i++;
        systemFileName = (i < nargs) ? args[i] : null;
      }
      else if (arg.equals("-override")) {
        i++;
        final String key = (i < nargs) ? args[i++] : null;
        final String value = (i < nargs) ? args[i] : null;
        if (key != null && value != null) {
          props.setProperty(key, value);
        }
      }
      else {
        if (props.containsKey(PROP_LOCALE)) {
          System.out.println(getString("Server.argumentIgnored") + arg + "...");
        }
        System.out.println("Ignorando argumento " + arg + "...");
      }
      i++;
    }
    return props;
  }

  /**
   * Faz a carga da chave primria do servidor.
   * 
   * @return a chave privada ou <code>null</code> se a mesma no estiver setada.
   */
  private String loadAndSetPrivateKeyPassword() {
    final String propertyName = "privateKeyPassword";
    final boolean isNull = isPropertyNull(propertyName);
    if (isNull) {
      final String fmt = getString("Server.keyNotFound");
      final String message = String.format(fmt, propertyName);
      Server.logWarningMessage(message);
      return null;
    }
    final String pwd = getStringProperty(propertyName);
    return pwd;
  }

  /**
   * Verifica se existe, ou  possvel criar, um registry RMI na porta
   * especificada.
   * 
   * @throws ServerException se a checagem falhar
   */
  private void setupRegistry() throws ServerException {
    if (!getRegistry()) {
      final String err = getString("Server.RMIRegNotfound") + registryPort;
      Server.logSevereMessage(err);
      throw new ServerException(err);
    }
  }

  /**
   * Faz a carga do nome do hostname (configura o endereo utilizado para
   * exportar objetos RMI do servidor).
   * 
   * @return o host name do servidor.
   * @throws ServerException se endereo IP no for encontrado.
   */
  private String loadAndSetServerHostName() throws ServerException {
    String hostname;
    final String propertyName = "hostName";
    final boolean isNull = isPropertyNull(propertyName);
    if (isNull) {
      final String info = getString("Server.hostnameNotFound");
      Server.logInfoMessage(info);

      try {
        final InetAddress localHost = InetAddress.getLocalHost();
        hostname = localHost.getCanonicalHostName();
      }
      catch (final UnknownHostException e) {
        final String msg = getString("Server.IPNotFound");
        Server.logSevereMessage(msg);
        throw new ServerException(msg, e);
      }
    }
    else {
      hostname = getStringProperty(propertyName);
    }

    System.setProperty("java.rmi.server.hostname", hostname);
    Server.logInfoMessage("Propriedade hostname do servidor: " + hostname);
    return hostname;
  }

  /**
   * Mtodo chamado aps a inicializao dos servios.
   * 
   * @throws InitFailureException caso ocorra alguma falha na ps inicializao
   */
  public abstract void postInitialization() throws InitFailureException;

  /**
   * Obtm o timestamp da inicializao do servidor.
   * 
   * @return timestamp da inicializao do servidor
   */
  public long getStartupTime() {
    return startupTime;
  }

  /**
   * Obtm o <i>broker</i> de mensagens.
   * 
   * @return o <i>broker</i> de mensagens.
   */
  public MessageBroker getMessageBroker() {
    return messageBroker;
  }

  /**
   * Cria o <i>broker</i> de mensagens.
   * 
   * @return um novo <i>broker</i> de mensagens.
   */
  private MessageBroker createMessageBroker() {

    // Obtm os parmetros para a configurao do broker de mensagem. 
    long persistFileMaxSize = getLongProperty("message.persist.file.maxsize");
    long persistPeriod = TimeUnit.SECONDS.toMillis(getLongProperty(
      "message.persist.period"));
    long receiveTimeout = TimeUnit.SECONDS.toMillis(getLongProperty(
      "message.receive.timeout"));
    int maxThreads = getIntProperty("message.threads.max");

    /*
     * Cria o objeto IMessageStoreDAO, objeto utilizado para persistir as
     * mensagens.
     */
    String messageBkpFileName = "messages.dat";
    File messagesBkpFile;
    try {
      String pName = getPersistencyRootDirectoryName();
      String dName = pName + File.separator + "messages";
      checkDirectory(dName);
      messagesBkpFile = new File(dName + File.separator + messageBkpFileName);
    }
    catch (Throwable t) {
      Server.logSevereMessage("Falha de aquisio das mensagens persistidas.",
        t);
      messagesBkpFile = new File(messageBkpFileName);
    }
    if (messagesBkpFile.exists()) {
      // Deleta o arquivo de persistncia se ele for grande demais.
      long size = messagesBkpFile.length() / 1024 / 1024;
      if (persistFileMaxSize < size) {
        if (messagesBkpFile.delete()) {
          Server.logWarningMessage(
            "Removendo arquivo de persistncia de mensagens por ser grande demais ("
              + size + "MB).");
        }
      }
    }

    IMessageStoreDAO store = new MessageStoreDAO(messagesBkpFile);

    // Cria o broker de mensagens.
    return new MessageBroker(store, persistPeriod, receiveTimeout, maxThreads);
  }
}
