/*
 * $Id$
 */
package csbase.server.services.dbmanagerservice;

import java.io.File;
import java.io.FileInputStream;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Iterator;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import csbase.server.Server;
import csbase.server.ServerException;

/**
 * Classe que implementa a definio de um pool para conexo com um banco de
 * dados qualquer.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class Pool {

  /**
   * nome identificador do pool
   */
  protected String name;

  /**
   * url jdbc para conexo com o banco
   */
  protected String url;

  /**
   * Usurio do banco para conexo
   */
  protected String user;

  /**
   * Senha do banco para conexo
   */
  protected String password;

  /**
   * Driver do banco
   */
  protected String driver;

  /**
   * Propriedades do pool carregadas pelo arquivo de propriedades.
   */
  private Properties properties = null;

  /**
   * Logger a ser usado pelo pool.
   */
  public Logger logger;

  /**
   * Verifica se a senha utilizada com o banco mudou.
   * 
   * @param passwd A senha para testar.
   */
  public abstract void checkPassword(String passwd);

  /**
   * Finaliza o pool
   */
  public abstract void destroy();

  /**
   * @return Retorna uma conexo do pool
   */
  public abstract Connection getConnection();

  /**
   * Consulta o driver de conexo com o banco
   * 
   * @return o driver
   */
  public final String getDriver() {
    return driver;
  }

  /**
   * @param key a chave da propriedade
   * @return Retorna uma propriedade do tipo int do pool
   * @throws ServerException em caso de falha do servidor.
   */
  public final int getIntProperty(final String key) throws ServerException {
    try {
      return Integer.parseInt(getProperty(key));
    }
    catch (final Exception e) {
      logger.log(Level.SEVERE, "Erro lendo propriedade: " + key);
      throw new ServerException(e);
    }
  }

  /**
   * @return Retorna o logger do pool
   */
  public final Logger getLogger() {
    return logger;
  }

  /**
   * @param key chave da propriedade
   * @return Retorna uam propriedade do tipo long do pool
   * @throws ServerException em caso de falha do servidor.
   */
  public final long getLongProperty(final String key) throws ServerException {
    try {
      return Long.parseLong(getProperty(key));
    }
    catch (final Exception e) {
      logger.log(Level.SEVERE, "Erro lendo propriedade: " + key, e);
      throw new ServerException(e);
    }
  }

  /**
   * Nome identificador deste pool
   * 
   * @return o nome
   */
  public final String getName() {
    return name;
  }

  /**
   * Retorna senha utilizada por este pool
   * 
   * @return a senha
   */
  public final String getPassword() {
    return password;
  }

  /**
   * @return Propriedades deste pool
   */
  public final Properties getProperties() {
    return properties;
  }

  /**
   * @param key chave da propriedade
   * @return Retorna uma propriedade do pool
   * @throws ServerException em caso de falha do servidor.
   */
  public final String getProperty(final String key) throws ServerException {
    final String v = properties.getProperty(key);
    if (v == null) {
      logger.log(Level.SEVERE, "Erro lendo propriedade: " + key);
      throw new ServerException("Propriedade no encontrada: " + key);
    }
    return v;
  }

  /**
   * Retorna url de conexo com o banco
   * 
   * @return url.
   */
  public final String getUrl() {
    return url;
  }

  /**
   * Retorna o surio de conexo com o banco
   * 
   * @return o usurio
   */
  public final String getUser() {
    return user;
  }

  /**
   * Inicializa o pool
   * 
   * @return true se inicializao feita com sucesso
   */
  public abstract boolean init();

  /**
   * Inicializao do log de execuo do pool.
   * 
   * @throws ServerException se houver falha de inicializao.
   */
  public void initServiceLog()
    throws ServerException {
    try {
      logger = Logger.getLogger(getClass().getName());
    }
    catch (final Exception e) {
      throw new ServerException(e);
    }
  }

  /**
   * Carrega as propriedades do pool contidas no arquivo que possui o mesmo nome
   * da classe que implementa Pool.
   * 
   * @param propertyFileName o arquivo
   * @throws ServerException em caso de falha no servidor
   */
  private void loadProperties(final String propertyFileName)
    throws ServerException {
    properties = new Properties();
    try {
      final FileInputStream in = new FileInputStream(propertyFileName);
      properties.load(in);
      in.close();
      final Properties trimmedProps = new Properties();
      final Iterator<Object> it = properties.keySet().iterator();
      while (it.hasNext()) {
        final String k = (String) it.next();
        final String v = properties.getProperty(k);
        trimmedProps.setProperty(k, v.trim());
      }
      properties = trimmedProps;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Erro lendo " + propertyFileName, e);
      throw new ServerException(e);
    }
  }

  /**
   * Devolve a conexo para o pool
   * 
   * @param conn conexo
   * @param stmt statement
   * @param rs result set.
   */
  public abstract void releaseConnection(Connection conn, Statement stmt,
    ResultSet rs);

  /**
   * Devolve a conexo para o pool
   * 
   * @param conn conexo
   * @param stmt statement
   * @param rs result set.
   * @param dispose se true a conexo ser descartada.
   */
  public abstract void releaseConnection(Connection conn, Statement stmt,
    ResultSet rs, boolean dispose);

  /**
   * Driver usado pelo pool
   * 
   * @param driver driver
   */
  public void setDriver(final String driver) {
    this.driver = driver;
  }

  /**
   * A senha para conexo no banco
   * 
   * @param password senha
   */
  public final void setPassword(final String password) {
    this.password = password;
  }

  /**
   * Url jdbc
   * 
   * @param url url
   */
  public final void setUrl(final String url) {
    this.url = url;
  }

  /**
   * Usurio do banco
   * 
   * @param user usurio
   */
  public final void setUser(final String user) {
    this.user = user;
  }

  /**
   * Inicializa logger e carrega properties
   * 
   * @param name Nome dado a este pool
   * 
   * @throws ServerException em caso de falha no servidor.
   */
  public Pool(final String name) throws ServerException {
    this.name = name;
    initServiceLog();
    final String propertyFileName =
      Server.getPropertiesRootDirectoryName() + File.separator
        + this.getClass().getSimpleName() + ".properties";
    loadProperties(propertyFileName);
    Server.logInfoMessage(properties.toString());
  }
}
