/**
 * $Id: Engine.java 141791 2013-06-12 14:54:50Z fpina $
 */

package csbase.server.services.diagnosticservice.engine;

import java.util.Hashtable;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import csbase.logic.diagnosticservice.ResourceException;
import csbase.logic.diagnosticservice.Status;
import csbase.logic.diagnosticservice.StatusCode;
import csbase.server.services.diagnosticservice.monitors.Monitor;
import csbase.server.services.diagnosticservice.monitors.ProactiveMonitor;

/**
 * Classe responsvel por obter o status dos recursos atravs dos monitores
 * cadastrados.
 * 
 * @author Tecgraf
 */
public class Engine {
  /**
   * O locale default
   */
  private static Locale locale = Locale.getDefault();
  /**
   * Executor da verificao
   */
  private ExecutorService executor;
  /**
   * Lista de monitores sob responsabilidade da engine.
   */
  private Map<String, Monitor> monitors = new Hashtable<String, Monitor>();
  /**
   * Mapa com o ltimo estado de cada monitor.
   */
  private Map<String, Status> monitorStatus = new Hashtable<String, Status>();

  /**
   * Construtor padro. O intervalo de atualizao padro  10 segundos.
   */
  public Engine() {
    this.executor = Executors.newSingleThreadExecutor();
  }

  /**
   * Obtm o locale que deve ser usado nas mensagens.
   * 
   * @return o locale
   */
  public static Locale getLocale() {
    return Engine.locale;
  }

  /**
   * Define o locale que deve ser utilizado nas mensagens.
   * 
   * @param locale o locale
   */
  public static void setLocale(Locale locale) {
    Engine.locale = locale;
  }

  /**
   * Adiciona um {@link Monitor}.
   * 
   * @param monitor monitor a adicionar
   * 
   * @throws ResourceException caso o recurso j esteja sendo monitorado
   */
  public void addMonitor(Monitor monitor) throws ResourceException {
    synchronized (monitorStatus) {
      if (ProactiveMonitor.class.isInstance(monitor)) {
        ProactiveMonitor.class.cast(monitor).registerEngine(
          new UpdateHandlerImpl(monitor.getResourceName()));
      }
      if (monitorStatus.containsKey(monitor.getResourceName())) {
        throw new ResourceException("O recurso " + monitor.getResourceName()
          + " j est sendo monitorado");
      }
      //Estado inicial de um monitor  StatusCode.INITIATING
      monitorStatus.put(monitor.getResourceName(),
        new Status(monitor.getResourceName(), StatusCode.INITIATING));
      monitors.put(monitor.getResourceName(), monitor);
    }

  }

  /**
   * Obtm os estados de todos os recursos.
   * 
   * @param locale o locale para traduo dos resultados da verificao dos
   *        recursos.
   * @return os estados de todos os recursos monitorados
   * @throws ResourceException se houver um erro na verificao dos recursos
   */
  public Status[] getAllStatus(final Locale locale) throws ResourceException {
    Callable<Void> worker = new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        for (Monitor monitor : monitors.values()) {
          monitorStatus.put(monitor.getResourceName(),
            monitor.checkResource(locale));
        }
        return null;
      }
    };
    Future<Void> future = executor.submit(worker);
    try {
      future.get();
      return monitorStatus.values().toArray(new Status[0]);
    }
    catch (Exception e) {
      throw new ResourceException("Erro na verificao dos recursos", e);
    }
  }

  /**
   * Obtm o estado de um monitor.
   * 
   * @param resourceName o nome do recurso
   * @param locale o locale para traduo dos resultados da verificao do
   *        recurso.
   * 
   * @return o estado atualizado do recurso
   * 
   * @throws ResourceException se houver um erro na verificao do recurso
   */
  public Status getStatus(final String resourceName, final Locale locale)
    throws ResourceException {
    Callable<Status> worker = new Callable<Status>() {
      @Override
      public Status call() throws Exception {
        Monitor monitor = monitors.get(resourceName);
        Status status = monitor.checkResource(locale);
        monitorStatus.put(monitor.getResourceName(), status);
        return status;
      }
    };
    Future<Status> future = executor.submit(worker);
    try {
      return future.get();
    }
    catch (Exception e) {
      throw new ResourceException("Erro na verificao do recurso "
        + resourceName, e);
    }
  }

  /**
   * Classe para atualizar o estado de monitores proativos na engine.
   */
  private class UpdateHandlerImpl implements UpdateHandler {
    /**
     * Nome do recurso
     */
    private String resourceName;

    /**
     * Construtor que recebe o nome do recurso.
     * 
     * @param resourceName nome do recurso
     */
    private UpdateHandlerImpl(String resourceName) {
      this.resourceName = resourceName;
    }

    /**
     * Callback para atualizar o estado de um monitor. Este metodo  chamado por
     * monitores do tipo {@link ProactiveMonitor}.
     * 
     * @param status o estado do recurso
     */
    @Override
    public void updateStatus(Status status) {
      monitorStatus.put(resourceName, status);
    }
  }
}
