/*
 * $Id: FileTransferService.java 115229 2011-02-03 19:57:31Z cassino $
 */

package csbase.server.services.filetransferservice;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import csbase.exception.ServiceFailureException;
import csbase.logic.User;
import csbase.logic.filetransferservice.FileTransferConnection;
import csbase.logic.filetransferservice.FileTransferElement;
import csbase.logic.filetransferservice.FileTransferPredefinedConnection;
import csbase.logic.filetransferservice.FileTransferProtocol;
import csbase.logic.filetransferservice.FileTransferRequest;
import csbase.logic.filetransferservice.FileTransferRequestStatus;
import csbase.logic.filetransferservice.FileTransferRequestType;
import csbase.remote.FileTransferServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.filetransferservice.ftp.FTPStub;
import csbase.server.services.filetransferservice.sftp.SFTPStub;

/**
 * Servio de transferncia de arquivos
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class FileTransferService extends Service implements
  FileTransferServiceInterface {

  /**
   * Tamanho dos chunks de transferncias 
   */
  final public static int CHUNK_SIZE = 64*1024;
  
  /**
   * Lista das requisies
   */
  final private FileTransferRequestQueue requestQueue =
    new FileTransferRequestQueue();

  /**
   * Thread de monitorao de requisies pendentes.
   */
  final private FileTransferServiceMonitorThread monitorThread;

  /**
   * Mapeamento de protocolos com objetos que implementam a requisio.
   * 
   * @see FileTransferProtocolStub
   */
  final private HashMap<FileTransferProtocol, FileTransferProtocolStub> protocolHash =
    new HashMap<FileTransferProtocol, FileTransferProtocolStub>();

  /**
   * Lista de connexes predefinidas do servio.
   */
  final private List<FileTransferPredefinedConnection> predefinedConnections =
    new ArrayList<FileTransferPredefinedConnection>();

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean checkConnection(final FileTransferConnection connection) {
    final FileTransferProtocol protocol = connection.getProtocol();
    final FileTransferProtocolStub stub = protocolHash.get(protocol);
    try {
      return stub.checkConnection(connection);
    }
    catch (final Exception e) {
      System.out.println(e.getMessage());
      throw new ServiceFailureException(e.getMessage());
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void download(final FileTransferConnection connection,
    final String remoteFilePath, final String[] localFilePath, final long size)
    throws ServiceFailureException {
    final FileTransferRequest request =
      new FileTransferRequest(FileTransferRequestType.DOWNLOAD, connection,
        remoteFilePath, localFilePath, size);

    try {
      queueRequest(request);
    }
    catch (final Exception e) {
      final String msg = e.getMessage();
      if (request.getStatus() != FileTransferRequestStatus.ERROR) {
        request.markError(msg);
      }
      throw new ServiceFailureException(msg);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FileTransferRequest> getAllRequests(final User user)
    throws ServiceFailureException {
    final List<FileTransferRequest> list = requestQueue.getAllRequests(user);
    return list;
  }

  /**
   * Consulta a fila de requisies.
   * 
   * @return a fila
   */
  final FileTransferRequestQueue getRequestQueue() {
    return requestQueue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    loadPredefinedConnections();
  }

  /**
   * Faz a carga de conexes predefinidas
   */
  private void loadPredefinedConnections() {
    final String prefix = "predefined.connections.";
    final List<String> labels = getStringListProperty(prefix + "label");
    int i = 1;
    for (final String label : labels) {
      Server.logInfoMessage("Conexo predefinida encontrada: "+label);
      final String serverName = getStringProperty(prefix + "server"+"."+i);
      final String userName = getStringProperty(prefix + "user"+"."+i);
      final String protName = getStringProperty(prefix + "protocol"+"."+i);
      final String homeString = getStringProperty(prefix + "home"+"."+i);
      final FileTransferProtocol protocol =
        FileTransferProtocol.valueOf(protName);
      final FileTransferPredefinedConnection preCon =
        new FileTransferPredefinedConnection(label, protocol, serverName,
          userName, homeString);
      predefinedConnections.add(preCon);
      i = i + 1;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void interruptRequest(final List<String> idList)
    throws ServiceFailureException {
    for (final String id : idList) {
      final FileTransferRequest req = requestQueue.getRequestFromId(id);
      if (req != null) {
        req.markCancelled();
      }
      requestQueue.moveToEndedList(id);
    }
  }

  /**
   * Move requisio para lista de terminados.
   * 
   * @param requestId id da requisio.
   */
  void moveToEndedList(final String requestId) {
    requestQueue.moveToEndedList(requestId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FileTransferElement> listContent(
    final FileTransferConnection connection, final String remoteDirPath)
    throws ServiceFailureException {
    final FileTransferProtocol protocol = connection.getProtocol();
    final FileTransferProtocolStub stub = protocolHash.get(protocol);
    try {
      return stub.listContent(connection, remoteDirPath);
    }
    catch (final Exception e) {
      final String conn = connection.toString();
      final String msg = "Falha de listagem de contedo em "+ conn+"::"+remoteDirPath+"!";
      Server.logSevereMessage(msg , e);
      throw new ServiceFailureException(e.getMessage());
    }
  }

  /**
   * Colocao de request em fila.
   * 
   * @param request request.
   */
  final private void queueRequest(final FileTransferRequest request) {
    request.markQueued();
    requestQueue.putToWaitList(request);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeRequest(final List<String> idList) throws RemoteException {
    for (final String id : idList) {
      final FileTransferRequest req = requestQueue.getRequestFromId(id);
      if (req != null) {
        unqueueRequest(req);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() throws ServerException {
    final List<FileTransferRequest> allRequests = getAllRequests(null);
    for (final FileTransferRequest req : allRequests) {
      req.markCancelled();
      req.markDeleted();
    }
    monitorThread.interrupt();
  }

  /**
   * Lana uma requisio
   * 
   * @param request a requisio.
   * @throws Exception em caso de falha.
   */
  final synchronized void startRequest(final FileTransferRequest request)
    throws Exception {
    final String id = request.getId();
    final FileTransferConnection connection = request.getConnection();
    final FileTransferProtocol protocol = connection.getProtocol();
    final FileTransferProtocolStub stub = protocolHash.get(protocol);
    final FileTransferRequestType type = request.getType();
    final FileTransferJob job;
    switch (type) {
      case DOWNLOAD:
        job = stub.createDownloadJob(request);
        break;
      case UPLOAD:
        job = stub.createUploadJob(request);
        break;
      default:
        throw new Exception("Unsupported request type detected");
    }

    requestQueue.moveFromWaitToRunList(id);

    final Thread thread = new Thread(job);
    thread.setName(id);
    thread.start();
  }

  /**
   * Colocao de request fora da fila.
   * 
   * @param request request.
   */
  final private void unqueueRequest(final FileTransferRequest request) {
    final FileTransferRequestStatus status = request.getStatus();
    if (status == FileTransferRequestStatus.QUEUED) {
      return;
    }
    if (status == FileTransferRequestStatus.RUNNING) {
      return;
    }

    request.markDeleted();
    requestQueue.removeFromAllLists(request);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void upload(final FileTransferConnection connection,
    final String remoteFilePath, final String[] localFilePath, final long size)
    throws RemoteException {
    final FileTransferRequest request =
      new FileTransferRequest(FileTransferRequestType.UPLOAD, connection,
        remoteFilePath, localFilePath, size);

    try {
      queueRequest(request);
    }
    catch (final Exception e) {
      final String msg = e.getMessage();
      if (request.getStatus() != FileTransferRequestStatus.ERROR) {
        request.markError(msg);
      }
      throw new ServiceFailureException(msg);
    }
  }

  /**
   * Constri a instncia do servio.
   * 
   * @throws ServerException se houver falha.
   */
  public static void createService() throws ServerException {
    new FileTransferService();
  }

  /**
   * Retorna a instncia do servio.
   * 
   * @return o objeto (no servidor) <code>FileTransferService</code>
   */
  public static FileTransferService getInstance() {
    return (FileTransferService) getInstance(SERVICE_NAME);
  }

  /**
   * Construtor padro do servio.
   * 
   * @throws ServerException em caso de falha no servio.
   */
  protected FileTransferService() throws ServerException {
    super(SERVICE_NAME);

    protocolHash.put(FileTransferProtocol.FTP, new FTPStub());
    protocolHash.put(FileTransferProtocol.SFTP, new SFTPStub());
    Collections.unmodifiableMap(protocolHash);

    monitorThread = new FileTransferServiceMonitorThread(this);
    monitorThread.start();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FileTransferPredefinedConnection> getPredefinedConnections()
    throws RemoteException {
    final User user = getUser();
    final String login = user.getLogin();
    final List<FileTransferPredefinedConnection> list =
      new ArrayList<FileTransferPredefinedConnection>();
    for (FileTransferPredefinedConnection conn : predefinedConnections) {
      final FileTransferPredefinedConnection connection =
        buildPredefinedConnectionForUser(login, conn);
      list.add(connection);
    }
    return list;
  }

  /**
   * Monta uma nova conexo predefinida.
   * 
   * @param login login do usurio (para substituio)
   * @param connection conexo configurada no servidor.
   * @return a nova conexo predefinida (com ajustes)
   */
  private FileTransferPredefinedConnection buildPredefinedConnectionForUser(
    final String login, final FileTransferPredefinedConnection connection) {
    final String label = connection.getLabel();
    final FileTransferProtocol protocol = connection.getProtocol();
    final String serverName = connection.getServerName();
    
    final String uName = connection.getUserName();
    final String userName = uName.replaceAll("\\$LOGIN", login);

    final String hString = connection.getHomePath();
    final String homeString = hString.replaceAll("\\$LOGIN", login);

    final FileTransferPredefinedConnection conn =
      new FileTransferPredefinedConnection(label, protocol, serverName, userName, homeString);
    return conn;
  }

}
