/**
 * $Id: SSHExecutor.java 169762 2015-11-13 12:37:30Z fpina $
 */
package csbase.sga.ssh;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import csbase.sga.executor.JobData;
import csbase.sga.executor.JobExecutor;
import csbase.sga.executor.JobInfo;
import csbase.sga.executor.JobObserver;
import csbase.sshclient.CommandResult;
import csbase.sshclient.SSHClient;
import csbase.sshclient.SSHClientException;
import sgaidl.ActionNotSupportedException;
import sgaidl.InvalidActionException;
import sgaidl.JobControlAction;

/**
 * Implements the interface {@link JobExecutor} using SSH as the execution layer
 * to talk to the execution environment.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SSHExecutor implements JobExecutor {
  /** Thread pool to notify the observers */
  private Executor notificationExecutor = Executors.newCachedThreadPool();
  /** Executor that periodically runs the task to search for finished jobs */
  private ScheduledExecutorService searchFinishedJobsExecutor =
    Executors.newScheduledThreadPool(1);
  /** The wating interval of the task that search for finished jobs */
  private long searchFinisedJobInterval = 10 * 1000;
  /** SGA properties */
  private Properties pluginProperties;
  /** Execution environment driver */
  private SGADriver driver;
  private SSHClientFactory sshClientFactory;
  /** Jobs storage */
  private JobStorage jobStorage;
  /** Logger */
  private Logger logger;

  /**
   * Initialize the executor with a properties set and a {@link SGADriver} tied
   * to an execution environment.
   *
   * @param pluginProperties the properties
   * @param driver the execution environment driver
   */
  public SSHExecutor(Properties pluginProperties, SGADriver driver,
    SSHClientFactory sshClientFactory) {
    this.pluginProperties = pluginProperties;

    String sgaName = pluginProperties.getProperty(sgaidl.SGA_NAME.value);
    logger = Logger.getLogger(SGASSH.class.getName() + "." + sgaName);

    if (pluginProperties.containsKey(SGASSH.SGA_COMPLETED_TIME_KEY)) {
      searchFinisedJobInterval =
        Long.parseLong(
          pluginProperties.getProperty(SGASSH.SGA_COMPLETED_TIME_KEY)) * 1000;
    }

    long jobInfoMaxAge = 60 * 1000;
    if (pluginProperties.containsKey(SGASSH.SGA_PROCESS_TIME_KEY)) {
      jobInfoMaxAge =
        Long.parseLong(
          pluginProperties.getProperty(SGASSH.SGA_PROCESS_TIME_KEY)) * 1000;
    }

    this.jobStorage = new JobStorage(jobInfoMaxAge);
    this.driver = driver;
    this.sshClientFactory = sshClientFactory;
    searchFinishedJobsExecutor.scheduleWithFixedDelay(
      new SearchFinishedJobsTask(), searchFinisedJobInterval,
      searchFinisedJobInterval, TimeUnit.MILLISECONDS);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobData executeJob(final String jobCommand,
    Map<String, String> extraParams, JobObserver observer) {
    JobData data = null;
    SSHClient client = null;
    try {
      client = sshClientFactory.getSSHClient();
      String command = driver.buildSubmitJobCommand(jobCommand, extraParams);
      logger.fine("Job's command line: " + command);
      CommandResult result = client.execute(command);
      if (result.getStatus() > 0) {
        logger.log(
          Level.WARNING, "Job execution return code: " + result.getStatus()
          + "\nOutput: " + result.getOutput() + "\nError: "
          + result.getError());
        return null;
      }
      logger.fine(
        "Job execution return code: " + result.getStatus() + "\nOutput: "
          + result.getOutput() + "\nError: " + result.getError());
      data = driver.parseJobSubmissionOutput(result.getOutput());
      logger.fine("Job's JobData: " + data);
    }
    catch (IOException | SSHClientException e) {
      logger.log(Level.SEVERE, "Erro submitting job", e);
    }
    finally {
      if (client != null) {
        client.disconnect();
      }
    }

    if (data == null) {
      logger.log(Level.SEVERE, "Cannot parse job submission output");
    }
    else {
      jobStorage.addJob(data, observer);
    }
    return data;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean recoveryJob(JobData data, JobObserver observer) {
    try {
      updateJobsInfo();
    }
    catch (SSHClientException | IOException e) {
      logger.log(Level.WARNING, "Erro recovering job", e);
    }

    return jobStorage.addObserver(data, observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void controlJob(JobData data, String child, JobControlAction action)
    throws InvalidActionException, ActionNotSupportedException {
    SSHClient client = null;
    try {
      client = sshClientFactory.getSSHClient();
      String controlCommand = driver.buildKillJobCommand(data);
      logger.fine("Job's control command line: " + controlCommand);
      CommandResult result = client.execute(controlCommand);
      if (result.getStatus() > 0) {
        logger.log(
          Level.WARNING, "Job control return code: " + result.getStatus()
          + "\nOutput: " + result.getOutput() + "\nError: "
          + result.getError());
        return;
      }
      logger.fine(
        "Job control return code: " + result.getStatus() + "\nOutput: "
          + result.getOutput() + "\nError: " + result.getError());
    }
    catch (IOException | SSHClientException e) {
      logger.log(Level.SEVERE, "Erro controlling job", e);
    }
    finally {
      if (client != null) {
        client.disconnect();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobInfo getJobInfo(JobData data) {
    try {
      updateJobsInfo();
    }
    catch (SSHClientException | IOException e) {
      logger.log(Level.SEVERE, "Erro updating job info", e);
    }

    return jobStorage.getJobInfo(data);
  }

  //  /**
  //   * Gets a SSH client.
  //   *
  //   * @return the SSH client
  //   *
  //   * @throws SSHClientException failure getting the SSH client
  //   */
  //  private SSHClient getSSHClient() throws SSHClientException {
  //    SSHClient sshClient = new SSHClient(sshHost, sshPort);
  //    logger.fine("SSH client to " + sshHost + ":" + sshPort + " created");
  //
  //    if (this.isTunnelEnable) {
  //      sshClient.createTunnel(
  //        sshTunnelHost, sshTunnelPort, sshTunnelUser, sshTunnelPrivKey,
  //        sshTunnelLocalPort, sshTunnelLocalPortRange);
  //      logger.fine(
  //        "SSH tunnel to " + sshTunnelHost + ":" + sshTunnelPort + " created");
  //    }
  //
  //    sshClient.connect(sshUserName, sshUserPrivKey);
  //    logger.fine(
  //      "SSH client authenticated [user: " + sshUserName + ", key: "
  //        + sshUserPrivKey);
  //
  //    return sshClient;
  //  }

  /**
   * Updates de jobs information. If the {@link SGADriver} in use implements
   * {@link SGADriver#buildCheckAllJobsCommand()} all jobs are updated in one
   * command. Otherwise, the update is done one by one.
   *
   * @throws SSHClientException failure in the SSH client
   * @throws IOException failure in the SSH client
   */
  private void updateJobsInfo() throws SSHClientException, IOException {
    if (!jobStorage.needsUpdate()) {
      return;
    }

    SSHClient client = null;
    try {
      client = sshClientFactory.getSSHClient();
      if (driver.buildCheckAllJobsCommand() != null) {
        String updateCommand = driver.buildCheckAllJobsCommand();
        logger.fine("Job's update command line: " + updateCommand);
        CommandResult result = client.execute(updateCommand);
        if (result.getStatus() > 0) {
          logger.log(
            Level.WARNING, "Update all jobs info return code: "
              + result.getStatus() + "\nOutput: " + result.getOutput()
              + "\nError: " + result.getError());
          return;
        }
        logger.fine(
          "Update all jobs info return code: " + result.getStatus()
          + "\nOutput: " + result.getOutput() + "\nError: "
          + result.getError());
        jobStorage.updateJobs(driver.parseCheckJobOutput(result.getOutput()));
      }
      else {
        Map<JobData, JobInfo> newJobsInfo = new HashMap<>();
        for (JobData d : jobStorage.getJobs()) {
          CommandResult result = client.execute(driver.buildCheckJobCommand(d));
          if (result.getStatus() > 0) {
            logger.log(
              Level.WARNING, "Update job info return code: "
                + result.getStatus() + "\nOutput: " + result.getOutput()
                + "\nError: " + result.getError());
            return;
          }
          logger.fine(
            "Update job info return code: " + result.getStatus() + "\nOutput: "
              + result.getOutput() + "\nError: " + result.getError());
          newJobsInfo.putAll(driver.parseCheckJobOutput(result.getOutput()));
        }
        jobStorage.updateJobs(newJobsInfo);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    finally {
      if (client != null) {
        client.disconnect();
      }
    }
  }

  /**
   * Task to search for finished jobs and to notify the corresponding observer.
   */
  private class SearchFinishedJobsTask implements Runnable {
    @Override
    public void run() {
      try {
        updateJobsInfo();
        for (Map.Entry<JobObserver, JobInfo> entry : jobStorage.getFinishedJobs().entrySet()) {
          final JobObserver observer = entry.getKey();
          final JobInfo info = entry.getValue();
          notificationExecutor.execute(new Runnable() {
            @Override
            public void run() {
              try {
                observer.onJobCompleted(info);
              }
              catch (Exception e) {
                observer.onJobLost();
              }
            }
          });
        }
      }
      catch (Exception e) {
        logger.log(Level.SEVERE, "Erro while searching for finished jobs", e);
      }
    }
  }
}
