/**
 * $Id: PBSDriver.java 169762 2015-11-13 12:37:30Z fpina $
 */

package csbase.sga.ssh.pbs;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.sga.executor.JobData;
import csbase.sga.executor.JobInfo;
import csbase.sga.monitor.SGAInfo;
import csbase.sga.ssh.SGADriver;
import sgaidl.COMMAND_EXEC_HOST;
import sgaidl.COMMAND_PID;
import sgaidl.COMMAND_STATE;
import sgaidl.ProcessState;

/**
 * @author Tecgraf
 */
public class PBSDriver implements SGADriver {
  private static final String SUBMIT_JOB = "echo \"{0}\" | qsub";
  private static final String CHECK_JOB = "qstat -f {0}";
  private static final String CHECK_ALL_JOBS = "qstat -f";
  private static final String KILL_JOB = "qdel {0}";

  private static final String JOB_INFO_SEPARATOR = "Job Id:";

  private Pattern JOB_ID_PATTERN = Pattern.compile("([^\\.]+)\\.(.*)\n");
  private Pattern JOB_INFO_PATTERN =
    Pattern.compile(
      "Job Id\\: (\\d+)\\.(\\S+)$(.*)", Pattern.DOTALL + Pattern.MULTILINE);
  private Pattern JOB_ATTRIBUTES_PATTERN =
    Pattern.compile("\\s*(\\S+) = ([\\S ]+)");

  private Properties properties;

  @Override
  public void init(Properties properties) {
    this.properties = properties;
  }

  @Override
  public String buildSubmitJobCommand(String script,
    Map<String, String> extraParam) {
    return MessageFormat.format(SUBMIT_JOB, script);
  }

  @Override
  public String buildKillJobCommand(JobData jobData) {
    PBSJobData data = (PBSJobData) jobData;
    return MessageFormat.format(KILL_JOB, data.getJobId());
  }

  @Override
  public String buildCheckJobCommand(JobData jobData) {
    PBSJobData data = (PBSJobData) jobData;
    return MessageFormat.format(CHECK_JOB, data.getJobId());
  }

  @Override
  public String buildCheckAllJobsCommand() {
    return CHECK_ALL_JOBS;
  }

  @Override
  public JobData parseJobSubmissionOutput(String output) {
    Matcher matcher = JOB_ID_PATTERN.matcher(output);
    if (matcher.matches()) {
      return new PBSJobData(matcher.group(1));
    }
    else {
      return null;
    }
  }

  @Override
  public Map<JobData, JobInfo> parseCheckJobOutput(String output) {
    Map<JobData, JobInfo> jobsInfo = new HashMap<>();

    int beginIdx = output.indexOf(JOB_INFO_SEPARATOR);
    if (beginIdx < 0) {
      return jobsInfo;
    }

    do {
      int endIdx = output.indexOf(JOB_INFO_SEPARATOR, beginIdx + 1);

      String entry =
        output.substring(beginIdx, endIdx < 0 ? output.length() : endIdx);
      Matcher matcher = JOB_INFO_PATTERN.matcher(entry);
      Map<String, String> attrsMap = new HashMap<>();
      while (matcher.find()) {
        attrsMap.put("pid", matcher.group(1));
        attrsMap.put("server", matcher.group(2));
        String attrs = matcher.group(3);
        Matcher attrsMatcher = JOB_ATTRIBUTES_PATTERN.matcher(attrs);
        while (attrsMatcher.find()) {
          attrsMap.put(attrsMatcher.group(1), attrsMatcher.group(2));
        }
      }

      beginIdx = endIdx;
      jobsInfo.put(
        new PBSJobData(attrsMap.get("pid")), convertToJobInfo(attrsMap));
    } while (beginIdx > 0);

    return jobsInfo;
  }

  private JobInfo convertToJobInfo(Map<String, String> jobMap) {
    JobInfo jobInfo = new JobInfo();

    jobInfo.jobParam.put(COMMAND_PID.value, jobMap.get("pid"));
    ProcessState jobState;
    switch (jobMap.get("job_state")) {

      case "C": // Job is completed after having run
        jobState = ProcessState.FINISHED;
        break;
      case "E": // Job is exiting after having run
        jobState = ProcessState.FINISHED;
        break;
      case "H": // Job is held
        jobState = ProcessState.SLEEPING;
        break;
      case "Q": // job is queued, eligible to run or routed
        jobState = ProcessState.WAITING;
        break;
      case "R": // job is running
        jobState = ProcessState.RUNNING;
        break;
      case "T": // job is being moved to new location
        jobState = ProcessState.RUNNING;
        break;
      case "W": // job is waiting for its execution time (-a option) to be reached.
        jobState = ProcessState.SLEEPING;
        break;
      case "S": // (Unicos only) job is suspend
        jobState = ProcessState.FINISHED;
        break;
      default:
        jobState = ProcessState.WAITING;
        break;
    }
    jobInfo.jobParam.put(COMMAND_STATE.value, jobState.toString());
    String host = jobMap.get("server");
    jobInfo.jobParam.put(COMMAND_EXEC_HOST.value, (host == null) ? "" : host);

    return jobInfo;
  }

  @Override
  public String buildCheckEnvironmentCommand() {
    // TODO Implementar
    return "";
  }

  @Override
  public SGAInfo parseCheckEnvironmentOutput(String output) {
    // TODO Implementar
    return new SGAInfo(properties);
  }

  public void parseNodes(String output) {
    Pattern NODES_PATTERN = Pattern.compile("^(\\S+)$", Pattern.MULTILINE);

    Matcher matcher = NODES_PATTERN.matcher(output);

    while (matcher.find()) {
      System.out.println(matcher.group(1));
    }
  }
}
