package csbase.sga.ssh;

import csbase.sshclient.SSHClient;
import csbase.sshclient.SSHClientException;

import java.util.*;

/**
 * A simple SSHClient pool that disconnect clients active longer than the
 * maximum lease.
 *
 * Call the method {@link SSHClientPool#retrieveSSHClient()} to retrieve a
 * SSHClient and, after done with it, call the method
 * {@link #returnSSHClient(SSHClient)} to put the SSHClient back into the pool.
 */
public class SSHClientPool {
  static private long MAX_LEASE_TIME = 60 * 1000;

  //TODO Definir um tamanho mximo de conexes do pool
  //Parece que se o servidor ficar sem acesso ao SGA o processo de verificao do ambiente ficar
  //consumindo os recursos da mquina
  private List<SSHClient> activeClients = new Vector<>();
  private List<SSHClient> availableClients = new Vector<>();
  private Map<SSHClient, Long> clientReleaseTime = new Hashtable<>();

  private String sshHost;
  private int sshPort;
  private String sshUserName;
  private String sshUserPass;
  private String sshUserPrivKey;
  private boolean isTunnelEnable;
  private String sshTunnelHost;
  private int sshTunnelPort;
  private String sshTunnelUser;
  private String sshTunnelPass;
  private String sshTunnelPrivKey;
  private int sshTunnelLocalPort;
  private int sshTunnelLocalPortRange;

  /**
   * Constructor.
   *
   * @param sshHost host
   * @param sshPort port
   * @param sshUserName username
   * @param sshUserPass password (optional if sshUserPrivKey is provided)
   * @param sshUserPrivKey private key (optional if sshUserPass is provided)
   * @param isTunnelEnable to indicate that a tunnel must be created
   * @param sshTunnelHost host of the tunnel
   * @param sshTunnelPort port of the tunnel
   * @param sshTunnelUser username of the tunnel
   * @param sshTunnelPass password of the tunnel (optional if sshTunnelPrivKey is provided)
   * @param sshTunnelPrivKey private key of the tunnel (optional if sshTunnelPass is provided)
   * @param sshTunnelLocalPort local port of the tunnel
   * @param sshTunnelLocalPortRange local port range of the tunnel
   */
  public SSHClientPool(String sshHost, int sshPort, String sshUserName,
                       String sshUserPass, String sshUserPrivKey, boolean isTunnelEnable,
                       String sshTunnelHost, int sshTunnelPort, String sshTunnelUser,
                       String sshTunnelPass, String sshTunnelPrivKey, int sshTunnelLocalPort,
                       int sshTunnelLocalPortRange) {
    super();
    this.sshHost = sshHost;
    this.sshPort = sshPort;
    this.sshUserName = sshUserName;
    this.sshUserPass = sshUserPass;
    this.sshUserPrivKey = sshUserPrivKey;
    this.isTunnelEnable = isTunnelEnable;
    this.sshTunnelHost = sshTunnelHost;
    this.sshTunnelPort = sshTunnelPort;
    this.sshTunnelUser = sshTunnelUser;
    this.sshTunnelPass = sshTunnelPass;
    this.sshTunnelPrivKey = sshTunnelPrivKey;
    this.sshTunnelLocalPort = sshTunnelLocalPort;
    this.sshTunnelLocalPortRange = sshTunnelLocalPortRange;
  }

  /**
   * Retrieves a SSHClient from the pool. If no one is available a new one is
   * created.
   *
   * @return a SSHClient
   *
   * @throws SSHClientException if there is a error creating a new SSHClient
   */
  protected SSHClient retrieveSSHClient() throws SSHClientException {
    synchronized (this) {
      if (!availableClients.isEmpty()) {
        SSHClient client;
        client = availableClients.remove(availableClients.size() - 1);
        activeClients.add(client);

        if(client.isConnected()) {
          return client;
        }
      }
    }

    SSHClient client;
    client = new SSHClient(sshHost, sshPort);
    if (this.isTunnelEnable) {
      client.createTunnel(
          sshTunnelHost, sshTunnelPort, sshTunnelUser, sshTunnelPrivKey,
          sshTunnelLocalPort, sshTunnelLocalPortRange);
    }

    client.connect(sshUserName, sshUserPrivKey);

    synchronized (this) {
      activeClients.add(client);
    }

    return client;
  }

  /**
   * Returns a SSHClient to the pool.
   *
   * @param sshClient the SSHClient returned
   */
  protected void returnSSHClient(SSHClient sshClient) {
    synchronized (this) {
      activeClients.remove(sshClient);
      availableClients.add(sshClient);
      clientReleaseTime.put(sshClient, System.currentTimeMillis());

      // Disconnect unused client within the lease time
      SSHClient leastUsedClient = availableClients.get(0);
      if (System.currentTimeMillis() - clientReleaseTime.get(leastUsedClient) > MAX_LEASE_TIME) {
        availableClients.remove(leastUsedClient);
        clientReleaseTime.remove(leastUsedClient);
        leastUsedClient.disconnect();
      }
    }
  }
}
