package csbase.sga.transfer.ssh;

import csbase.server.plugin.service.IServiceManager;
import csbase.server.plugin.service.algorithmservice.IAlgorithmService;
import csbase.server.plugin.service.projectservice.IProjectService;
import csbase.server.plugin.service.sgaservice.ISGADataTransfer;
import csbase.server.plugin.service.sgaservice.SGADataTransferException;
import csbase.sshclient.SSHClient;
import csbase.sshclient.SSHClientException;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.MessageFormat;
import java.util.*;
import java.util.regex.Pattern;

public class SGASSHTransferMechanism implements ISGADataTransfer {
  /**
   * Chave do diretrio raiz do mecanismo de transferncia
   */
  final static private String ROOT_KEY = sgaidl.SGA_CSFS_ROOT_DIR.value;
  /**
   * Chave do servidor
   */
  final static private String HOST_KEY = "ssh_host";
  /**
   * Chave da porta
   */
  final static private String PORT_KEY = "ssh_port";
  /**
   * Chave do caminho da chave privada
   */
  final static private String PRIVATE_KEY_PATH_KEY = "ssh_private_key_path";
  /**
   * Chave do usurio
   */
  final static private String USER_NAME_KEY = "ssh_user_name";
  /**
   * Senha do usurio
   */
  final static private String USER_PASS_KEY = "ssh_user_pass";

  /**
   * Chave da flag que indica o uso de servidor de tunelamento
   */
  final static private String TUNNEL_ENABLE_KEY = "ssh_tunnel_enable";
  /**
   * Chave do servidor de tunelamento
   */
  final static private String TUNNEL_HOST_KEY = "ssh_tunnel_host";
  /**
   * Chave da porta do servidor de tunelamento
   */
  final static private String TUNNEL_PORT_KEY = "ssh_tunnel_port";
  /**
   * Chave do caminho da chave privada do servidor de tunelamento
   */
  final static private String TUNNEL_PRIVATE_KEY_PATH_KEY =
    "ssh_tunnel_private_key_path";
  /**
   * Chave do usurio do servidor de tunelamento
   */
  final static private String TUNNEL_USER_NAME_KEY = "ssh_tunnel_user_name";
  /**
   * Chave da porta local do tnel
   */
  final static private String TUNNNEL_LOCAL_PORT_KEY = "ssh_tunnel_local_port";
  /**
   * Chave do intervalo de porta local do tnel
   */
  final static private String TUNNNEL_LOCAL_PORT_RANGE_KEY =
    "ssh_tunnel_local_port_range";

  /**
   * Referncia para o gernte de servios
   */
  private IServiceManager serviceManager;

  /**
   * Propriedades definidas pelo SGA
   */
  private Properties properties;

  /**
   * Diretrio raiz do mecanismo de transferncia
   */
  private String root;

  /**
   * Cliente SSH
   */
  private SSHClient sshClient;

  /**
   * Servidor
   */
  private String host;
  /**
   * Porta
   */
  private int port;
  /**
   * Caminho para a chave privada
   */
  private String privateKey;
  /**
   * Usurio
   */
  private String userName;
  /**
   * Senha
   */
  private String password;

  /**
   * Flag que indica se o tunelamento est ativo
   */
  private boolean isTunnelEnable = false;
  /**
   * Servidor de tunelamento
   */
  private String tunnelHost;
  /**
   * Porta do servidor de tunelamento
   */
  private int tunnelPort;
  /**
   * Caminho para a chave privada do servidor de tunelamento
   */
  private String tunnelPrivateKeyFilePath;
  /**
   * Usurio do servidor de tunelamento
   */
  private String tunnelUserName;
  /**
   * Porta local do tnel
   */
  private int tunnelLocalPort;
  /**
   * Intevala de porta local do tnel
   */
  private int tunnelLocalPortRange;

  /**
   * Construtor
   *
   * @param serviceManager referncia para o gerenciador de servios
   */
  public SGASSHTransferMechanism(IServiceManager serviceManager) {
    this.serviceManager = serviceManager;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws Exception {
    sshClient.disconnect();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void copyFrom(String[] remotePath, String[] localPath)
    throws SGADataTransferException {
    try {
      connect();
      sshClient.download(join(localPath), joinRemotePath(remotePath));
    }
    catch (Exception e) {
      String msg =
        MessageFormat.format(
          "Erro while coping flie from remote {0} [{1}:{2}] to local {3}",
          joinRemotePath(remotePath), join(localPath), host, port);
      throw new SGADataTransferException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void copyTo(String[] localPath, String[] remotePath)
    throws SGADataTransferException {
    try {
      connect();
      createDirectory(Arrays.copyOf(remotePath, remotePath.length - 1));
      sshClient.upload(join(localPath), joinRemotePath(remotePath));
    }
    catch (Exception e) {
      String msg =
        MessageFormat.format(
          "Erro while coping from local {0} to remote {1} [{2}:{3}]", join(
            localPath), joinRemotePath(remotePath), host, port);
      throw new SGADataTransferException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean checkExistence(String[] remotePath)
    throws SGADataTransferException {
    try {
      connect();
      return sshClient.stat(joinRemotePath(remotePath));
    }
    catch (Exception e) {
      String msg =
        MessageFormat.format(
          "Erro while checking existence of remote file {0} [{1}:{2}]", join(
            root, join(remotePath)), host, port);
      throw new SGADataTransferException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createDirectory(String[] remotePath)
    throws SGADataTransferException {
    try {
      connect();
      //sshClient.createDirectory(joinRemotePath(remotePath));
      // WA para o servidor que no permite criar o caminho completo em uma nica chamada.
      String partialPath = "";
      for(String p : remotePath) {
        if(p.length() > 0) {
          partialPath = join(partialPath, p);
          sshClient.createDirectory(joinRemotePath(partialPath));
        }
      }
    }
    catch (Exception e) {
      String msg =
        MessageFormat.format(
          "Erro while creating remote directory {0} [{1}:{2}]", joinRemotePath(
            remotePath), host, port);
      throw new SGADataTransferException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void remove(String[] remotePath) throws SGADataTransferException {
    try {
      connect();
      sshClient.remove(joinRemotePath(remotePath));
    }
    catch (Exception e) {
      String msg =
        MessageFormat.format(
          "Erro while removing remote directory/file {0} [{1}:{2}]",
          joinRemotePath(remotePath), host, port);
      throw new SGADataTransferException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getAlgorithmsRootPath() throws SGADataTransferException {
    IAlgorithmService algorithmService =
      (IAlgorithmService) serviceManager.getService("AlgorithmService");
    return new String[] { algorithmService.getAlgorithmRepositoryPath() };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getProjectsRootPath() throws SGADataTransferException {
    IProjectService projectService =
      (IProjectService) serviceManager.getService("ProjectService");
    return new String[] { projectService.getProjectRepositoryPath() };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String[], Long> getLocalTimestamps(String[] localPath)
    throws SGADataTransferException {
    final Map<String[], Long> timestampsMap = new HashMap<>();
    Path sandbox = Paths.get(join(localPath));

    try {
      Files.walkFileTree(sandbox, new SimpleFileVisitor<Path>() {
        @Override
        public FileVisitResult visitFile(Path path, BasicFileAttributes attrs)
          throws IOException {
          File file = path.toFile();
          if (!file.isDirectory()) {
            timestampsMap.put(
              splitPath(file.getAbsolutePath()), file.lastModified());
          }

          return FileVisitResult.CONTINUE;
        }
      });
    }
    catch (IOException e) {
      String msg =
        MessageFormat.format(
          "Erro while retriving local timestamp {0} [{1}:{2}]", join(localPath),
          host, port);
      throw new SGADataTransferException(msg, e);
    }

    return timestampsMap;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String[], Long> getRemoteTimestamps(String[] remotePath)
    throws SGADataTransferException {
    HashMap<String[], Long> timestampsMap = new HashMap<>();

    try {
      connect();
      Map<String, Long> fileMap =
        sshClient.listFiles(joinRemotePath(remotePath));
      for (String filePath : fileMap.keySet()) {
        Path relPath = Paths.get(root).relativize(Paths.get(filePath));

        timestampsMap.put(splitPath(relPath.toString()), fileMap.get(filePath));
      }
    }
    catch (Exception e) {
      String msg =
        MessageFormat.format(
          "Erro while retriving remote timestamp {0} [{1}:{2}]", joinRemotePath(
            remotePath), host, port);
      throw new SGADataTransferException(msg, e);
    }

    return timestampsMap;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setSGAProperties(Properties properties) {
    //TODO Validar se as propriedades esto preenchidas corretamente
    this.properties = (Properties) properties.clone();
    this.root = this.properties.getProperty(ROOT_KEY);
    this.host = this.properties.getProperty(HOST_KEY);
    this.port = Integer.parseInt(this.properties.getProperty(PORT_KEY));
    this.userName = this.properties.getProperty(USER_NAME_KEY);
    this.privateKey = this.properties.getProperty(PRIVATE_KEY_PATH_KEY);
    this.password = this.properties.getProperty(USER_PASS_KEY);

    String value = this.properties.getProperty(TUNNEL_ENABLE_KEY);
    if (value != null && (value.equalsIgnoreCase("true")
      || value.equalsIgnoreCase("false"))) {
      this.isTunnelEnable = Boolean.valueOf(value);
    }

    if (isTunnelEnable) {
      this.tunnelLocalPort =
        Integer.parseInt(this.properties.getProperty(TUNNNEL_LOCAL_PORT_KEY));
      this.tunnelLocalPortRange =
        Integer.parseInt(
          this.properties.getProperty(TUNNNEL_LOCAL_PORT_RANGE_KEY));
      this.tunnelHost = this.properties.getProperty(TUNNEL_HOST_KEY);
      this.tunnelPort =
        Integer.parseInt(this.properties.getProperty(TUNNEL_PORT_KEY));
      this.tunnelUserName = this.properties.getProperty(TUNNEL_USER_NAME_KEY);
      this.tunnelPrivateKeyFilePath =
        this.properties.getProperty(TUNNEL_PRIVATE_KEY_PATH_KEY);
    }
  }

  private void connect() throws SSHClientException {
    if (sshClient == null) {
      sshClient = new SSHClient(this.host, this.port);
      if (isTunnelEnable) {
        sshClient.createTunnel(
            this.tunnelHost, this.tunnelPort, this.tunnelUserName,
            this.tunnelPrivateKeyFilePath, this.tunnelLocalPort,
            this.tunnelLocalPortRange);
      }
      if (this.privateKey != null) {
        sshClient.connect(this.userName, this.privateKey);
      } else if (this.password != null) {
        sshClient.connectByPassword(this.userName, this.password);
      }
    }
  }

  /**
   * Constri uma string concatenando o array de strings usando o separador
   * informado.
   *
   * @param str o array de strings a serem concatenadas
   *
   * @return a string resultante da concatenao
   */
  private String join(String... str) {
    String retval = "";
    for (String s : str) {
      retval += File.separator + s;
    }
    return retval.replaceFirst(File.separator, "");
  }

  /**
   * Monta o path remoto usando como prefixo o valor da {@link #root raiz}.
   *
   * @param remotePath caminho remoto
   *
   * @return caminho remote prefixado com a {@link #root raiz}
   */
  private String joinRemotePath(String... remotePath) {
    return join(root, join(remotePath));
  }

  /**
   * Separa um caminho em um array de strings usando o separador da plataforma.
   *
   * @param path o caminho
   *
   * @return array de strings que representa o caminho
   */
  private static String[] splitPath(String path) {
    Scanner scanner = new Scanner(path);
    scanner.useDelimiter(Pattern.quote(File.separator));
    List<String> pathAsList = new ArrayList<>();
    while (scanner.hasNext()) {
      String dir = scanner.next();
      if (!dir.isEmpty()) {
        pathAsList.add(dir);
      }
    }
    scanner.close();

    return pathAsList.toArray(new String[pathAsList.size()]);
  }
}
