/*
 * $Id: BuildVariants.java 117759 2011-05-05 12:54:01Z clinio $
 */
package csbase.tools;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;

import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

import tecgraf.javautils.core.lng.LNG;

/**
 * Utilitrio para gerao de cones a partir da composio de imagens de fundo,
 * de primeiro plano e, opcionalmente, de decorao.
 * 
 * @author Tecgraf
 */
public class IconComposer {

  /**
   * Opes de linha de comando.
   */
  public static class ImageOptions {

    /**
     * Imagem de fundo
     */
    @Option(name = "--background", aliases = { "-b" }, usage = "Set background image file")
    public File backgroundImageFile;

    /**
     * Imagem de primeiro plano.
     */
    @Option(name = "--foreground", aliases = { "-f" }, usage = "Set foreground image file")
    public File foregroundImageFile;

    /**
     * Imagem de decorao.
     */
    @Option(name = "--decoration", aliases = { "-d" }, usage = "Set decoration image file")
    public File decorationImageFile;

    /**
     * Imagem gerada a partir das entradas.
     */
    @Option(name = "--output", aliases = { "-o" }, usage = "Set output image file")
    public File outputImageFile = new File("output.png");

    /**
     * Formato da imagem a ser gerada.
     */
    @Option(name = "--type", aliases = { "-t" }, usage = "Set output image type (jpg, png, gif, bmp)")
    public String imageType = "png";

    /**
     * Formato da imagem a ser gerada.
     */
    @Option(name = "--height", aliases = { "-h" }, usage = "Set output image height (in number of pixels)")
    public int imageHeight = 16;

    /**
     * Formato da imagem a ser gerada.
     */
    @Option(name = "--width", aliases = { "-w" }, usage = "Set output image width (in number of pixels)")
    public int imageWidth = 16;

    /**
     * Deslocamento da imagem de primeiro plano relativo ao fundo.
     */
    @Option(name = "--fOffset", aliases = { "-fo" }, usage = "Set foreground image offset relative to background (in number of pixels)")
    public int foregroundOffset = 5;

    /**
     * Deslocamento da imagem de decorao relativo ao fundo.
     */
    @Option(name = "--dOffset", aliases = { "-do" }, usage = "Set decoration image offset relative to background (in number of pixels)")
    public int decorationOffset = 8;

    /**
     * Indica se a imagem gerada deve ser mostrada ao final da execuo.
     */
    @Option(name = "--showOutput", aliases = { "-s" }, usage = "Show output image")
    public boolean showOutput = true;

    /**
     * Apresenta os possveis argumentos da linha de comando (ajuda).
     */
    @Option(name = "--help", usage = "Show help")
    boolean showHelp = false;

  }

  /**
   * Funo principal.
   * 
   * @param args argumentos de linha de comando.
   * @throws IOException em caso de erro ao ler/escrever as imagens.
   */
  public static void main(String[] args) throws IOException {
    ImageOptions options = new ImageOptions();
    CmdLineParser parser = new CmdLineParser(options);
    try {
      parser.parseArgument(args);
    }
    catch (CmdLineException e) {
      System.err.println(e.getMessage());
      System.err.println(LNG.get("IconComposer.cmdline"));
      parser.printUsage(System.err);
      System.exit(1);
    }

    if (options.showHelp) {
      System.out.println(LNG.get("IconComposer.cmdline"));
      parser.printUsage(System.out);
      System.exit(0);
    }

    IconComposer builder = new IconComposer();
    builder.mergeImages(options);
    if (options.showOutput) {
      showImage(options.outputImageFile.getAbsolutePath());
    }
  }

  /**
   * Mostra a imagem.
   * 
   * @param path caminho para a imagem.
   */
  public static void showImage(String path) {
    JFrame frame = new JFrame(path);
    frame.setPreferredSize(new Dimension(100, 100));
    frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    JLabel label = new JLabel(new ImageIcon(path));
    frame.add(label, BorderLayout.CENTER);
    frame.pack();
    frame.setVisible(true);
  }

  /**
   * Combina as imagens de fundo, de frente e decorao (caso existam).
   * 
   * @param options as opes para composio da imagem.
   * @throws IOException em caso de erro ao ler/escrever as imagens.
   */
  public void mergeImages(ImageOptions options) throws IOException {
    BufferedImage backgroundImage = null;
    if (options.backgroundImageFile != null) {
      backgroundImage = ImageIO.read(options.backgroundImageFile);
    }
    BufferedImage foregroundImage = null;
    if (options.foregroundImageFile != null) {
      foregroundImage = ImageIO.read(options.foregroundImageFile);
    }
    BufferedImage decorationImage = null;
    if (options.decorationImageFile != null) {
      decorationImage = ImageIO.read(options.decorationImageFile);
    }
    BufferedImage composedImage =
      new BufferedImage(options.imageWidth, options.imageHeight,
        BufferedImage.TYPE_INT_ARGB);
    Graphics2D graphics = composedImage.createGraphics();
    if (backgroundImage != null) {
      graphics.drawImage(backgroundImage, null, 0, 0);
    }
    if (foregroundImage != null) {
      graphics.drawImage(foregroundImage, null, options.foregroundOffset,
        options.foregroundOffset);
    }
    if (decorationImage != null) {
      graphics.drawImage(decorationImage, null, options.decorationOffset,
        options.decorationOffset);
    }
    ImageIO.write(composedImage, options.imageType, options.outputImageFile);
    graphics.dispose();
  }
}
