/**
 * $Id: NoDuplicatesCollection.java 110729 2010-09-30 17:08:59Z clinio $
 */

package csbase.util.collections;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.NoSuchElementException;
import java.util.Set;

/**
 * Representa uma coleo que no contm elementos duplicados e no contm
 * elementos nulos (null).
 * <p>
 * A diferena desta coleo para um {@link Set}  que ao inserirmos elementos
 * que j faam parte da coleo, eles so colocados no final da coleo e ser
 * informado que houve mudana na coleo. No caso do Set, a coleo
 * permaneceria sem modificaes.
 * <p>
 * Todos os elementos devem ser clonveis, ou o mtodo {@link #clone()}
 * levantar uma {@link CloneNotSupportedException}.
 * 
 * @author Rodrigo Carneiro Henrique (rodrigoh)
 * @param <T> tipo do objeto a ser armazenado na coleo
 */
public final class NoDuplicatesCollection<T> implements Collection<T>,
  Cloneable {
  /** A lista que armazenar os elementos da coleo. */
  private LinkedList<T> linkedList;

  /**
   * Cria uma coleo vazia.
   */
  public NoDuplicatesCollection() {
    this.linkedList = new LinkedList<T>();
  }

  /**
   * Cria uma coleo contendo os elementos da coleo recebida. Todos os
   * elementos duplicados sero removidos.
   * 
   * @param c A coleo com os elementos a serem inseridos.
   * 
   * @throws IllegalArgumentException Caso a coleo passada como parmetro seja
   *         nula (null).
   */
  public NoDuplicatesCollection(Collection<T> c) {
    if (c == null) {
      throw new IllegalArgumentException(
        "A coleo recebida no pode ser nula.");
    }
    this.linkedList = new LinkedList<T>(c);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int size() {
    return this.linkedList.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void clear() {
    this.linkedList.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isEmpty() {
    return this.linkedList.isEmpty();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object[] toArray() {
    return this.linkedList.toArray();
  }

  /**
   * Retorna o primeiro elemento da coleo.
   * 
   * @return O primeiro elemento da coleo.
   * 
   * @throws NoSuchElementException Caso a coleo esteja vazia.
   */
  public Object getFirst() throws NoSuchElementException {
    return this.linkedList.getFirst();
  }

  /**
   * Retorna o ltimo elemento da coleo.
   * 
   * @return O ltimo elemento da coleo.
   * 
   * @throws NoSuchElementException Caso a coleo esteja vazia.
   */
  public Object getLast() throws NoSuchElementException {
    return this.linkedList.getLast();
  }

  /**
   * Insere o elemento na ltima posio da coleo.
   * 
   * <ul>
   * <li>
   * Caso o elemento no esteja na coleo, ele  inserido na ltima posio.</li>
   * <li>
   * Caso o elemento j esteja na coleo e esteja na ltima posio, nada 
   * feito.</li>
   * <li>
   * Caso o elemento j esteja na coleo e no esteja na ltima posio, ser
   * movido para a ltima posio.</li>
   * </ul>
   * 
   * 
   * @param o O objeto a ser inserido na coleo.
   * 
   * @return true, se a coleo foi alterada, ou false, caso contrrio.
   * 
   * @throws IllegalArgumentException Caso o elemento recebido seja nulo.
   * 
   * @see java.util.Collection#add(java.lang.Object)
   */
  @Override
  public boolean add(T o) {
    if (o == null) {
      throw new IllegalArgumentException(
        "Essa coleo no aceita elementos nulos.");
    }
    if (this.linkedList.contains(o)) {
      if (this.linkedList.indexOf(o) == (this.linkedList.size() - 1)) {
        return false;
      }
      this.linkedList.remove(o);
    }
    return this.linkedList.add(o);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean contains(Object o) {
    return this.linkedList.contains(o);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean remove(Object o) {
    return this.linkedList.remove(o);
  }

  /**
   * Insere todos os elementos da coleo recebida. Possui o mesmo funcionamento
   * do mtodo {@link #add(Object)}. Conforme os elementos vo aparecendo na
   * coleo de origem, eles vo sendo inseridos ao final da coleo destino (a
   * instncia atual). Elementos repetidos so movidos para o final.
   * 
   * @param c A coleo de origem.
   * 
   * @return true, se a coleo foi alterada, ou false, caso contrrio.
   * 
   * @throws IllegalArgumentException Caso a coleo passada como parmetro seja
   *         nula (null).
   * 
   * @see java.util.Collection#addAll(java.util.Collection)
   */
  @Override
  public boolean addAll(Collection<? extends T> c) {
    if (c == null) {
      throw new IllegalArgumentException(
        "A coleo recebida no pode ser nula.");
    }
    boolean wasChanged = false;
    for (T element : c) {
      if (add(element)) {
        wasChanged = true;
      }
    }
    return wasChanged;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean containsAll(Collection<?> c) {
    return this.linkedList.containsAll(c);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeAll(Collection<?> c) {
    return this.linkedList.removeAll(c);
  }

  /**
   * Mtodo no implementado.
   * 
   * @see java.util.Collection#retainAll(java.util.Collection)
   */
  @Override
  public boolean retainAll(Collection<?> c) {
    throw new UnsupportedOperationException();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Iterator<T> iterator() {
    return this.linkedList.iterator();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public NoDuplicatesCollection<T> clone() {
    return new NoDuplicatesCollection<T>(linkedList);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null || obj instanceof NoDuplicatesCollection<?> == false) {
      return false;
    }
    NoDuplicatesCollection<?> other = (NoDuplicatesCollection<?>) obj;
    if (this.size() != other.size()) {
      return false;
    }
    Iterator<T> iterator = this.iterator();
    Iterator<?> otherIterator = other.iterator();
    while (iterator.hasNext()) {
      if (!iterator.next().equals(otherIterator.next())) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.linkedList.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.linkedList.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T> T[] toArray(T[] a) {
    return this.linkedList.<T> toArray(a);
  }
}
