package csbase.util.messages;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * Representa um consumidor assncrono.
 * </p>
 * <p>
 *  utilizado como destino das mensagens do
 * {@link csbase.util.data.channel.DataChannel canal de mensagens} instanciado
 * pelo tpico.
 * </p>
 * 
 * @author Tecgraf
 */
class AsynchronousConsumer implements IMessageListener {
  /**
   * Identificador do consumidor de mensagens.
   */
  private final Serializable consumerId;
  /**
   * Interface utilizada para consumir as mensagens de forma assncrona.
   */
  private final IMessageListener listener;

  /**
   * {@link MessageStore} que armazena as mensagens como entradas.
   */
  private final MessageStore store;

  /**
   * Construtor.
   * 
   * @param consumerId Identificador do consumidor.
   * @param listener Interface utilizada para consumir as mensagens de forma
   *        assncrona.
   * @param store
   */
  public AsynchronousConsumer(Serializable consumerId,
    IMessageListener listener, MessageStore store) {
    super();

    if (null == consumerId) {
      throw new NullPointerException("consumerId == null");
    }

    this.consumerId = consumerId;
    this.listener = listener;
    this.store = store;
  }

  /**
   * Obtm o identificador do consumidor.
   * 
   * @return o identificador do consumidor.
   */
  public Serializable getId() {
    return consumerId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onMessagesReceived(Message... messages) throws Exception {

    List<MessageStore.Entry> entries2send = new ArrayList<MessageStore.Entry>();
    for (Message aMessage : messages) {
      MessageStore.Entry anEntry = store.getEntry(aMessage.getId());
      if (anEntry == null) {
        continue;
      }
      if (anEntry.setBeingSentTo(consumerId, true)) {
        entries2send.add(anEntry);
      }
    }

    if (entries2send.size() == 0) {
      return;
    }

    Message[] messages2send;
    if (entries2send.size() == messages.length) {
      messages2send = messages;
    }
    else {
      messages2send = new Message[entries2send.size()];
      for (int inx = 0; inx < entries2send.size(); inx++) {
        messages2send[inx] = entries2send.get(inx).getMessage();
      }
    }

    try {
      listener.onMessagesReceived(messages2send);
      for (MessageStore.Entry anEntry : entries2send) {
        anEntry.setAcknowledgedBy(consumerId);
      }
    }
    catch (Exception e) {
      for (MessageStore.Entry anEntry : entries2send) {
        anEntry.setBeingSentTo(consumerId, false);
      }
      throw e;
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @return o <i>hash code</i> do identificador do consumidor.
   */
  @Override
  public int hashCode() {
    return consumerId.hashCode();
  }

  /**
   * <p>
   * Compara se o outro objeto  um {@link AsynchronousConsumer} com o mesmo
   * identificador.
   * </p>
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    AsynchronousConsumer other = (AsynchronousConsumer) obj;
    if (consumerId == null) {
      if (other.consumerId != null) {
        return false;
      }
    }
    else if (!consumerId.equals(other.consumerId)) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return consumerId.toString();
  }
}
