/*
 * $Id$
 */

package csbase.util;

import java.io.File;
import java.io.FileFilter;
import java.security.AccessControlException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Encapsula mtodos utilitrios do sistema de arquivos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class FileSystemUtils {

  /**
   * Nome do diretrio de controle usado pelo sistema de controle de verso.
   * Ex.: CVS = "CVS", Subversion = ".svn".
   */
  public static final String VERSION_CONTROL_DIR = ".svn";

  /**
   * Indica se o sistema permite a leitura de arquivos.
   */
  private static final boolean READ_ALLOWED = _canRead();

  /**
   * Verifica se o sistema permite a leitura de arquivos. <br>
   * Obs.: Esse valor  cacheado.
   * 
   * @return <tt>true</tt> se o sistema permite a leitura de arquivos.
   */
  public static final boolean canRead() {
    return READ_ALLOWED;
  }

  /**
   * Verifica se o sistema permite a leitura de arquivos.
   * 
   * @return <tt>true</tt> se o sistema permite a leitura de arquivos.
   */
  private static boolean _canRead() {
    try {
      new File(".").canRead();
      return true;
    }
    catch (AccessControlException ex) {
      return false;
    }
  }

  /**
   * Verifica se um path aponta para um arquivo.
   * 
   * @param path o path
   * @return <code>true</code> se o path aponta para um arquivo
   */
  public static boolean fileExists(String path) {
    return fileExists(new File(path));
  }

  /**
   * Verifica se um arquivo existe e se  realmente um arquivo (i.e. no  um
   * diretrio).
   * 
   * @param file o arquivo.
   * @return <code>true</code> se o arquivo existe e se  realmente um arquivo
   */
  public static boolean fileExists(File file) {
    return file.exists() && file.isFile();
  }

  /**
   * Verifica se um path aponta para um diretrio.
   * 
   * @param path o path
   * @return <code>true</code> se o path aponta para um diretrio
   */
  public static boolean dirExists(String path) {
    return dirExists(new File(path));
  }

  /**
   * Verifica se um arquivo ({@link File}) aponta para um diretrio.
   * 
   * @param dir o diretrio
   * @return true se o <code>File</code> aponta para um diretrio
   */
  public static boolean dirExists(File dir) {
    return dir.exists() && dir.isDirectory();
  }

  /**
   * Obtm os subdiretrios de um determinado diretrio.
   * 
   * @param dir diretrio de referncia
   * @return array com os subdiretrios
   * @throws IllegalArgumentException se o diretrio de referncia no 
   *         realmente um diretrio
   */
  public static File[] getSubDirs(File dir) {
    if (!dir.exists()) {
      throw new IllegalArgumentException(dir.getAbsolutePath() + " no existe");
    }
    if (!dir.isDirectory()) {
      throw new IllegalArgumentException(dir.getAbsolutePath()
        + " no  um diretrio");
    }
    File[] subDirs = dir.listFiles(new FileFilter() {
      @Override
      public boolean accept(File path) {
        return path.isDirectory();
      }
    });
    return subDirs;
  }

  /**
   * Remoo recursiva de diretrios. Registra os erros em um logger utilizando
   * o nvel {@link Level#SEVERE}.
   * 
   * @param file arquivo ou diretrio a ser apagado
   * @param logger o logger (pode ser <code>null</code>)
   * @return <code>true</code> se o arquivo/diretrio foi removido com sucesso
   */
  public static boolean deleteRecursively(final File file, final Logger logger) {
    if (!file.exists()) {
      return true;
    }
    /*
     * se  um diretrio, removemos seus filhos
     */
    if (file.isDirectory()) {
      final File[] children = file.listFiles();
      for (File child : children) {
        boolean success = deleteRecursively(child, logger);
        if (!success) {
          /*
           * a mensagem de erro j foi exibida
           */
          return false;
        }
      }
    }

    /*
     * neste ponto ou temos um arquivo ou um diretrio vazio, podemos remover
     */

    // DEBUG
    //    System.out.println("removendo arquivo " + file.getAbsolutePath());
    //    final boolean removed = true;

    final boolean removed = file.delete();
    if (!removed) {
      final String path = file.getAbsolutePath();
      final String err =
        String.format("Falha na remoo do %s %s",
          file.isDirectory() ? "diretrio" : "arquivo", path);
      if (logger != null) {
        logger.log(Level.SEVERE, err);
      }
    }
    return removed;
  }

  /**
   * Remoo recursiva de diretrios. No registra erros em nenhum logger.
   * 
   * @param file arquivo ou diretrio a ser apagado
   * @return <code>true</code> se o arquivo/diretrio foi removido com sucesso
   */
  public static boolean deleteRecursively(File file) {
    return deleteRecursively(file, null);
  }
}
