package csbase.util.proxy;

import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;

/**
 * <p>
 * Decora um {@link InvocationHandler}, repassando remotamente a ele, as
 * chamadas ao mtodo {@link #invoke(Object, Method, Object[])}.
 * </p>
 * <p>
 * IMPORTANTE:<br>
 * A documentao do mtodo
 * {@link InvocationHandler#invoke(Object, Method, Object[])} diz que o proxy no
 * qual a chamada foi executada, ser passado como primeiro argumento.
 * Entretanto, para evitar a constante serializao do proxy, ser passado o
 * objeto para o qual o proxy foi criado para o {@link InvocationHandler}
 * responssvel pela execuo dos mtodos. No consideramos isso um problema,
 * pois a funo do parmetro proxy  apenas a de diferenciar onde a chamada
 * foi executada, j que um mesmo {@link InvocationHandler} pode ser utilizado
 * em vrios proxies e um proxy pode ser criado apenas com interfaces.  
 * </p>
 * 
 * @author Tecgraf
 */
public class RemoteInvocationHandler implements InvocationHandler, Serializable {

  /**
   * Objeto remoto ao qual as chamadas sero redirecionadas.
   */
  private RMIInvocationHandler rmiHandler;

  /**
   * Construtor.
   * 
   * @param obj Objeto para o qual est se criando este <i>handler</i>.
   * @param handler Responsvel por tratar as chamadas aos mtodos do objeto,
   *        obj.
   */
  public RemoteInvocationHandler(Object obj, InvocationHandler handler) {
    rmiHandler = new RMIInvocationHandlerImpl(obj, handler);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object invoke(Object proxy, Method method, Object[] args)
    throws Throwable {
    // Criar proxies dos parmetros.
    return rmiHandler
      .invoke(method.getName(), method.getParameterTypes(), args);
  }

  /**
   * Faz com que esse objeto no possa mais aceitar chamadas RMI.
   * 
   * @throws NoSuchObjectException Se esse objeto no estiver exportado.
   */
  public void unexport() throws NoSuchObjectException {
    UnicastRemoteObject.unexportObject(rmiHandler, true);
  }

  /**
   * Torna esse objeto disponvel para receber chamadas RMI, usando a porta
   * fornecida.
   * 
   * @param port Porta no qual o objeto ser exportado.
   * 
   * @throws RemoteException Em caso de falha da exportao.
   */
  public void export(int port) throws RemoteException {
    UnicastRemoteObject.exportObject(rmiHandler, port);
  }
}