/**
 * $Id: StringUtils.java 179359 2017-03-13 19:58:16Z cviana $
 */
package csbase.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.CollationKey;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

// TODO migrar para o JavaUtils?

/**
 * Mtodos utilitrios para manipulao e comparao de strings.
 * 
 * @author Tecgraf
 */
public class StringUtils {
  /**
   * Comparador de strings que no diferencia minsculas de maisculas.
   * 
   * @see #getPlainSortCollator()
   */
  private static Comparator<String> plainSortComparator;
  /**
   * {@link Collator} que no diferencia minsculas de maisculas. Este objeto 
   * um clone do <code>collator</code> default.
   */
  private static Collator plainSortCollator;

  /**
   * Divide um texto em partes, dado um separador.
   *
   * Diferente da verso {@link String#split(String)}, essa mantm elementos
   * vazios no final. <br>
   * Por exemplo: <blockquote>
   * <table cellpadding=1 cellspacing=0 summary="Exemplo do resultado da diviso de um texto dado um delimitador.">
   * <tr>
   * <th>Expresso</th>
   * <th>Resultado</th>
   * </tr>
   * <tr>
   * <td align=center>:</td>
   * <td><tt>"boo:and:foo".split("o")</tt></td>
   * </tr>
   * <tr>
   * <td align=center>o</td>
   * <td><tt>{ "b", "", ":and:f" }</tt></td>
   * </tr>
   * <tr>
   * <td align=center>:</td>
   * <td><tt>StringUtils.("boo:and:foo", 'o')</tt></td>
   * </tr>
   * <tr>
   * <td align=center>o</td>
   * <td><tt>{ "b", "", ":and:f", "" }</tt></td>
   * </tr>
   * </table>
   * </blockquote>
   *
   * @param text Texto a ser dividido.
   * @param delimiter Delimitador do texto.
   * @return texto dividido.
   */
  public static List<String> split(String text, char delimiter) {
    List<String> splitted = new ArrayList<String>();
    int begin = 0, cursor = text.indexOf(delimiter, begin);
    while (cursor > 0) {
      splitted.add(text.substring(begin, cursor));
      begin = cursor + 1;
      cursor = text.indexOf(delimiter, begin);
    }
    if (begin < text.length()) {
      splitted.add(text.substring(begin));
    }
    return splitted;
  }

  /**
   * Retorna um {@link Collator} para comparao de strings que no diferencia
   * minsculas de maisculas, i.e. {ZZZ, aaa}  ordenado como {aaa, ZZZ}.
   * 
   * OBS.: o objeto retornado  um singleton
   * 
   * IMPORTANTE: se o collator for usado em mltiplas comparaes das mesmas
   * strings (p.ex. ordenao de longas listas), considerar o uso de
   * {@link CollationKey}. Referncia:
   * <code>http://java.sun.com/docs/books/tutorial/i18n/text/perform.html</code>
   * 
   * @return <code>Collator</code> para comparao de strings que no diferencia
   *         minsculas de maisculas
   */
  public static Collator getPlainSortCollator() {
    if (plainSortCollator == null) {
      /*
       * clonamos para garantir que no h efeitos colaterais
       */
      plainSortCollator = (Collator) Collator.getInstance().clone();
      plainSortCollator.setStrength(Collator.TERTIARY);
    }
    return plainSortCollator;
  }

  /**
   * Retorna um <code>Comparator&lt;String&gt;</code> que no diferencia
   * minsculas de maisculas.
   * 
   * OBS.: o objeto retornado  um <i>singleton</i>
   * 
   * @return <code>Comparator&lt;String&gt;</code> que no diferencia minsculas
   *         de maisculas
   */
  public static Comparator<String> getPlainSortComparator() {
    if (plainSortComparator == null) {
      plainSortComparator = new Comparator<String>() {
        @Override
        public int compare(String o1, String o2) {
          return StringUtils.compare(o1, o2);
        }
      };
    }
    return plainSortComparator;
  }

  /**
   * Compara alfabeticamente duas strings, sem diferenciar minsculas de
   * maisculas.
   *
   * @see Collator#compare(String, String)
   *
   * @param s1 - primeira string
   * @param s2 - segunda string
   * @return -1 caso {@code s1 < s2}, 0 caso {@code s1 == s2} e 1 caso {@code
   * s1 > s2}
   */
  public static int compare(String s1, String s2) {
    return getPlainSortCollator().compare(s1, s2);
  }

  /**
   * Ordena alfabeticamente um array de strings, sem diferenciar minsculas de
   * maisculas. A ordenao acontece <i>in place</i>, i.e. o prprio array 
   * modificado e retornado como resultado.
   * 
   * @param array - array a ser ordenado
   * @return o array fornecido como entrada, ordenado
   */
  public static String[] sort(String[] array) {
    Arrays.sort(array, getPlainSortComparator());
    return array;
  }

  /**
   * Converte um objeto {@link InputStream} em um {@link String}.
   * 
   * @param in - Stream de entrada.
   * @return String que representa o stream de entrada.
   */
  public static String convertToString(InputStream in) {
    if (in == null) {
      return null;
    }

    BufferedReader br = new BufferedReader(new InputStreamReader(in));
    StringBuilder sb = new StringBuilder();
    String line = null;

    try {
      while ((line = br.readLine()) != null) {
        sb.append(line + "\n");
      }
      br.close();
      return sb.toString();
    }
    catch (IOException e) {
      return null;
    }
  }
}
